
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*! \file mapping.cpp
    \brief map instrument layer's text representations to QuantLib types
*/

#include "mapping.h"

//-----------------------------------------------------------------------------
Date mapDate( std::string text, Date asof ) {
  return parseDate (text.c_str(), asof);
}
Date mapDate( const char *text, Date asof ) {
  return parseDate (text, asof);
}
Date parseDate( std::string text, Date asof ) {
  return parseDate (text.c_str(), asof);
}
Date parseDate( const char *text, Date asof ) {
//-----------------------------------------------------------------------------
  static Debug log("parseDate");

  // assume formats:
  // YYYY-MM-DD
  // YYYYMMDD
  // DD/MM/YY
  // DD-MM-YY
  // DD.MM.YY
  // generic: 1D 2W 3M 1Y 5Y ... (needs offset date asof, default minDate())

  char buf[11];
  int y, m, d;

  if ( strlen(text)==10 ) {
    // YYYY-MM-DD
    strncpy(buf,text,4);   buf[4]='\0'; y = atoi(buf);
    strncpy(buf,text+5,2); buf[2]='\0'; m = atoi(buf);
    strncpy(buf,text+8,2); buf[2]='\0'; d = atoi(buf);
  }
  else if (strlen(text)==8) {
    if (strstr(text,"/")==NULL && strstr(text,"-")==NULL &&
        strstr(text,".")==NULL) {
      // YYYYMMDD
      strncpy(buf,text,4);   buf[4]='\0'; y = atoi(buf);
      strncpy(buf,text+4,2); buf[2]='\0'; m = atoi(buf);
      strncpy(buf,text+6,2); buf[2]='\0'; d = atoi(buf);
    }
    else {
      // DD*MM*YY
      strncpy(buf,text,2);   buf[2]='\0'; d = atoi(buf);
      strncpy(buf,text+3,2); buf[2]='\0'; m = atoi(buf);
      strncpy(buf,text+6,2); buf[4]='\0'; y = atoi(buf) + 2000; // FIXME?
    }
  }
  else {
    return asof + PeriodParser::parse( text );
  }

  Month mon;
  switch(m) {
  case 1:  mon = January; break;
  case 2:  mon = February; break;
  case 3:  mon = March; break;
  case 4:  mon = April; break;
  case 5:  mon = May; break;
  case 6:  mon = June; break;
  case 7:  mon = July; break;
  case 8:  mon = August; break;
  case 9:  mon = September; break;
  case 10: mon = October; break;
  case 11: mon = November; break;
  case 12: mon = December; break;
  default: printf("month %d out of range [1;12]\n", m); exit(1); break;
  }

  //  printf( "parsing data done\n" );

  return Date(d,mon,y);
}

//-----------------------------------------------------------------------------
bool isBlank( std::string text ) {
  return isBlank (text.c_str());
}
bool isBlank( const char *text ) {
//-----------------------------------------------------------------------------
  if ( text==NULL || strlen(text)==0 || strcmp(text,"")==0) return true;
  else {
    for(unsigned int i=0; i<strlen(text); i++) {
      if ( text[i]!=' ' && text[i]!='\t' ) return false;
    }
    return true;
  }
}

//-----------------------------------------------------------------------------
Calendar mapCalendar( std::string text ) {
  return mapCalendar (text.c_str());
}
Calendar mapCalendar( const char *text ) {
//-----------------------------------------------------------------------------
  static Debug log("mapCalendar");

  if (      strcmp( "NULL",   text ) == 0 ) return NullCalendar();
  else if ( strcmp( "TARGET", text ) == 0 ) return TARGET();
  else if ( strcmp( "FRA",    text ) == 0 ) return Germany();
  else if ( strcmp( "LON",    text ) == 0 ) return UnitedKingdom();
  else if ( strcmp( "NY",     text ) == 0 ) return UnitedStates();
  else {
    log.msg("error mapping calendar %s, exit", text );
    exit(1);
  }
}

//-----------------------------------------------------------------------------
DayCounter mapDayCounter( std::string text ) {
  return mapDayCounter(text.c_str());
}
DayCounter mapDayCounter( const char *text ) {
//-----------------------------------------------------------------------------
  static Debug log("mapDayCounter");

  if (      strcmp( "A360", text ) == 0 ) return Actual360();
  else if ( strcmp( "A365", text ) == 0 ) return Actual365Fixed();
  else if ( strcmp( "T360", text ) == 0 ) return Thirty360();
  else if ( strcmp( "ACT",  text ) == 0 ) return ActualActual();
  else {
    log.msg("error mapping day counter %s, exit", text );
    exit(1);
  }
}

//-----------------------------------------------------------------------------
BusinessDayConvention mapRollingConvention( std::string s ) {
  return mapRollingConvention(s.c_str());
}
BusinessDayConvention mapRollingConvention( const char *text ) {
//-----------------------------------------------------------------------------
  static Debug log("mapRollingConvention");

  if (      strcmp( "F", text ) == 0 )  return Following;
  else if ( strcmp( "MF", text ) == 0 ) return ModifiedFollowing;
  else if ( strcmp( "P", text ) == 0 )  return Preceding;
  else if ( strcmp( "MP", text ) == 0 ) return ModifiedPreceding;
  else {
    log.msg("error mapping day counter %s, exit", text );
    exit(1);
  }
}

//-----------------------------------------------------------------------------
Frequency mapFrequency( std::string text ) {
  return mapFrequency (text.c_str());
}
Frequency mapFrequency( const char *text ) {
//-----------------------------------------------------------------------------
  static Debug log("mapFrequency");

  if (      strcmp( "A", text ) == 0 )  return Annual;
  else if ( strcmp( "S", text ) == 0 )  return Semiannual;
  else if ( strcmp( "Q", text ) == 0 )  return Quarterly;
  else if ( strcmp( "M", text ) == 0 )  return Monthly;
  else {
    log.msg("error mapping frequency %s, exit", text );
    exit(1);
  }
}

//-----------------------------------------------------------------------------
bool mapYorN( std::string text ) {
  return mapYorN (text.c_str());
}
bool mapYorN( const char *text ) {
//-----------------------------------------------------------------------------
  static Debug log("mapYorN");

  if (      strcmp( "Y", text ) == 0 )  return true;
  else if ( strcmp( "N", text ) == 0 )  return false;
  else {
    log.msg("error mapping adjusted: %s", text );
    exit(1);
  }
}

//-----------------------------------------------------------------------------
boost::shared_ptr<Xibor>
mapIndex( std::string ccy, std::string name, std::string term,
          Handle<YieldTermStructure> rhTermStructure ) {
  return mapIndex (ccy.c_str(), name.c_str(), term.c_str(), rhTermStructure);
}
boost::shared_ptr<Xibor>
mapIndex( const char *ccy, const char *name, const char *term,
          Handle<YieldTermStructure> rhTermStructure ) {
//-----------------------------------------------------------------------------
  static Debug log("mapIndex");

  Period p = PeriodParser::parse( term );

  if ( strcmp( ccy,  "EUR" ) == 0 &&
       strcmp( name, "EURIBOR" ) == 0 ) {
    return boost::shared_ptr<Xibor>(new Euribor( p.length(), p.units(),
                                                 rhTermStructure ));
  }
  else if ( strcmp( ccy,  "USD" ) == 0 &&
	    strcmp( name, "LIBOR" ) == 0 ) {
    return boost::shared_ptr<Xibor>(new USDLibor( p.length(), p.units(),
                                                  rhTermStructure ));
  }
  else if ( strcmp( ccy,  "GBP" ) == 0 &&
	    strcmp( name, "LIBOR" ) == 0 ) {
    return boost::shared_ptr<Xibor>(new GBPLibor( p.length(), p.units(),
                                                  rhTermStructure ));
  }
  else if ( strcmp( ccy,  "JPY" ) == 0 &&
	    strcmp( name, "LIBOR" ) == 0 ) {
    return boost::shared_ptr<Xibor>(new JPYLibor( p.length(), p.units(),
                                                  rhTermStructure ));
  }
  else if ( strcmp( ccy,  "CAD" ) == 0 &&
	    strcmp( name, "LIBOR" ) == 0 ) {
    return boost::shared_ptr<Xibor>(new CADLibor( p.length(), p.units(),
                                                  rhTermStructure ));
  }
  else {
    log.msg( "error mapping index %s %s %s", ccy, name, term );
    exit(1);
  }

}

