/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef UTIL_H
#define UTIL_H

/*! \file util.h
  Frequently used general purpose function
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <string>

#include <ql/quantlib.hpp>
#include <ql/Math/matrix.hpp>
#include <ql/Math/loglinearinterpolation.hpp>
#include <ql/Math/cubicspline.hpp>


using namespace QuantLib;


/** Get command line arguments
    Parameters:
    - argc and argv are passed on from main
    - flag is the flag to search for, e.g. "-H" for displaying help
    - argument is a pointer to already allocated memory to store the argument
      following "flag", e.g. for "-x bla" getarg will put "bla" into the
      argument string, when called with flag "-x"
    Flags are not case sensitive, i.e. "-h" retrieves "-H" and vice versa.

    Return values:
    -1, flag not found
     0, flag found, but no corresponding argument
     1, flag and corresponding argument found
*/

int  getarg (int argc, char **argv, const char *flag, char *argument);

/** Basic data formatters from releases < 0.3.10
 */

//! Formats dates for output
/*! \deprecated use streams and manipulators for proper formatting */
class DateFormatter {
public:
  enum Format { Long, Short, ISO };
  static std::string toString(const Date& d,
                              Format f = Long);
};

//! Formats weekday for output
/*! \deprecated use streams and manipulators for proper formatting */
class WeekdayFormatter {
public:
  enum Format { Long, Short, Shortest };
  static std::string toString(Weekday wd,
                              Format f = Long);
};

//! Formats frequency for output
/*! \deprecated use streams and manipulators for proper formatting */
class FrequencyFormatter {
public:
  static std::string toString(Frequency f);
};

//! Formats integers for output
/*! \deprecated use streams and manipulators for proper formatting */
class IntegerFormatter {
public:
  static std::string toString(BigInteger l, Integer digits = 0);
  static std::string toPowerOfTwo(BigInteger l, Integer digits = 0);
};

//! Formats unsigned integers for output
/*! \deprecated use streams and manipulators for proper formatting */
class SizeFormatter {
public:
  static std::string toString(Size l, Integer digits = 0);
  static std::string toOrdinal(Size l);
  static std::string toPowerOfTwo(Size l, Integer digits = 0);
};

//! Formats real numbers for output
/*! \deprecated use streams and manipulators for proper formatting */
class DecimalFormatter {
public:
  static std::string toString(Decimal x,
                              Integer precision = 6,
                              Integer digits = 0);
  /*! \deprecated use streams and manipulators for proper formatting */
  static std::string toExponential(Decimal x,
                                   Integer precision = 6,
                                   Integer digits = 0);
  static std::string toPercentage(Decimal x,
                                  Integer precision = 6,
                                  Integer digits = 0);
};

//! Formats strings as lower- or uppercase
/*! \deprecated use the lowercase() and uppercase() functions */
class StringFormatter {
public:
  static std::string toLowercase(const std::string& s);
  static std::string toUppercase(const std::string& s);
};

//! Formats numeric sequences for output
class SequenceFormatter {
public:
  template<class Iterator>
  static std::string toString(Iterator begin,
                              Iterator end,
                              Integer precision = 6,
                              Integer digits = 0,
                              Size elementsPerRow = QL_MAX_INTEGER) {
    std::ostringstream s;
    s.precision(precision);
    s << "[ ";
    for (Size n=0; begin!=end; ++begin, ++n) {
      if (n == elementsPerRow) {
        s << ";\n  ";
        n = 0;
      }
      if (n!=0)
        s << " ; ";
      s << std::setw(digits) << *begin;
    }
    s << " ]";
    return s.str();
  }
};

//! Formats rates for output
/*! \deprecated use streams and manipulators for proper formatting */
class RateFormatter {
public:
  static std::string toString(Rate rate,
                              Integer precision = 5);
};

//! Formats volatilities for output
/*! \deprecated use streams and manipulators for proper formatting */
    class VolatilityFormatter {
    public:
      static std::string toString(Volatility vol,
                                  Integer precision = 5);
    };

/** Math functions from QuantLib releases < 0.3.6
 */

template<class I1, class I2, class I3>
  std::vector<double> interpolate(const I1& xx_begin,
				  const I1& xx_end,
				  const I2& yy_begin,
				  const I3& x_begin,
				  const I3& x_end,
				  int interpolationType,
				  bool allowExtrapolation,
				  CubicSpline::BoundaryCondition leftCondition,
				  double leftConditionValue,
				  CubicSpline::BoundaryCondition rightCondition,
				  double rightConditionValue,
				  bool monotonicityConstraint,
				  int derivativeOrder) {

  Interpolation f;

  switch (interpolationType) {
  case 1:
    f = LinearInterpolation(xx_begin, xx_end, yy_begin);
    break;
  case 2:
    f = CubicSpline(xx_begin, xx_end, yy_begin,
		    leftCondition, leftConditionValue,
		    rightCondition, rightConditionValue,
		    monotonicityConstraint);
    break;
  case 3:
    f = LogLinearInterpolation(xx_begin, xx_end, yy_begin);
    break;
  default:
    QL_FAIL("interpolate: invalid interpolation type");
  }

  Size i, n = x_end-x_begin;
  std::vector<double> result(n);
  switch (derivativeOrder) {
  case -1:
    for (i=0; i<n; i++) {
      result[i] = f.primitive(*(x_begin+i),allowExtrapolation);
    }
    break;
  case 0:
    for (i=0; i<n; i++) {
      result[i] = f(*(x_begin+i),allowExtrapolation);
    }
    break;
  case 1:
    for (i=0; i<n; i++) {
      result[i] = f.derivative(*(x_begin+i),allowExtrapolation);
    }
    break;
  case 2:
    for (i=0; i<n; i++) {
      result[i] = f.secondDerivative(*(x_begin+i),
				     allowExtrapolation);
    }
    break;
  default:
    QL_FAIL(IntegerFormatter::toString(derivativeOrder)
	    + " is an invalid derivative order");
  }

  return result;
}

double interpolate1D (const std::vector<double>& x_values,
		      const std::vector<double>& y_values,
		      double x,
		      int interpolationType,
		      bool allowExtrapolation);

double interpolate2D(const std::vector<double>& x_values,
		     const std::vector<double>& y_values,
		     const Matrix& dataMatrix,
		     double x,
		     double y,
		     int interpolation2DType,
		     bool allowExtrapolation);

#endif
