
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*! \file mkxmlio.cpp
    \brief Code generator for simplifying the SXP mechanism.

    This program scans header files containing definitions of "xml persistent"
    objects
    (refer to file xml.h) and generates cpp output suggesting definitions for
    previously declared "xml tags" as well as code for reading and writing
    object instances from and to xml files.

    Usage (see mkxmlio -h for more information):
    <pre>
    mkxmlio  -i classes.h  -t tag.cpp  -c io.cpp
    </pre>

    For examples of generated code refer to files xmlio.cpp and xmltag.cpp

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <string>
#include <list>
#include <map>

#define VERSIONNUM  "0.1.0"
#define VERSIONDATE "2005-08-11"

#define MAX_LINE_LENGTH 1000

typedef struct Class_struct {
  std::string nameSpace;
  std::string className;
  std::string fullName;
  void set( std::string &ns, std::string &cn ) {
    nameSpace = ns;
    className = cn;
    if (ns != "") fullName = ns + "::" + cn;
    else          fullName = cn;
  }
  void clear() {
    nameSpace = "";
    className = "";
    fullName = "";
  }
} Class;

typedef struct reflex_struct {
  Class       clazz;
  std::string memberType;
  std::string sxpType;
  std::string memberName;
  std::string tagName;
  std::string attrName;

  void clear() {
    //    nameSpace = "";
    //    className  = "";
    //    fullName  = "";
    memberType = "";
    sxpType    = "";
    memberName = "";
    tagName    = "";
    attrName   = "";
  }
  void printHeader(FILE * fp = stdout) {
    fprintf( fp, "%-20s %-20s %-20s %-8s %-20s %-s\n",
	     "Class ", "Member ", "Type ", "SxpType ", "Tag ", "Attr" );
    fprintf( fp, "%-20s %-20s %-20s %-8s %-20s %-s\n",
	     "----- ", "------ ", "---- ", "------- ", "--- ", "----" );
  }
  void print(FILE * fp = stdout) {
    fprintf( fp, "%-20s %-20s %-20s %-8s %-20s %-s\n",
	     //nameSpace.c_str(),
	     clazz.fullName.c_str(),
	     memberName.c_str(),
	     memberType.c_str(),
	     sxpType.c_str(),
	     tagName.c_str(),
	     attrName.c_str());
  }
} Reflex;

void        version  ( char *name );
int         usage    ( char *name );
int         getarg   ( int argc, char **argv, const char *flag, char *argument );
int         classList( const char *ifile,
		       std::list<Class> &clist );
std::string whichType( std::string tok,
		       std::list<Class> &clist );
int         parse    ( const char *ifile,
		       std::list<Class> &clist,
		       std::list<Reflex> &refList );
int         dump     ( std::list<Reflex> &refList,
		       std::list<Class> &clist,
		       const char *name,
		       const char *ifile,
		       const char *cfile,
		       const char *tfile,
		       const char *rfile);
// in linux available as GNU extension
int         getline  ( char* line, size_t* len, FILE* f );

bool debug = false;

//--------------------------------------------------------------------------------------------------
int main( int argc, char **argv ) {
//--------------------------------------------------------------------------------------------------
  char buf[100], ifile[100], cfile[100], tfile[100], rfile[100];
  bool append = false;

  if ( getarg(argc, argv, "-D", buf) >= 0 ) {
    debug = true;
  }

  if (debug) version(argv[0]);

  if ( getarg(argc, argv,"-H",buf) == 0 ) return usage(argv[0]);

  if ( getarg(argc, argv, "-I", ifile) < 1 ) {
    printf( "ERROR: input header file is required\n" );
    return usage(argv[0]);
  }

  if ( getarg(argc, argv, "-C", cfile) < 1 ) {
    sprintf( cfile, "xmlio.cpp" );
    printf( "WARNING: no output file name passed \n" );
    printf( "         using %s\n", cfile );
  }

  if ( getarg(argc, argv, "-T", tfile) < 1 ) {
    sprintf( tfile, "xmltag.cpp" );
    printf( "WARNING: no output file name passed \n" );
    printf( "         using %s\n", tfile );
  }

  if ( getarg(argc, argv, "-L", rfile) < 1 ) {
    sprintf( rfile, "mkxmlio.log" );
    printf( "WARNING: no log file name passed \n" );
    printf( "         using %s\n", rfile );
  }

  if ( getarg(argc, argv, "-A", buf) >= 0 ) {
    append = true;
  }

  FILE *fc = NULL;
  FILE *ft = NULL;

  if (append) {
    fc = fopen( cfile, "a" );
    ft = fopen( tfile, "a" );
  }
  else {
    fc = fopen( cfile, "w" );
    ft = fopen( tfile, "w" );
  }

  if (fc) {
    fprintf( fc, "#include \"%s\"\n\n", ifile );
    fclose( fc );
  }
  else {
    printf( "error opening io file %s\n", cfile );
    return -1;
  }

  if (ft) {
    fprintf( ft, "#include \"%s\"\n\n", ifile );
    fclose( ft );
  }
  else {
    printf( "error opening tag file %s\n", tfile );
    return -1;
  }

  std::list<Class> clist;

  if ( classList(ifile, clist) ) {
    if (debug) printf( "reading class list from file %s failed\n", ifile );
    return 1;
  }

  if (debug) printf( "number of classes: %ld\n", clist.size() );
  std::list<Class>::iterator i;
  for( i=clist.begin(); i!=clist.end(); i++ ) {
    if (debug) printf( "class name: %s\n", i->fullName.c_str() );
  }

  std::list<Reflex> refList;

  if ( parse(ifile, clist, refList) ) {
    printf( "parsing header file %s failed\n", ifile );
    return 1;
  }
  else if (debug) printf( "parsing header file %s successful\n", ifile );

  if ( dump(refList, clist, argv[0], ifile, cfile, tfile, rfile) ) {
    printf( "writing to files %s and %s failed\n", cfile, tfile );
    return 1;
  }
  else if (debug) {
    printf( "class information written to log file %s\n", rfile );
    printf( "i/o source code written to file %s\n", cfile );
    printf( "tag definitions written to file %s\n", tfile );
  }

  if (debug) printf( "done\n" );

  return 0;
}

//--------------------------------------------------------------------------------------------------
int getline(char* line, size_t* len, FILE* f) {
// in linux available as GNU extension, but with first argument of type char**
//-------------------------------------------------------------------------------------------------
  int buf, i = 0, brk = 0;

  do {	buf = fgetc (f);
	if (buf != EOF && buf != '\n' && i < MAX_LINE_LENGTH)
	  line[i++] = buf;
	else
	  brk = 1;
  } while (!brk);

  if (i == MAX_LINE_LENGTH) {
	printf ("maximum line length (%d) exceeded, exit\n", MAX_LINE_LENGTH);
	exit(-1);
  }

  line[i] = '\0';
  *len = i;

  // printf ("%d:%s\n", i, line);

  if (buf == EOF) return -1;
  else            return *len;
}

//--------------------------------------------------------------------------------------------------
void rm_leading_blanks( char * line, size_t len ) {
//--------------------------------------------------------------------------------------------------
  unsigned int i=0, cnt=0;
  while( line[cnt] == ' ' ) cnt++;
  if (cnt) {
    for( i=0; i<len-cnt; i++ ) line[i] = line[i+cnt];
    for( i=len-cnt; i<len; i++ ) line[i] = ' ';
  }
}

//--------------------------------------------------------------------------------------------------
void rm_comment( char * line, size_t len ) {
//--------------------------------------------------------------------------------------------------
  unsigned int i, cnt;

  if ( strstr(line, "//")==NULL ) return;
  for( cnt=0; cnt<len-1; cnt++ ) if ( line[cnt] == '/' && line[cnt+1] == '/' ) break;
  for( i=cnt; i<len; i++ ) line[i] = ' ';
  line[cnt] = '\n';
}

//--------------------------------------------------------------------------------------------------
void rm_leading_chars( char * line, size_t len, int n ) {
//--------------------------------------------------------------------------------------------------
  unsigned int i, j;
  if (len-n>=0) j = len-n;
  else          j = 0;

  for( i=0; i<len-n; i++ ) line[i] = line[i+n];
  for( i=j; i<len; i++ )   line[i] = ' ';
}

//--------------------------------------------------------------------------------------------------
int classList( const char *ifile, std::list<Class> &clist ) {
//--------------------------------------------------------------------------------------------------

  char line[MAX_LINE_LENGTH+1];
  size_t len;
  int read;
  std::string space;
  std::string name;   // last class name
  Class       aclass; // class buffer

  clist.clear();

  FILE * f = fopen( ifile, "r" );

  while( (read = getline(line, &len, f)) != -1 ) {
    //printf( "retrieved line of length %zu:\n", read );
    //printf( "LINE:%s", line );
    rm_leading_blanks(line, len);
    //printf( "LINE:%s", line );

    if ( strncmp(line, "//", 2) == 0 ) {
      //printf( "this is a comment line, skip\n" );
      continue;
    }

    if ( strncmp(line, "namespace", 9) == 0 ) {
      rm_leading_chars(line, len, 9);
      space = strtok(line, " ");
      if (debug) printf( "namespace is %s\n", space.c_str() );
      continue;
    }

    if ( strncmp(line, "class", 5) == 0 ) {
      rm_leading_chars(line, len, 5);
      name = strtok(line, " ");
      aclass.set( space, name );
      if (strstr(aclass.className.c_str(),";") != NULL || strstr(line,";") != NULL)
	// this is just a class declaration, skip
	continue;
      clist.push_back( aclass );
      continue;
    }
  }

  fclose(f);

  return 0;
}

//--------------------------------------------------------------------------------------------------
std::string whichType( std::string tok, std::list<Class> &clist ) {
//--------------------------------------------------------------------------------------------------
  std::string type = "";

  if ( tok == "SXP::Tag" ||
       tok == "Tag" ||
       tok == "XML_TAG" ) {
    type = "tag";
  }
  else if ( tok == "XML_ATTR" ||
	    tok == "XML_ATTRIBUTE" ) {
    type = "attr";
  }
  else if ( tok == "bool" ||
       tok == "int" ||
       tok == "long" ||
       tok == "float" ||
       tok == "double" ||
       tok == "std::string" ||
       tok == "string" ) {
    type = "simple";
  }
  else if ( strncmp( tok.c_str(), "XML::list", 9) == 0 ) {
    if (debug) printf( "*** found sxp::list type\n" );
    type = "list";
  }
  else if ( strncmp( tok.c_str(), "XML::ptr", 8) == 0 ) {
    if (debug) printf( "*** found sxp::ptr type\n" );
    type = "ptr";
  }
  else {
    std::list<Class>::iterator i;
    // FIXME: the token is "clean", assuming that the class appears in only one relevant namespace
    for( i = clist.begin(); i != clist.end(); i++ ) {
      if ( strcmp(tok.c_str(), i->className.c_str()) == 0 ) {
	type = "complex";
	break;
      }
    }
  }
  return type;
}

//--------------------------------------------------------------------------------------------------
int pop( std::string &tok, const char * line, int len, int sep ) {
//--------------------------------------------------------------------------------------------------
  char buf[1000];
  strcpy(buf,"");
  int i, ret=1;

  for( i=0; i<len; i++ ) {
    if (line[i] == sep ) {
      ret=0;
      break;
    }
    else if ( line[i] == '\n' ) {
      ret=1;
      break;
    }
    else buf[i] = line[i];
  }
  buf[i] = '\0';
  tok = buf;
  return ret;
}

//--------------------------------------------------------------------------------------------------
int parse( const char *ifile,
	   std::list<Class> &clist,
	   std::list<Reflex> &refList ) {
//--------------------------------------------------------------------------------------------------

  char line[MAX_LINE_LENGTH+1];
  size_t len;
  int read;
  std::string name, space;
  Reflex ref;

  ref.clear();

  FILE * f = fopen( ifile, "r" );

  while( (read = getline(line, &len, f)) != -1 ) {
    if (debug) printf( "LINE:%s|\n", line );

    rm_comment(line, len);
    rm_leading_blanks(line, len);

    // skip comments
    if ( strncmp(line, "//", 2) == 0 ) {
      continue;
    }

    // class end
    if ( strncmp(line, "};", 2) == 0 ) {
      ref.clazz.clear();
      continue;
    }

    if ( strncmp(line, "namespace", 9) == 0 ) {
      rm_leading_chars(line, len, 9);
      space = strtok(line, " ");
      continue;
    }

    if ( strncmp(line, "class", 5) == 0 ) {
      rm_leading_chars(line, len, 5);
      name = strtok(line, " ");
      ref.clazz.set( space, name );
      continue;
    }

    if ( ref.clazz.className != "" ) {
      //printf( "LINE:%s", line );

      // assume at least two tokens per line, separated by blanks
      // the first one is interpreted as type
      if ( pop(ref.memberType, line, len, ' ') == 0 ) {

	// check type and classify according to SXP
	// valid complex types (classes) need to be declared within the given header file
	ref.sxpType = whichType(ref.memberType, clist);

	rm_leading_chars(line, len, ref.memberType.size());
	rm_leading_blanks(line, len);

	if (debug) printf ("PARSE: type %s is %s\n", ref.memberType.c_str(), ref.sxpType.c_str());

	// these are potential members
	if ( (ref.sxpType == "simple" ||
	      ref.sxpType == "complex" ||
	      ref.sxpType == "list" ||
	      ref.sxpType == "ptr" ||
	      ref.sxpType == "tag" ||
	      ref.sxpType == "attr") &&
             strstr(line,"(")==NULL &&
             strstr(line,")")==NULL &&
             strstr(line,";")!=NULL ) {
	  // find comma separated members on same line
	  while ( pop(ref.memberName, line, len, ',') == 0 ) {
	    if (debug) printf ("PARSE: member %s|\n", ref.memberName.c_str() );
	    // ignore member functions, identified by opening bracket
	    if ( strstr(ref.memberName.c_str(), "(") != NULL ) {
	      if (debug) printf( "*** ignore member function %s\n", ref.memberName.c_str() );
	      pop(ref.memberName, line, len, ')');
//printf( "here:\n"); // comma at the end of the line plus one blank causes seg fault here
	      rm_leading_chars(line, len, ref.memberName.size());
//printf( "no\n");
	      rm_leading_blanks(line, len);
	      ref.clear();
	      //	      continue;
	    }
	    else {
	      if (debug) printf( "comma separated member %s\n", ref.memberName.c_str() );
	      // keep in mind, remove member from line, drop remaining leading blank characters
	      if (debug && ref.sxpType == "list")
		printf( "*** type list, member %s\n", ref.memberName.c_str() );

	      refList.push_back( ref );
	      rm_leading_chars(line, len, ref.memberName.size()+1);
	      rm_leading_blanks(line, len);
	      ref.clear();
	    }
	  }
	  // check last member on this line, terminated by colon
	  if (debug) printf ("PARSE: remaining: %s\n", ref.memberName.c_str() );
	  if (debug) printf ("PARSE: line (%ld): %s\n", len, line );
	  if ( pop(ref.memberName, line, len, ';') == 0 &&
	       strstr(ref.memberName.c_str(),"(") == NULL &&
	       ref.memberName != "" ) {
	    if (debug && ref.sxpType == "list")
	      printf( "*** type list, member %s\n", ref.memberName.c_str() );
	    if (debug) printf ("PARSE: member %s\n", ref.memberName.c_str() );
	    refList.push_back( ref );
	  }
	}
	else {
	  // the first token does not look like a valid type, neither simple nor complex nor tag
	  // skip this line
	  continue;
	}
      }
      else {
	// there seems to be only one token on this line, not a type/member pair
	// skip	this line
	continue;
      }
    }
    else {
      //printf( "outside class definition, skip\n" );
    }
  }

  fclose(f);

  return 0;
}

//--------------------------------------------------------------------------------------------------
int dump( std::list<Reflex> &refList,
	  std::list<Class> &clist,
	  const char *name,
	  const char *ifile,
	  const char *cfile,
	  const char *tfile,
	  const char *rfile) {
//--------------------------------------------------------------------------------------------------

  if (debug) printf( "dump called!\n" );

  if ( refList.size() > 0 ) {
    // FIXME, here the SXP code should be written

    std::list<Class>::iterator  ic;
    std::list<Reflex>::iterator it, im;

    std::string tag, attr, member;
    int match=0;

    if (debug) refList.begin()->printHeader();

    for( im = refList.begin(); im != refList.end(); im++ ) {
      if ( im->sxpType == "simple"
	   || im->sxpType == "complex"
	   || im->sxpType == "list"
	   || im->sxpType == "ptr" ) {
	if ( strstr( im->memberName.c_str(), "_" ) == NULL )
	  member = im->memberName.c_str();
	else
	  member = strstr( im->memberName.c_str(), "_" ) + sizeof(char);

	match  = 0;

	for( it = refList.begin(); it != refList.end(); it++ ) {
	  if ( it->clazz.fullName == im->clazz.fullName &&
	       it->sxpType == "tag" ) {
	    if ( strstr( it->memberName.c_str(), "_" ) == NULL ) {
	      printf( "tag name %s needs underscore for matching with member name, exit\n",
		      it->memberName.c_str() );
	      it->print();
	      exit( 1 );
	    }
	    tag = strstr( it->memberName.c_str(), "_" ) + sizeof(char);
	    //printf( "%s %s\n", member.c_str(), tag.c_str() );
	    if ( tag == member ) {
	      match++;
	      im->tagName = it->memberName;
	      break;
	    }
	  }
	}

	if (!match) im->tagName = "N/A";

	match = 0;

	for( it = refList.begin(); it != refList.end(); it++ ) {
	  if ( it->clazz.fullName == im->clazz.fullName &&
	       it->sxpType == "attr" ) {
	    if ( strstr( it->memberName.c_str(), "_" ) == NULL ) {
	      printf( "attr name %s needs underscore for matching with member name, exit\n",
		      it->memberName.c_str() );
	      it->print();
	      exit( 1 );
	    }
	    attr = strstr( it->memberName.c_str(), "_" ) + sizeof(char);
	    if ( attr == member ) {
	      match++;
	      im->attrName = it->memberName;
	      break;
	    }
	  }
	}

	if (!match) im->attrName = "N/A";

	if (debug) im->print();

	member = "";
	tag    = "";
	attr   = "";
      }
    }
    if (debug) printf( "\n" );

    // write the parse result - reflected list - to a log file

    FILE * f;

    if ( !(f = fopen( rfile, "w" )) ) {
      printf( "error opening file %s, exit\n", rfile );
      exit(-1);
    }

    fprintf( f, "---------------------------------------------------------------------------\n" );
    fprintf( f, "Derived from header file: %s\n", ifile );
    fprintf( f, "Generated source code in: %s, %s\n\n", cfile, tfile );
    fprintf( f, "This file reflects all members found in the header file's classes.\n\n");
    fprintf( f, "Check \"tags\" in column 5 carefully.\n\n" );
    fprintf( f, "Note that any missing \"tag\" (N/A in column 5) means that its corresponding\n" );
    fprintf( f, "member variable is ignored, i.e. the member variable is\n" );
    fprintf( f, "NEITHER READ FROM NOR WRITTEN TO XML.\n\n" );
    fprintf( f, "Adding a corresponding tag variable to the header file will resolve this.\n" );
    fprintf( f, "---------------------------------------------------------------------------\n\n");

    refList.begin()->printHeader(f);

    for( im = refList.begin(); im != refList.end(); im++ ) {
      if ( im->sxpType == "simple"
	   || im->sxpType == "complex"
	   || im->sxpType == "list"
	   || im->sxpType == "ptr" ) {
	im->print(f);
      }
    }

    fclose(f);

    // write tag definitions ///////////////////////////////////////////////////////////////////////

    if ( !(f = fopen( tfile, "a" )) ) {
      printf( "error opening file %s, exit\n", tfile );
      exit(-1);
    }

    fprintf( f, "/*! \\file xmltag.cpp\n" );
    fprintf( f, "    \\brief Example for 'tag' definitios required by SXP.\n" );
    fprintf( f, "*/\n\n");

    time_t     t;
    struct tm *tm;

    time(&t);
    tm = localtime(&t);

    fprintf( f, "/*******************************************************************\n" );
    fprintf( f, " * This file was automatically generated at %s", asctime(tm) );
    fprintf( f, " * Do not edit, it might be overwritten again. \n" );
    fprintf( f, " * Amend the code generator (%s) instead. \n", name );
    fprintf( f, " *******************************************************************/ \n\n" );
    char xmlTagName[100], xmlIdName[100];
    std::string text;

    for( ic = clist.begin(); ic != clist.end(); ic++ ) { // class loop

      if (debug) printf( "writing class %s ... \n", ic->fullName.c_str() );
      fflush(stdout);

      fprintf( f, "// class %s\n\n", ic->fullName.c_str() );

      // FIXME: hard coded t_own
      fprintf( f, "XML::Tag %s::%s (\"%s\");\n",
	       ic->fullName.c_str(),
	       "t_own",
	       ic->className.c_str() );
      fprintf( f, "\n" );

      fprintf( f, "// class %s, containers \n\n", ic->fullName.c_str() );

      // FIXME: define list container xml tags for all classes
      fprintf( f, "template <> XML::Tag XML::list<%s>::%s (\"%sList\");\n",
	       ic->fullName.c_str(),
	       "t_own",
	       ic->className.c_str() );
      fprintf( f, "template <> XML::Tag XML::list<%s>::%s (\"Items\");\n",
	       ic->fullName.c_str(),
	       "t_items");
      fprintf( f, "\n" );

      // FIXME: define ptr container xml tags for all classes
      fprintf( f, "template <> XML::Tag XML::ptr<%s>::%s (\"%sPointer\");\n",
	       ic->fullName.c_str(),
	       "t_own",
	       ic->className.c_str() );
      fprintf( f, "\n" );

      fprintf( f, "// class %s, members \n\n", ic->fullName.c_str() );

      // define member xml tags
      for( im = refList.begin(); im != refList.end(); im++ ) { // tag loop
	if ( im->clazz.fullName == ic->fullName
	     && (im->sxpType == "simple"
		 || im->sxpType == "complex" )
	     &&  im->tagName != "" ) {

	  if (im->tagName == "N/A") {
	   if (debug) printf( "*** warning: missing tag, member %s::%s is not persistent\n",
		    ic->fullName.c_str(), im->memberName.c_str() );
	    continue;
	  }

	  if ( im->tagName == "t_own" ) {
	    continue;
	  }

	  strcpy( xmlTagName, strstr( im->tagName.c_str(), "_") + 1 );
	  xmlTagName[0] = toupper( xmlTagName[0] );
	  if ( im->tagName == "t_own" )
	    strcpy(xmlTagName, im->clazz.className.c_str() );
	  else
	    strcpy(xmlTagName, strstr( im->tagName.c_str(), "_") + 1 );
	  xmlTagName[0] = toupper( xmlTagName[0] );
	  fprintf( f, "XML::Tag %s::%s (\"%s\");\n",
		   im->clazz.fullName.c_str(),
		   im->tagName.c_str(),
		   xmlTagName );
	}
      }
      fprintf( f, "\n" );

      //define attributes
      for( im = refList.begin(); im != refList.end(); im++ ) { // tag loop
	if ( im->clazz.fullName == ic->fullName
	     && (im->sxpType == "simple"
		 || im->sxpType == "complex")
	     &&  im->attrName != "N/A" ) {

	  if (im->tagName == "N/A") {
	   if (debug) printf( "*** warning: missing tag, member %s::%s is not persistent\n",
		    ic->fullName.c_str(), im->memberName.c_str() );
	    continue;
	  }

	  strcpy( xmlIdName, strstr( im->attrName.c_str(), "_") + 1 );
	  xmlIdName[0] = toupper( xmlIdName[0] );

	  fprintf( f, "char * %s::%s = \"%s\";\n",
		   ic->fullName.c_str(),
		   im->attrName.c_str(),
		   xmlIdName );
	}
      }

      fprintf( f, "\n" );
    }

    fclose( f );

    // write io code ///////////////////////////////////////////////////////////////////////////////

    if ( !(f = fopen( cfile, "a" )) ) {
      printf( "error opening file %s, exit\n", cfile );
      exit(-1);
    }

    fprintf( f, "/*! \\file xmlio.cpp\n" );
    fprintf( f, "    \\brief Example for 'read/write' functiones required by SXP.\n" );
    fprintf( f, "*/\n\n");

    fprintf( f, "/*******************************************************************\n" );
    fprintf( f, " * This file was automatically generated at %s", asctime(tm) );
    fprintf( f, " * Do not edit, it might be overwritten again. \n" );
    fprintf( f, " * Amend the code generator (%s) instead. \n", name );
    fprintf( f, " *******************************************************************/ \n\n" );

    for( ic = clist.begin(); ic != clist.end(); ic++ ) { // class loop

      if (debug) printf( "writing class %s ... \n", ic->fullName.c_str() );
      fflush(stdout);

      // define read function for complex members
      fprintf( f, "void %s::BeginElement(SXP::IParser *pIn, SXP::IElement *pElement) {\n",
	       ic->fullName.c_str() );

      text = "\tif (SXP::debug) { printf(\"<%s> \", pElement->Name() ); fflush(stdout); }\n";
      fprintf( f, "%s", text.c_str() );

      for( im = refList.begin(); im != refList.end(); im++ ) { // complex member loop
	if ( im->clazz.fullName == ic->fullName
	     && ( im->sxpType == "complex" || im->sxpType == "list" || im->sxpType == "ptr" )
	     && im->tagName != "" ) {

	  if (im->tagName == "N/A") {
	   if (debug) printf( "*** warning: missing tag, member %s::%s is not persistent\n",
		    ic->fullName.c_str(), im->memberName.c_str() );
	    continue;
	  }

	  fprintf( f, "\tif( pElement->IsA(%s::t_own) ) {\n",  im->memberType.c_str() );
	  if (im->attrName != "N/A")
	    fprintf( f, "\tif( pElement->AttribIs(\"id\", %s) ) {\n", im->attrName.c_str() );
	  fprintf( f, "\t\t%s.init();\n",                      im->memberName.c_str() );
	  fprintf( f, "\t\tpIn->ReadTo( &%s );\n",             im->memberName.c_str() );
	  fprintf( f, "\t\t%s.BeginElement(pIn, pElement);\n", im->memberName.c_str() );
	  if (im->attrName != "N/A")
	    fprintf( f, "\t}\n" );
	  fprintf( f, "\t}\n" );
	}
      } // end of complex member loop

      fprintf( f, "}\n\n" );

      // define read function for simple members
      fprintf( f, "void %s::EndElement(SXP::IParser *pIn, SXP::IElement *pElement) {\n",
	       ic->fullName.c_str() );

      text = "\tif (SXP::debug) { printf(\"</%s> \", pElement->Name() ); fflush(stdout); }\n";
      fprintf( f, "%s", text.c_str() );

      for( im = refList.begin(); im != refList.end(); im++ ) { // simple member loop
	if ( im->clazz.fullName == ic->fullName
	     && im->sxpType == "simple"
	     && im->tagName != "" ) {

	  if (im->tagName == "N/A") {
	   if (debug) printf( "*** warning: missing tag, member %s::%s is not persistent\n",
		    ic->fullName.c_str(), im->memberName.c_str() );
	    continue;
	  }

	  fprintf( f, "\tif( pElement->IsA(%s) ) pElement->Retrieve(%s);\n",
		   im->tagName.c_str(), im->memberName.c_str() );

	}
      } // end of simple member loop

      fprintf( f, "}\n\n" );

      // define write function
      fprintf( f, "void %s::WriteElement(SXP::IOutStream *pOut, SXP::dict& attribs) {\n",
	       ic->fullName.c_str() );
      fprintf( f, "\tSXP::dict localAttribs;\n" );
      fprintf( f, "\tpOut->BeginObject(t_own, attribs);\n" );

      for( im = refList.begin(); im != refList.end(); im++ ) { // member loop
	if ( im->clazz.fullName == ic->fullName
	     && (im->sxpType == "simple"
		 || im->sxpType == "complex"
		 || im->sxpType == "list"
		 || im->sxpType == "ptr" )
	     && im->tagName != "" ) { // simple or complex sxpType

	  if (im->tagName == "N/A") {
	   if (debug) printf( "*** warning: missing tag, member %s::%s is not persistent\n",
		    ic->fullName.c_str(), im->memberName.c_str() );
	    continue;
	  }

	  if (im->sxpType == "simple") {
	    fprintf( f, "\tpOut->WriteElement(%s, %s);\n",
		     im->tagName.c_str(),
		     im->memberName.c_str() );
	  }
	  else { // complex or list
	    if (im->attrName != "N/A") {
	      fprintf( f, "\tlocalAttribs[\"id\"] = %s;\n" , im->attrName.c_str() );
	    }
	    fprintf( f, "\tpOut->WriteSubElement(&%s, localAttribs);\n", im->memberName.c_str() );
	    if (im->attrName != "N/A") {
	      fprintf( f, "\tlocalAttribs.clear();\n" );
	    }
	  }
	}
      } // end of member loop

      fprintf( f, "\tpOut->EndObject(t_own);\n" ); // closing tag

      fprintf( f, "}\n\n" );

     if (debug) printf( "done\n" );
    } // end of class loop

    fprintf( f, "// eof\n\n" );

    fclose(f);
  }
  else {
    //printf( "nothing to dump\n" );
  }

  return 0;
}

//--------------------------------------------------------------------------------------------------
void version( char *name ) {
//--------------------------------------------------------------------------------------------------
  unsigned int  i;
  char msg[1000];
  sprintf( msg, "| %s version %s date %s |", name, VERSIONNUM, VERSIONDATE );

  char header[1000];
  header[0]='+';
  for( i=1; i<strlen(msg)-1; i++ ) strcpy( header+i, "-" );
  header[i]='+';
  header[i+1]='\0';

  char blank[1000];
  blank[0]='|';
  for( i=1; i<strlen(msg)-1; i++ ) strcpy( blank+i, " " );
  blank[i]='|';
  blank[i+1]='\0';

  printf( "%s\n", header );
  printf( "%s\n", blank );
  printf( "%s\n", msg );
  printf( "%s\n", blank );
  printf( "%s\n", header );
}

//--------------------------------------------------------------------------------------------------
int usage( char *name ) {
//--------------------------------------------------------------------------------------------------
  printf( "\n" );
  printf( "purpose: generate SXP source code for reading/writing objects\n" );
  printf( "         as declared in the header file passed as an argument\n" );
  printf( "usage: %s [options]\n", name );
  printf( "options:\n" );
  printf( "  -h           show this screen\n" );
  printf( "  -i <file>    header\n" );
  printf( "  -c <file>    file name for io source code\n" );
  printf( "  -t <file>    file name for tag name source code\n" );
  printf( "  -a           append code to existing output file\n" );
  printf( "  -d           enable debug messages\n" );
  printf( "\n" );
  return 1;
}

// -------------------------------------------------------------------------------------------------
int getarg( int argc, char **argv, const char *flag, char *argument ) {
// -------------------------------------------------------------------------------------------------
  // search for specified flag in the command line arguments
  // write the command line argument following 'flag' to 'argument' (if it exists)
  // returns: -1, flag not detected
  //           0, flag detected, but no argument given
  //           1, flag and argument detected

  int i;
  unsigned int j;
  char a[1000]; // arguments converted to upper case
  char f[1000]; // flag converted to upper case

  if ( strlen(flag) > 1000 ) {
    printf( "string length of flag is too long (%ld)\n", strlen(flag) );
    exit(1);
  }

  // convert flag to upper case
  for( j=0; j<strlen(flag); j++ ) f[j] = toupper(flag[j]);
  f[j] = '\0';

  for( i=1; i<argc; i++ ) {

    if ( strlen(argv[i]) > 1000 ) {
      printf( "string length of argument %s is too long (%ld)\n", argv[i], strlen(flag) );
      exit(1);
    }

    // convert command line argument to uppercase
    for( j=0; j<strlen(argv[i]); j++ ) a[j] = toupper(argv[i][j]);
    a[j] = '\0';

    if ( strcmp(a,f)==0 ) {
      //      printf( "flag: %s -> %s\n", argv[i], a );
      if (i==argc-1 || strncmp(argv[i+1],"-",1)==0 ) return 0;
      else {
	strcpy( argument, argv[i+1] );
	return 1;
      }
    }
  }

  return -1;
}

// eof
