/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*! \file xml.h
    \brief Root object and definitions for xml persistent classes

    The root object XmlObject is a (partially) virtual class defining the
    minimum required set of member functions to be implemented by derived
    classes.

    The XML persistence functionality used here is based on the SXP package
    provided by Ivan-Assen Ivanov and published at http://sxp.sourceforge.net.
    This package is a C++ wrapper around James Clark's expat XML parser
    available at http://expat.sourceforge.net.

    Using SXP is simplified here a bit more by

    - hiding function declarations as far as possible from the user;
    this is achieved through macro SXP_IO that has to be used within the
    definition of any class that should be made persistent;

    - automatic code generation for the i/o functions and static members (tags)
    declared via SXP_IO;
    the code generator source is provided in mksxpio.cpp;
    it can be called for each header file containing persistent class
    definitions (classes.h) to produce a file containing the corresponding i/o
    code and static members definitions (classes.cpp):
    <pre>
          mkxmlio  -i classes.h
    </pre>
    code generation can be incorporated into the Makefile for the project

    The remaining user tasks are for each class:

    1 - Derive any "xml persistent" class from "public IXmlObject"

    2 - Add macro XML_IO before the end of the class definition

    3 - Declare a static "tag" member variable (type XML_TAG) for each class
    member that should become persistent (i.e. not neccessarily all class
    members). Condition for the code generation to work:
    - Tag names need an underscore, members not neccessarily.
    - Member and Tag names have to be identical beyond the underscore "_".
    Note that the code generator derives the xml Tag names from the member
    variable name by omitting any character before and including "_" and
    converting the remaining first character to upper case.
    For example, member "m_assetList or assetList" translates to xml tag
    "<AssetList>".

    4 - Implement the following class methods to support the persistent list
    container SXP::List<T>
    - void init()
    - int isValid()   (return values < 0 means invalid, >= 0 means valid)

    If the user wants to define xml tag names more independent from variable
    names, the
    - i/o functions have to be written by the user (tedious, though straight
    forward), or
    - the code generator has to be extended to e.g. use some configuration
    (file) for mapping tag variable names to xml tag names
    - ...
*/

#ifndef XML_H
#define XML_H

#include "sxp.h"

/*! \brief  Purely virtual base class for any xml persistent object
*/

//interface XmlObject : public SXP::IPersistObj {
struct XmlObject : public SXP::IPersistObj {

  virtual SXP::Tag& GetClassTag() const = 0;

  virtual void BeginElement(SXP::IParser *pIn, SXP::IElement *pElement) = 0;
  virtual void EndElement(SXP::IParser *pIn, SXP::IElement *pElement) = 0;
  virtual void WriteElement(SXP::IOutStream *pOut, SXP::dict& attribs) = 0;

  // initialise all members
  virtual void init() = 0;
  // basic validation of the object itself such that a return value < 0 means "invalid"
  virtual int isValid() = 0;

  virtual ~XmlObject(){}

  int readObject (char *file) {
    // clear the object first
    // init() needs to be implemented in derived classes
    init ();
    // receive the root element
    SXP::CParser p (this);
    // process the file
    if (p.FeedFile(file) != SXP::err_no_error) {
      printf ("oops, error \"%s\" at line %d, char %d",
	      p.GetErrorStr().c_str(), p.GetErrorLine(), p.GetErrorCol());
      return -1;
    }
    return 0;
  }

  int writeObject (char *file) {
    if (!file || strcmp(file,"") == 0) {
      printf ("oops, missing output file name...");
      return -1;
    }
    SXP::CFileOutStream *o = new SXP::CFileOutStream (file);
    o->BeginXML ();
    SXP::dict attribs;
    this->WriteElement (o, attribs);
    // the destructor of o closes the file
    delete o;
    return 0;
  }
};

namespace XML { typedef SXP::Tag Tag; } // non-static XML::Tag = SXP::Tag

#define XML_TAG  static SXP::Tag
#define XML_ATTR static char *

#define XML_IO \
static SXP::Tag t_own; \
SXP::Tag& GetClassTag() const { return t_own; } \
void BeginElement(SXP::IParser *pIn, SXP::IElement *pElement); \
void EndElement(SXP::IParser *pIn, SXP::IElement *pElement);   \
void WriteElement(SXP::IOutStream *pOut, SXP::dict& attribs);

#include "xml_ptr.h"
#include "xml_list.h"

#endif
