/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*! \file xml_list.h
    \brief A simple XML "enabled" list class
*/

#ifndef SXP_LIST_H
#define SXP_LIST_H

#include <string>
#include <list>
#include <typeinfo>
#include "debug.h"

#include "xml.h"

namespace XML {


/*!
 \brief A simple XML "enabled" list class

 Instances of this class can be read from or written to an xml file using the methods
 <pre>
 read (char *file);
 write (char *file);
 </pre>
 This class is derived from std::list and the abstract xml interface "XmlObject",
 where the latter enforces the declaration and definition of read and write functions
 BeginElement, EndElement, WriteElement that are invoked by James Clark's expat xml
 parser through Ivan-Assen Ivanov's SXP package.

 The class needs a "buffer" of type T for reading from xml.

 Member "items" is always set equal to the size() of the list.
 It has been added for convenience, to show the list size explicitly in an xml tag.

 Member "trials" has been added for debugging purposes. It counts the number of
 "tried" appends to the list and is written to the log file when a buffer is refused
 by the append() function, because the buffer is not "valid". Any first append() will
 fail since the buffer will not yet be filled at that time, see functions BeginElement
 and EndElement below.
*/

template<class T>
class list : public XmlObject, public std::list<T> {
 private:
  T   buffer;
  int trials;
  int items;

  void msg (T t, int err) {
    static Debug log ("XML::list::msg");
    if (trials) log.msg (ALERT, "invalid buffer (err %d trials %d) not appended to list<%s>",
			 err, trials, typeid (t).name ());
    else        log.msg (DEBUG, "invalid buffer (err %d trials %d) not appended to list<%s>",
			 err, trials, typeid (t).name ());
  }

 public:

  list () {
    trials = 0;
    items = 0;
    std::list<T>::clear ();
    buffer.init ();
  }

  void append (XML::ptr<T> ptr) {
    static Debug log ("XML::list::append(ptr<T>)");
    int err;
    T * p = ptr.get();
    if ( (err = p->isValid ()) >= 0) {
      log.msg (DEBUG, "append list item type %s", typeid(*p).name());
      push_back (*p);
      items = std::list<T>::size ();
      buffer.init ();
    }
    else msg (*p, err);
    trials ++;
  }

  void append (T * t) {
    static Debug log ("XML::list::append(T)");
    int err;
    if ( (err=t->isValid ()) >= 0) {
      log.msg (DEBUG, "append list item type %s", typeid(*t).name());
      push_back (*t);
      items = std::list<T>::size ();
    }
    else msg (*t, err);
    trials ++;
  }

  void append (T t) {
    static Debug log ("XML::list::append(T)");
    int err;
    if ( (err = t.isValid ()) >= 0) {
      log.msg (DEBUG, "append list item type %s", typeid(t).name());
      push_back (t);
      items = std::list<T>::size ();
    }
    else msg (t, err);
    trials ++;
  }

  int read (char *file) {
    static Debug log ("XML::list::read()");
    std::list<T>::clear ();
    buffer.init ();
    trials = 0;
    log.msg (DEBUG, "starting");
    // receive the root element
    SXP::CParser p (this);
    // process the file
    if (p.FeedFile(file) != SXP::err_no_error) {
      log.msg (ALERT, "oops, error \"%s\" at line %d, char %d",
	      p.GetErrorStr().c_str(), p.GetErrorLine(), p.GetErrorCol());
      return -1;
    }
    log.msg (DEBUG, "done");
    return 0;
  }

  int write (char *file) {
    static Debug log ("XML::list::write()");
    log.msg (DEBUG, "starting");
    if (!file || strcmp(file,"") == 0) {
      log.msg (ALERT, "oops, missing output file name...");
      return 1;
    }
    SXP::CFileOutStream *o = new SXP::CFileOutStream (file);
    o->BeginXML ();
    SXP::dict attribs;
    this->WriteElement (o, attribs);
    // the destructor of o closes the file
    delete o;
    log.msg (DEBUG, "done");
    return 0;
  }

  // SXP ---

  void init() {
    std::list<T>::clear ();
    items = 0;
    trials = 0;
  }

  int isValid() {
    if (items) return  0;
    else       return -1;
  }

  static SXP::Tag t_own;
  static SXP::Tag t_items;

  // attributes ...
  // none

  SXP::Tag& GetClassTag() const { return t_own; }

  // read complex subtags
  void BeginElement(SXP::IParser *pIn, SXP::IElement *pElement) {
    if (SXP::debug) { printf( "<%s> ", pElement->Name() ); fflush(stdout); }
    if (pElement->IsA (T::t_own)) {
      // flush buffer into list before it is overwritten by next item
      append (buffer);
      buffer.init ();
      // redirect tag stream to child
      pIn->ReadTo (&buffer);
      // send child its own start tag
      buffer.BeginElement (pIn, pElement);
    }
  }

  // read simple subtags
  void EndElement (SXP::IParser *pIn, SXP::IElement *pElement) {
    if (SXP::debug) { printf( "</%s> ", pElement->Name() ); fflush(stdout); }
    if (pElement->IsA(t_items)) pElement->Retrieve (items);
    // do not forget to append the last list item ;-)
    if (pElement->IsA(t_own)) {
      append (buffer);
      buffer.init ();
    }
  }

  // write object
  void WriteElement (SXP::IOutStream *pOut, SXP::dict& attribs) {
    pOut->BeginObject (t_own, attribs);
    // write "simple" members
    pOut->WriteElement (t_items, items);
    // write "complex" members
    // child members of identical type will be saved using the same tag name
    // they have to be distinguished using an "id" attribute like in: <window id="toolbar">
    SXP::dict localAttribs;
    typename std::list<T>::iterator iter;
    for (iter = std::list<T>::begin(); iter != std::list<T>::end(); iter++) {
      pOut->WriteSubElement (&(*iter), localAttribs);
    }
    pOut->EndObject (t_own);
  }
};

} // eof namespace XML

#endif
