/* WARANTY NOTICE AND COPYRIGHT
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Copyright (C) Michael J. Meyer

matmjm@mindspring.com
spyqqqdia@yahoo.com

*/

#include "QuasiMonteCarlo.h"
#include "Utils.h"
#include "Array.h"
#include "Random.h"
#include <algorithm>
#include <math.h>                                 // gcc4, sqrt otherwise undefined

MTGL_BEGIN_NAMESPACE(Martingale)

/*******************************************************************************
 *
 *                     LOW DISCREPANCY SEQUENCE
 *
 ******************************************************************************/

LowDiscrepancySequence::
LowDiscrepancySequence(int d) :
dim(d), index(1), x(dim), z(dim)
{  }

void
LowDiscrepancySequence::
writeNextPoint(RealArray1D& r) {
  const RealArray1D& x=nextPoint();
  for(int k=0;k<dim;k++) r[k]=x[k];
}


// L^2-DISCREPANCY

// r[j][] is the jth point in the low discrepancy sequence
Real
LowDiscrepancySequence::
product(int i, int j, const RealArray2D& r) {
  Real f=1.0;
  for(int k=0;k<dim;k++)f*=(1.0-std::max<Real>(r(i,k),r(j,k)));
  return f;
}


// r[j][] is the jth point in the low discrepancy sequence
Real
LowDiscrepancySequence::
product(int i, const RealArray2D& r) {
  Real f=1.0;
  for(int k=0;k<dim;k++)f*=(1.0-r(i,k));
  return f;
}


// r[j][] is the jth point in the low discrepancy sequence
Real
LowDiscrepancySequence::
productSQ(int i, const RealArray2D& r) {
  Real f=1.0;
  for(int k=0;k<dim;k++)f*=(1.0-r(i,k)*r(i,k));
  return f;
}


// L_2 discrepancy, direct computation

Real
LowDiscrepancySequence::
l2Discrepancy(int N, const RealArray2D& r) {
  // compute a=1/2^d, b=1/3^d
  Real a=1.0, b=1.0;
  for(int k=0;k<dim;k++){ a/=2; b/=3; }

  Real sum_1=0.0, sum_2=0.0;

  // report progress on main loop
  LoopStatus ls("Computing L^2-discrepancy.");
  int iteration=0,
    iterations=N*(N-1)/2,                         // number of iterations in following loop
    one_prcnt=iterations/100;                     // loop status reported every 1% completed

  // first sum over 1<=i<j<=N
  for(int i=0;i<N;i++)
  for(int j=i+1;j<N;j++) {
    sum_1+=product(i,j,r);

    // progress report, algorithm is of order N^2
    iteration++;
    if(iteration % one_prcnt == 0) ls.consoleReport(iteration,iterations);
  }

  // first sum over 1<=i\neq j<=N
  sum_1*=2;

  // add in the terms for i=j
  for(int i=0;i<N;i++)sum_1+=product(i,r);

  sum_1/=(N*N);

  // second sum
  for(int i=0;i<N;i++)sum_2+=productSQ(i,r);
  sum_2/=N; sum_2*=(2*a);

  return sqrt(sum_1-sum_2+b);

}                                                 // end L2-discrepancy


// L_2 discrepancy, recursive computation

Real
LowDiscrepancySequence::
l2Discrepancy(int N, const RealArray2D& r, Real T_N) {

  // compute a=1/2^d, b=1/3^d
  Real a=1.0, b=1.0;
  for(int k=0;k<dim;k++){ a/=2.0; b/=3.0; }

  //    if(N==0){ Real two_h_1=product(0.0,r)-2.0*a*productSQ(0.0,r)+b;
  if(N==0) {
    Real two_h_1=product(0,r)-2.0*a*productSQ(0,r)+b;
    return sqrt(two_h_1);
  }

  Real f=N*T_N,
    h_N=0.5*f*f,
    h;                                            // h_{N+1}

  // h=h_{N+1} from x=h_N
  h=h_N;
  for(int i=0;i<N;i++)h+=product(i,N,r);
  h+=0.5*product(N,r);

  Real sum=0.0;
  for(int i=0;i<N;i++)sum+=productSQ(i,r);
  sum+=(N+1)*productSQ(N,r);
  sum*=a;

  h-=sum;
  h+=(N+0.5)*b;

  return sqrt(2.0*h)/(N+1);

}                                                 // end L2-discrepancy


// TRANSFORM UNIFORM -> MULTINORMAL

const RealArray1D&
LowDiscrepancySequence::
nextQuasiNormalVector() {
  const RealArray1D& u=nextPoint();
  for(int k=0;k<dim;k++)z[k]=Random::nInverse(u[k]);
  return z;
}


/*******************************************************************************
 *
 *                     NIEDERREITER-XING SEQUENCE
 *
 ******************************************************************************/

NX::
~NX() {
  for(int j=4;j<21;j++) {
    for(int k=0;k<j;k++) delete[] genMats[j-4][k];
    delete[] genMats[j-4];
  }
  delete[] genMats;
}


void
NX::
restart() {
  index=1;
  // return the integer vector to the initial state
  for(int j=0;j<dim;j++)x_int[j]=genMats[d-4][j][0];
}


const RealArray1D&
NX::
nextPoint() {
  // find the position k of the rightmost zero bit of index
  // 0<=k<M
  int k=0, n=index;
  while(n%2==1){ n=n>>1; k++; }

  for(int j=0;j<dim;j++) {
    x_int[j]^=genMats[d-4][j][k];
    x[j]=((Real)x_int[j])/N;
  }

  index++;
  return x;
}


// RL CHANGE: definition moved here after introducing gcc 4 on Tiger

const Real NX::N = 1073741842;                    // 2^30

// CONSTRUCTOR

NX::NX(int dim)
: LowDiscrepancySequence(dim),
x_int(dim),
genMats(new int**[17]),
d(max(dim,4)) {
  if(dim>20) {
    std::cerr << "NX: dimension must be <= 20. Aborting.";
    exit(0);
  }

  // each Cj=genMats[j] is a j by 30 matrix, j=4,...,20
  for(int j=4;j<21;j++) {

    genMats[j-4]=new int*[j];
    for(int k=0;k<j;k++) genMats[j-4][k]=new int[30];
  }

  // generator matrices C_dim(j) for each coordinate j in all dimensions
  // dim=4,5,...,20 encoded row by row.

  // gMR.................generator matrix rows.
  //  gMR[j][k] is the kth <i>row</i> of the generator matrix C(j)
  //  encoded as a decimal integer n by viewing the the sequence
  //  of zeros and ones in the row r_k=row_k(C(j)) as the digits of n
  //  in base 2 with the <i>most</i> significant digit being the first
  //  component of the row vector r_k.
  //  This is the download from the NX sequences site.
  //

  // DIMENSION dim=4,  generator matrices column encodings:

  int j=4;

  int C_4[4][30]= { {
    // C(0), columns:
      939524096,771751936,729808896,182452224,582483968,145620992,
      573276160,143319040,35829760,8957440,539110272,134777568,
      33694392,8423598,538976811,134744202,33686050,8421512,
      2105378,526344,131586,32896,536879136,134219784,
      33554946,8388736,2097184,524296,131074,32768
    },
    // C(1), columns:
    {
      1010580540,698984873,501358050,723004184,463969191,571187723,
      132511770,185808928,465315712,573148032,127769215,195797171,
      437911997,538968619,8386661,11745427,29107772,36276649,
      127770082,195799832,437918631,538976779,8357861,11766547,
      29203389,36413995,127535002,195038112,436214272,536879104
    },
    // C(2), columns:
    {
      757935405,151587081,835596750,582689467,53347118,34275851,
      1007535090,722055824,842080028,539109675,3944946,2826896,
      3277596,2097707,1069548493,699400889,823919405,573243913,
      1009925134,723528459,838873650,536879136,16320,10672,
      12572,8747,15410,11040,12800,8192
    },
    // C(3), columns:
    {
      469762048,721420288,230686720,162529280,823918592,573243392,
      63815680,45723648,53287936,34155264,1007553984,722143664,
      842150172,538976811,4178893,2732729,3219245,2238729,
      3944910,2826939,3277614,2097675,1069548530,699400848,
      823919388,573243947,1009925170,723528480,838873600,536879104
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_4[k][l];
  j++;

  // DIMENSION dim=5,  generator matrices column encodings:

  int C_5[5][30]= { {

      713031680,1028653056,276430848,620052480,854321152,1063687424,
      832587680,1045106516,468108990,593634153,455302564,588966884,
      831404242,508908875,570011537,447011402,58925105,22147102,
      711520267,1027043335,274395139,619589633,853889316,526625828,
      556899090,450414731,61695349,22003374,711319143,1026803237
    },

    {
      469762048,301989888,897581056,37748736,556793856,872939520,
      43384832,558432256,339001344,370679808,924604416,51127296,
      563352320,340279424,371599712,922817680,587236172,352338306,
      377498053,390075586,934283841,55051268,565463810,341337217,
      372011844,923021702,587340231,352404675,377528897,390090756
    },

    {
      444180378,582771426,386103319,820878768,768469037,464323199,
      540534784,147816993,431574890,613378415,440641348,585935378,
      231724032,321626113,957130649,762735301,11922288,24841932,
      431584074,613353990,440628151,585993112,231593821,321875383,
      956409753,761948869,9300848,19861196,419525450,636044067
    },

    {
      874725940,276345168,1005962939,404842328,44797378,390406220,
      719975565,573261292,995508545,418874694,907865252,322234891,
      226350248,200248046,996057078,201936327,719123621,922205246,
      327333406,761281382,321461212,767251478,664019960,1027015197,
      482492772,825457823,37181888,54617459,914535782,331364768
    },

    {
      1010287484,387092183,210512716,885098740,230022349,551664607,
      58382349,737801095,851645666,472257767,20236143,1039091209,
      58775360,44000576,1061407804,1021417497,1044512992,686817112,
      210706479,885241161,830590908,506889871,1048831907,677547034,
      209243052,497930786,3673294,3045292,1006772076,389294580
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_5[k][l];
  j++;

  // DIMENSION dim=6,  generator matrices column encodings:

  int C_6[6][30]= { {

      644587520,320281088,403904282,548421841,437785367,394981521,
      506554803,530293121,648244152,185106849,11569178,557480225,
      973123720,361462440,504459536,142843915,545798656,337900032,
      1040972186,196311331,413671560,562725384,572041248,35752578,
      2234536,139658,8728,556888961,973078562,898331042
    },

    {
      685031084,856664270,430555838,298298601,984050532,453855559,
      613834890,991631314,1068941538,74048175,151788468,813918083,
      973205740,441353078,646982048,835775508,562397940,735494680,
      129726984,352099622,830745068,555559847,2216356,5036425,
      10885656,21140430,33858208,537335420,453587814,441440925
    },

    {
      628217850,325454664,998314354,268731841,453835512,758493955,
      558219927,493683099,988488409,366666052,736178184,809509177,
      462094524,742319014,574824549,633529913,188824518,221007958,
      560132781,96960952,818606902,554926544,52675512,80590561,
      139458802,290959604,589418696,537223600,999014834,742375611
    },

    {
      448749995,998890424,413198730,94371370,334225648,107954880,
      955423729,849826956,143809279,92746299,191107394,559297372,
      464313148,388043114,458511115,558604207,53072112,660938837,
      995508297,349305116,810412340,538757973,2568240,4249692,
      9384126,18320328,58841356,827833309,992673974,371872246
    },

    {
      990132955,386692475,335605549,861087837,982651609,84928281,
      6917107,513462859,927225443,147357176,230147361,580863685,
      400855275,1050928175,593828120,332815393,779729968,513430758,
      591690417,173375938,441512876,572594787,53330721,44572176,
      63823260,35170609,51159688,719696755,445730208,84769844
    },

    {
      499122176,447741952,925761536,176029696,962341888,302734336,
      376373248,62016864,205310848,834126720,762654859,423141431,
      399325739,730034628,911033337,334795476,777243070,197128747,
      913256528,3945220,923566130,431370928,537112076,826011012,
      764805349,994803843,602680533,293929560,443642261,284450072
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_6[k][l];
  j++;

  // DIMENSION dim=7,  generator matrices column encodings:

  int C_7[7][30]= { {

      644587520,320281088,403904282,548421841,437785367,394981521,
      506554803,530293121,648244152,185106849,11569178,557480225,
      973123720,361462440,504459536,142843915,545798656,337900032,
      1040972186,196311331,413671560,562725384,572041248,35752578,
      2234536,139658,8728,556888961,973078562,898331042
    },

    {
      685031084,856664270,430555838,298298601,984050532,453855559,
      613834890,991631314,1068941538,74048175,151788468,813918083,
      973205740,441353078,646982048,835775508,562397940,735494680,
      129726984,352099622,830745068,555559847,2216356,5036425,
      10885656,21140430,33858208,537335420,453587814,441440925
    },

    {
      628217850,325454664,998314354,268731841,453835512,758493955,
      558219927,493683099,988488409,366666052,736178184,809509177,
      462094524,742319014,574824549,633529913,188824518,221007958,
      560132781,96960952,818606902,554926544,52675512,80590561,
      139458802,290959604,589418696,537223600,999014834,742375611
    },

    {
      448749995,998890424,413198730,94371370,334225648,107954880,
      955423729,849826956,143809279,92746299,191107394,559297372,
      464313148,388043114,458511115,558604207,53072112,660938837,
      995508297,349305116,810412340,538757973,2568240,4249692,
      9384126,18320328,58841356,827833309,992673974,371872246
    },

    {
      990132955,386692475,335605549,861087837,982651609,84928281,
      6917107,513462859,927225443,147357176,230147361,580863685,
      400855275,1050928175,593828120,332815393,779729968,513430758,
      591690417,173375938,441512876,572594787,53330721,44572176,
      63823260,35170609,51159688,719696755,445730208,84769844
    },

    {
      499122176,447741952,925761536,176029696,962341888,302734336,
      376373248,62016864,205310848,834126720,762654859,423141431,
      399325739,730034628,911033337,334795476,777243070,197128747,
      913256528,3945220,923566130,431370928,537112076,826011012,
      764805349,994803843,602680533,293929560,443642261,284450072
    },

    {
      394486004,974886838,170929084,939407551,831916941,1059229394,
      328736998,51297358,557459938,178298032,143307296,901863802,
      967646216,901529372,455215978,1069593776,999700512,925375062,
      328241918,10271058,176291970,34081282,134219784,178954250,
      142772744,167903914,134744194,896126085,958899621,1068804408
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_7[k][l];
  j++;

  // DIMENSION dim=8,  generator matrices column encodings:

  int C_8[8][30]= { {

      760554632,893269568,1018726252,397550846,486706887,709025900,
      691463814,897059424,771925038,48245314,445661732,365996202,
      738733570,180388576,964722734,537575554,167816200,144706240,
      723044396,895150218,771787264,48236704,137232522,142794760,
      8924672,557792,448710830,500318210,747110536,180912136
    },

    {
      191925899,850322866,184392687,549799844,1024443066,576347035,
      382532166,16627116,182150555,475711155,390430684,312432314,
      738311052,271793309,380782643,1020390196,815275293,540233864,
      292597895,239456792,478389643,808653086,557457846,40619436,
      129541521,246531490,821638469,472330680,978942525,1012590597
    },

    {
      883717684,96518021,652853286,466291355,693221929,545287968,
      966891722,388157465,457902881,1043440021,767486751,96647075,
      1015198876,817049866,59417384,1020938648,805505024,537001984,
      943048725,356580925,487641606,853040652,664254599,238742324,
      399224619,780317982,208309733,119593013,994773809,1043440053
    },

    {
      527160192,861192064,574201038,56565168,4568205,48833925,
      95077866,255844538,189274428,503510557,792005641,294530466,
      1024051984,853178650,1028795142,806055692,538444458,3240220,
      35861445,17606958,478325819,817042729,567019160,48393984,
      131186604,243438122,247019859,334236546,995232528,1043571127
    },

    {
      966096851,159603416,391853503,534263005,567925083,754632580,
      605300517,525481798,749929250,951300503,831759277,730950479,
      90347043,911409212,381929330,911329204,459292567,581683098,
      776501376,328445808,627119239,911278881,459320868,581640838,
      848849327,746103428,695461275,185665125,882556987,411067682
    },

    {
      592598110,477898738,153058420,620511864,746949233,214503603,
      1035856176,135138758,748241599,948069951,742923334,312806868,
      1040260755,351989255,286634848,439053624,592972212,44821798,
      935742183,83151353,626597915,911288994,458790296,581632030,
      848845844,746096664,881612722,840751721,266897418,974806028
    },

    {
      814597320,726846282,153113138,530614223,894963566,209244070,
      761266718,1027120870,391299851,447447865,651951006,624563529,
      109126031,888852356,141706978,447434656,600454415,44812218,
      658370173,830073561,506161194,348721804,695606583,241751322,
      179316754,214663084,1035206920,128477917,884028701,405904022
    },

    {
      212183505,575897462,674724232,907724773,335232225,249625608,
      406679306,978524888,801158541,983070477,82256637,246020036,
      111731374,885723697,756331345,445791144,601390858,43522354,
      290858668,872585938,626715393,912326453,458692903,580025402,
      842012457,751523504,529968469,738202308,884017946,405909655
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_8[k][l];
  j++;

  // DIMENSION dim=9,  generator matrices column encodings:

  int C_9[9][30]= { {

      696344576,177875968,198830848,332466784,765128780,786752711,
      700394490,710743797,80402060,582321570,531330516,600994053,
      250343324,166325309,469206004,127787017,350209407,959505056,
      402000190,101740930,58507941,533330381,53139563,235760460,
      927807685,1048967523,1039779877,893524859,774165468,417325468
    },

    {
      912973312,273086528,1043348344,304024545,144844181,373870990,
      620509713,1032943564,953994563,579754739,988202473,136063485,
      174254401,979221296,243294609,210750988,191980114,384136855,
      119830278,960709408,152810196,87797445,725033016,935489600,
      513691672,375388711,601961381,646256774,844711761,782078212
    },

    {
      1070658743,541351219,177168414,857506082,954298862,189990685,
      928237287,714563635,825097568,486326679,620190618,234665099,
      969800906,10921771,117651574,960455758,712481989,104218246,
      49191030,395016413,243053293,50889952,533282744,507581774,
      365648082,362117277,943766598,452105566,134382757,600036003
    },

    {
      468149054,356323242,531691055,533479845,369430665,1050335897,
      855633172,121777424,926972951,861803436,633315042,147299592,
      44904118,716540604,848053812,161692489,18786120,175446231,
      329890245,289816975,992443882,144833905,280706725,393983156,
      1024471095,244194217,705811717,343438231,1010252774,369916015
    },

    {
      47463531,688500491,156046897,840884700,909106686,269785563,
      239678700,930221356,503006993,979373410,786124814,22097096,
      585759744,310525712,75113259,233079227,876226168,352865085,
      796223239,120697769,233031239,848080728,960129683,786952936,
      709641922,119053788,66120245,256432093,854840906,1062655013
    },

    {
      823931987,369032796,955029757,104856506,770334010,240341623,
      205772931,911259079,915206596,930217353,690026418,625901845,
      48782362,390178869,852575964,836019537,694715425,89906349,
      150902223,5834217,11009007,485593977,400623134,1069427788,
      691791464,405765657,1019838073,551252580,986644545,225903139
    },

    {
      382091319,739313283,942124987,288243186,229802861,751574681,
      41151136,730405574,502619122,149340930,382496547,383277427,
      951451155,587583426,191540436,308805631,1064908987,81380314,
      127274604,62709513,46705551,1032153411,829062413,747188657,
      444648745,199504873,237992630,378356548,772791966,833105476
    },

    {
      587078599,203960422,511194114,755679438,139542925,28367724,
      165315148,768111470,999912497,1027667728,638438543,359237201,
      196888839,463057534,268847481,214959631,431660447,480659247,
      337817622,224901951,151877581,263454068,928091576,989398959,
      722566738,924770779,629317449,713695576,181619785,564028034
    },

    {
      453809691,380853116,899434263,666973159,182611982,1008223646,
      637955544,153759368,411114377,566828272,44056958,585359217,
      34385615,801789890,507572939,48458184,992087037,411762415,
      286362362,826271647,604665201,298685982,211695003,664079058,
      265179510,352453822,1031228380,533517717,914087299,668725875
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_9[k][l];
  j++;

  // DIMENSION dim=10,  generator matrices column encodings:

  int C_10[10][30]= { {

      696344576,177875968,198830848,332466784,765128780,786752711,
      700394490,710743797,80402060,582321570,531330516,600994053,
      250343324,166325309,469206004,127787017,350209407,959505056,
      402000190,101740930,58507941,533330381,53139563,235760460,
      927807685,1048967523,1039779877,893524859,774165468,417325468
    },

    {
      912973312,273086528,1043348344,304024545,144844181,373870990,
      620509713,1032943564,953994563,579754739,988202473,136063485,
      174254401,979221296,243294609,210750988,191980114,384136855,
      119830278,960709408,152810196,87797445,725033016,935489600,
      513691672,375388711,601961381,646256774,844711761,782078212
    },

    {
      1070658743,541351219,177168414,857506082,954298862,189990685,
      928237287,714563635,825097568,486326679,620190618,234665099,
      969800906,10921771,117651574,960455758,712481989,104218246,
      49191030,395016413,243053293,50889952,533282744,507581774,
      365648082,362117277,943766598,452105566,134382757,600036003
    },

    {
      468149054,356323242,531691055,533479845,369430665,1050335897,
      855633172,121777424,926972951,861803436,633315042,147299592,
      44904118,716540604,848053812,161692489,18786120,175446231,
      329890245,289816975,992443882,144833905,280706725,393983156,
      1024471095,244194217,705811717,343438231,1010252774,369916015
    },

    {
      47463531,688500491,156046897,840884700,909106686,269785563,
      239678700,930221356,503006993,979373410,786124814,22097096,
      585759744,310525712,75113259,233079227,876226168,352865085,
      796223239,120697769,233031239,848080728,960129683,786952936,
      709641922,119053788,66120245,256432093,854840906,1062655013
    },

    {
      823931987,369032796,955029757,104856506,770334010,240341623,
      205772931,911259079,915206596,930217353,690026418,625901845,
      48782362,390178869,852575964,836019537,694715425,89906349,
      150902223,5834217,11009007,485593977,400623134,1069427788,
      691791464,405765657,1019838073,551252580,986644545,225903139
    },

    {
      382091319,739313283,942124987,288243186,229802861,751574681,
      41151136,730405574,502619122,149340930,382496547,383277427,
      951451155,587583426,191540436,308805631,1064908987,81380314,
      127274604,62709513,46705551,1032153411,829062413,747188657,
      444648745,199504873,237992630,378356548,772791966,833105476
    },

    {
      587078599,203960422,511194114,755679438,139542925,28367724,
      165315148,768111470,999912497,1027667728,638438543,359237201,
      196888839,463057534,268847481,214959631,431660447,480659247,
      337817622,224901951,151877581,263454068,928091576,989398959,
      722566738,924770779,629317449,713695576,181619785,564028034
    },

    {
      453809691,380853116,899434263,666973159,182611982,1008223646,
      637955544,153759368,411114377,566828272,44056958,585359217,
      34385615,801789890,507572939,48458184,992087037,411762415,
      286362362,826271647,604665201,298685982,211695003,664079058,
      265179510,352453822,1031228380,533517717,914087299,668725875
    },

    {
      931239939,828945438,363684192,95705596,248262652,526274471,
      61950506,542401026,593450089,139006496,220316088,1019960300,
      192398181,834630841,1072820851,474696622,1017601997,925673718,
      774313960,12886932,8871591,362158636,293953661,573034866,
      595008664,935547810,757597401,7383456,476631037,577177943
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_10[k][l];
  j++;

  // DIMENSION dim=11,  generator matrices column encodings:

  int C_11[11][30]= { {

      640076413,38619923,880276074,258462862,218061951,110875386,
      782095512,253820603,543879802,1050676528,655726172,1061996849,
      738961624,599680434,1010784250,414752449,180264169,676013088,
      361315153,950490779,626020133,142567564,521541876,618804908,
      178037201,362432873,391430156,528491869,1063457085,141830450
    },

    {
      393823488,967702564,996414134,380720145,300495822,884499191,
      768736405,91406721,741603631,624224565,359456934,89283930,
      537918092,829699217,349844160,264781100,1031472974,300442722,
      219489904,623147931,359027908,503369389,414890524,46246130,
      734200851,550896721,990511752,756780245,917175879,319155042
    },

    {
      606494016,917965809,1018376187,163092929,747799491,115078791,
      939997329,17414853,302426061,921867709,602154136,999186600,
      411725464,223769413,703344350,841532862,797735996,1020553390,
      1009041997,950379831,627636356,178720078,409380538,70557298,
      673255255,835970611,286203702,266426382,560101456,39390863
    },

    {
      600866816,685960568,103280453,309088841,955194087,1044309885,
      605453218,957769029,348362389,812806796,302838834,313274906,
      300197590,275279179,332855626,37604359,698428890,206208084,
      238230667,367047777,4381534,735883523,1064951460,203911503,
      78225905,998688573,340732841,941500948,201590684,371703833
    },

    {
      346200071,734643499,123089088,649304842,605054314,790301134,
      48561423,242319144,801905870,66727240,424813982,158270750,
      162634162,146649881,731766097,380723582,33118141,861489947,
      741323110,237146728,263233061,427681584,870037056,474509960,
      337598642,712260834,886629205,541733652,450393070,280197089
    },

    {
      986274698,148131086,870716920,447282332,1026196869,472749574,
      32460871,272231963,410368694,178649215,888633116,1041483275,
      940674463,931538664,956736924,524225383,936411974,969547861,
      512648518,381104443,205372356,13905180,333072039,496115283,
      546594229,1065067452,179675074,227185735,993747182,39536207
    },

    {
      261598117,611550090,211615492,100722117,634969513,368972092,
      686506470,533599043,530639277,873784702,575933634,411444686,
      102047956,816671590,1010229544,185842073,978642477,562673342,
      1003690264,941331870,814018105,637603710,143573597,1037408318,
      307920125,339901701,832688107,722546136,412159266,769219252
    },

    {
      922063332,565134652,82071906,769088394,944402391,138740959,
      433429701,862296561,993983047,899835002,1037945714,417060381,
      752393547,88184049,966044098,680894913,514682970,719476347,
      886465807,486029512,62535029,555167353,307430836,709660495,
      1041707699,786398307,825953853,750169210,221302275,595424191
    },

    {
      512467929,300113655,674083661,689445985,148105911,866522716,
      313630486,395717289,301814491,655585295,577555299,917231748,
      888920033,927718938,724360554,377554296,248732447,174080875,
      559178264,89877466,399554243,298097523,363683154,825448247,
      854096856,766321194,215392664,646036130,1892589,311854397
    },

    {
      986027855,132090846,621223411,333974992,790598050,79387364,
      267995867,599454495,232907008,15913258,302191763,772790356,
      584503088,738046272,188438137,921819200,697430148,137752459,
      873342196,227078802,442976560,1034616614,172909972,40524786,
      747034583,462482257,28803124,462481449,885322935,182269347
    },

    {
      648793944,607217858,974232141,283030681,2576790,535183650,
      157523670,496460795,334546972,8482709,896794205,786499822,
      228269069,998643015,607626918,151122715,717253132,1010802792,
      529041112,437142038,545850927,663932102,587251879,161482998,
      958459306,918804181,617570748,471288770,418354821,666780001
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_11[k][l];
  j++;

  // DIMENSION dim=12,  generator matrices column encodings:

  int C_12[12][30]= { {

      376009787,616649261,1038915818,449176030,680969485,844929903,
      469210199,515424615,955287332,441067371,672960609,10667578,
      518196311,909235475,829362448,606869120,1069286697,548650217,
      281564703,956233651,30374325,339266501,406232992,308994419,
      308329741,419071262,157656941,113798237,580526181,312086276
    },

    {
      113325173,802884388,776006141,240665347,1067428575,237210076,
      433405280,542548953,206172120,892707582,938920668,268533774,
      427179081,64599497,584946987,850670289,1029168313,406383405,
      567739064,243337945,534671208,284667432,604603719,624171720,
      638094189,480469014,81680586,875930397,165521623,713379925
    },

    {
      365589568,795832513,352548612,791353731,639030956,875424815,
      460322970,209202842,769072557,62593514,554917907,422257425,
      988433200,845378618,964807902,469755058,490235585,36511719,
      305707082,338554719,877932899,475033153,48231578,414880648,
      706291263,70011941,255603371,718605877,1049680271,203585716
    },

    {
      435632494,172170459,1059343654,565249317,65081497,498668119,
      696359679,815656005,627318345,886975033,230810440,958457568,
      170678992,928599693,50537551,95312547,130331293,202074131,
      1045327915,1032883628,259200551,538263735,525010700,117358056,
      867099869,70373620,495094033,522626779,38232465,95758096
    },

    {
      32678341,505635296,425856624,1054825470,1033723186,938870058,
      269291076,814524247,656722885,524631983,597477638,436880436,
      697998740,64815620,786707110,297283695,494619587,996221642,
      743211695,532501720,485366627,575598237,246747572,221087582,
      195011953,225867526,624744711,494497451,771083163,465002468
    },

    {
      551397128,489766929,195538897,629476791,1006219829,921379579,
      836280603,978070866,1007507930,236561171,759639615,439796207,
      338293708,24298018,558561689,657466702,188516653,553216800,
      981177690,1010977003,791689962,158536313,388269785,487774813,
      224170145,624982216,529627459,530281195,133158627,81743442
    },

    {
      768717957,184677116,591120478,258801695,981458210,434399026,
      697605612,787798639,845498401,27242131,198461923,975748095,
      873564500,808362448,829314929,599428700,553956900,646455049,
      715447491,990180123,901857731,341740714,518651244,433436011,
      393492963,156099629,1014527846,267932060,338771950,491714313
    },

    {
      609470369,578981355,496170689,646129714,870096655,49561033,
      642998690,66791072,319166205,892427797,521637492,815247648,
      244676424,701940472,831147871,570779345,89071257,880115963,
      152451089,13233731,318042497,408461437,1007221712,775537592,
      174347183,811508503,942904086,319793289,566294292,956161085
    },

    {
      690191426,28134813,207393949,406243981,500443338,244003934,
      327631593,692702121,76532237,120264433,305592447,23783510,
      936434940,129432986,891044703,133326731,759067238,805090750,
      393846010,895707158,773285066,682629395,68829201,219221301,
      116077786,1009888418,978576807,641438017,169610131,526023641
    },

    {
      226253065,147381635,1009733192,730981142,441127473,1068685129,
      902159719,362034294,888137773,862283775,338017958,22628013,
      340658897,130197395,630654134,42912700,324565062,640826151,
      540055184,214577500,940181002,475921976,880471073,962578486,
      937098890,1035691007,496051649,546273064,104382506,758339435
    },

    {
      980220447,391809923,926256294,896109598,310914840,979419742,
      492998431,917440954,1036679165,877461802,482469020,413101932,
      360109925,34299936,186285141,291647287,420363989,406330999,
      854416866,979161387,721088013,775806082,25986602,601668135,
      282692574,256034532,1047609668,427797668,402427615,284346209
    },

    {
      805428583,869599887,1014763568,659252565,973942038,754651348,
      897847746,182137879,679510472,128246679,807357794,396531141,
      578604545,486979889,898367645,336668900,827682136,327305202,
      213888167,744078157,351257983,316843621,473816217,495813680,
      483886281,885162661,246466767,823380730,817597230,113843058
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_12[k][l];
  j++;

  // DIMENSION dim=13,  generator matrices column encodings:

  int C_13[13][30]= { {

      376009787,616649261,1038915818,449176030,680969485,844929903,
      469210199,515424615,955287332,441067371,672960609,10667578,
      518196311,909235475,829362448,606869120,1069286697,548650217,
      281564703,956233651,30374325,339266501,406232992,308994419,
      308329741,419071262,157656941,113798237,580526181,312086276
    },

    {
      113325173,802884388,776006141,240665347,1067428575,237210076,
      433405280,542548953,206172120,892707582,938920668,268533774,
      427179081,64599497,584946987,850670289,1029168313,406383405,
      567739064,243337945,534671208,284667432,604603719,624171720,
      638094189,480469014,81680586,875930397,165521623,713379925
    },

    {
      365589568,795832513,352548612,791353731,639030956,875424815,
      460322970,209202842,769072557,62593514,554917907,422257425,
      988433200,845378618,964807902,469755058,490235585,36511719,
      305707082,338554719,877932899,475033153,48231578,414880648,
      706291263,70011941,255603371,718605877,1049680271,203585716
    },

    {
      435632494,172170459,1059343654,565249317,65081497,498668119,
      696359679,815656005,627318345,886975033,230810440,958457568,
      170678992,928599693,50537551,95312547,130331293,202074131,
      1045327915,1032883628,259200551,538263735,525010700,117358056,
      867099869,70373620,495094033,522626779,38232465,95758096
    },

    {
      32678341,505635296,425856624,1054825470,1033723186,938870058,
      269291076,814524247,656722885,524631983,597477638,436880436,
      697998740,64815620,786707110,297283695,494619587,996221642,
      743211695,532501720,485366627,575598237,246747572,221087582,
      195011953,225867526,624744711,494497451,771083163,465002468
    },

    {
      551397128,489766929,195538897,629476791,1006219829,921379579,
      836280603,978070866,1007507930,236561171,759639615,439796207,
      338293708,24298018,558561689,657466702,188516653,553216800,
      981177690,1010977003,791689962,158536313,388269785,487774813,
      224170145,624982216,529627459,530281195,133158627,81743442
    },

    {
      768717957,184677116,591120478,258801695,981458210,434399026,
      697605612,787798639,845498401,27242131,198461923,975748095,
      873564500,808362448,829314929,599428700,553956900,646455049,
      715447491,990180123,901857731,341740714,518651244,433436011,
      393492963,156099629,1014527846,267932060,338771950,491714313
    },

    {
      609470369,578981355,496170689,646129714,870096655,49561033,
      642998690,66791072,319166205,892427797,521637492,815247648,
      244676424,701940472,831147871,570779345,89071257,880115963,
      152451089,13233731,318042497,408461437,1007221712,775537592,
      174347183,811508503,942904086,319793289,566294292,956161085
    },

    {
      690191426,28134813,207393949,406243981,500443338,244003934,
      327631593,692702121,76532237,120264433,305592447,23783510,
      936434940,129432986,891044703,133326731,759067238,805090750,
      393846010,895707158,773285066,682629395,68829201,219221301,
      116077786,1009888418,978576807,641438017,169610131,526023641
    },

    {
      226253065,147381635,1009733192,730981142,441127473,1068685129,
      902159719,362034294,888137773,862283775,338017958,22628013,
      340658897,130197395,630654134,42912700,324565062,640826151,
      540055184,214577500,940181002,475921976,880471073,962578486,
      937098890,1035691007,496051649,546273064,104382506,758339435
    },

    {
      980220447,391809923,926256294,896109598,310914840,979419742,
      492998431,917440954,1036679165,877461802,482469020,413101932,
      360109925,34299936,186285141,291647287,420363989,406330999,
      854416866,979161387,721088013,775806082,25986602,601668135,
      282692574,256034532,1047609668,427797668,402427615,284346209
    },

    {
      805428583,869599887,1014763568,659252565,973942038,754651348,
      897847746,182137879,679510472,128246679,807357794,396531141,
      578604545,486979889,898367645,336668900,827682136,327305202,
      213888167,744078157,351257983,316843621,473816217,495813680,
      483886281,885162661,246466767,823380730,817597230,113843058
    },

    {
      775536582,900718787,360197547,657428423,989613149,276818072,
      35434343,207452155,929686556,379446274,350997024,403659830,
      201733068,183406655,581133571,783416532,685549740,39396776,
      872208726,306668169,749911467,1003397486,871981435,563905409,
      1059652571,635124084,913427898,606299715,882851360,902877678
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_13[k][l];
  j++;

  // DIMENSION dim=14,  generator matrices column encodings:

  int C_14[14][30]= { {

      122455510,773296462,80386005,883424770,513326595,981703937,
      1038812148,243570348,778200383,102755401,595654303,880576513,
      296182727,1055400600,996075111,1052355578,964134372,482021623,
      1010269190,146064639,699374654,545735549,783309448,241955223,
      391613045,316095920,691188546,107203704,913774004,919011693
    },

    {
      904829865,125800112,744836867,56211074,962495785,1014331605,
      31766538,1036440779,378917045,863948098,170171268,150539741,
      291559995,144340146,936281598,725120753,165033544,957473065,
      142516263,917214920,406122355,215612829,806188527,444563495,
      94075866,902187783,921358076,515829670,842057841,754580171
    },

    {
      144252053,719681002,764865240,360782177,8473587,170796252,
      322670896,824841090,715394417,308461745,767985005,603286018,
      282278856,370444133,799573691,207061925,378626792,443460925,
      747932389,190656701,112052892,838906526,456638239,308815925,
      314952227,615497055,197589208,851292505,114291449,749275465
    },

    {
      508763882,863822867,579200240,341866102,1020791415,952571345,
      739394084,867999885,527833410,840385800,558226716,581629303,
      539677350,61386309,344675504,571043943,679597514,131099271,
      983978597,401123019,825124467,1023625675,887540408,932091587,
      134304604,538330537,662740073,203901635,988267545,346153028
    },

    {
      621391692,190216341,240288833,47344896,848418599,660064937,
      724208392,17567804,947270235,290260854,65728614,661215123,
      409991240,708141568,308244599,200143731,276033077,525109062,
      395293500,464968947,618144500,326508566,615516113,557986837,
      45672459,435884540,701004880,63546760,785500141,862516802
    },

    {
      943528403,7424764,444055955,410453694,92186155,283874613,
      17575750,133964545,995073250,957980469,277245753,188272488,
      587777482,95089141,581464409,824556694,437316241,640484097,
      696373407,412102094,239622123,522038752,228604636,453088305,
      1071365639,498667903,869393790,449562132,1033358097,817761061
    },

    {
      1023859687,860420336,437777716,959192061,251318774,343778092,
      413416680,259452981,676804407,122487975,95708507,1054678216,
      363422616,466995666,325488046,373888941,515222328,781087025,
      997868815,459829544,41014936,226257344,331113322,597700410,
      963882199,450825004,1020970018,804124363,40896571,505564148
    },

    {
      755137672,742859173,936475141,354522321,797948600,730638890,
      204492398,159152738,471715321,123471289,839297694,642348689,
      750583992,76497646,744562819,577560611,139554140,626389783,
      521865008,377281746,344637867,286741651,324330987,196287246,
      804548810,417370381,309802854,984272094,97243419,1025617989
    },

    {
      510369304,236761812,13110549,616873707,521953617,792634244,
      531230079,775306128,407484179,747791618,694386176,331508215,
      818343003,781023012,308300789,805796298,944521705,592032883,
      269937303,405771211,145315315,973429799,145186174,1009885150,
      723080138,412171449,770686949,674735092,953203371,379728005
    },

    {
      1070788282,363613908,401556515,737472501,916971534,741851784,
      15970839,575437740,308634173,456365136,610081937,655056737,
      566499647,146062168,11265212,228034960,802031966,1016657337,
      869981249,137320736,78544281,829883313,783146120,970701867,
      916026634,195792316,317440031,798963377,662445144,432575740
    },

    {
      512569269,413297986,617992871,699943554,930550099,360306596,
      942139075,636897660,837676953,680872486,958145976,99814098,
      1030494902,481157406,7043250,631984833,693125308,721326947,
      543308348,845963501,280625890,62529592,1069504905,437525669,
      15289504,1055367702,1030656218,1046177956,480251288,524111444
    },

    {
      992641068,539842223,236658812,663216381,130221500,781554959,
      707555422,697175772,211358555,98635777,101893269,685395385,
      212974975,754694168,348080315,797978796,610517654,353789515,
      740861326,561595503,738001445,148519155,1012540275,866018055,
      814844210,985429376,726145203,572582243,242483828,104550108
    },

    {
      1011761701,257012084,545603242,377614867,1001660721,374400720,
      330700341,414707616,712426496,517978447,955401637,627264514,
      163257057,560991681,704985699,647379336,71773275,710543791,
      1023151414,22605165,301433606,348091287,20570843,637748854,
      761625262,348250326,509470220,66005223,199807360,904227117
    },

    {
      30303575,619474536,775105061,606619342,1020249710,1007526750,
      1028531643,325064071,547200869,136261058,931704242,479402249,
      970006018,326468943,265514350,545747566,716265983,1066574193,
      925563822,652793902,970474175,832205384,982810047,1048889944,
      439489154,226597362,888136689,138618060,814046664,550833263
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_14[k][l];
  j++;

  // DIMENSION dim=15,  generator matrices column encodings:

  int C_15[15][30]= { {

      122455510,773296462,80386005,883424770,513326595,981703937,
      1038812148,243570348,778200383,102755401,595654303,880576513,
      296182727,1055400600,996075111,1052355578,964134372,482021623,
      1010269190,146064639,699374654,545735549,783309448,241955223,
      391613045,316095920,691188546,107203704,913774004,919011693
    },

    {
      904829865,125800112,744836867,56211074,962495785,1014331605,
      31766538,1036440779,378917045,863948098,170171268,150539741,
      291559995,144340146,936281598,725120753,165033544,957473065,
      142516263,917214920,406122355,215612829,806188527,444563495,
      94075866,902187783,921358076,515829670,842057841,754580171
    },

    {
      144252053,719681002,764865240,360782177,8473587,170796252,
      322670896,824841090,715394417,308461745,767985005,603286018,
      282278856,370444133,799573691,207061925,378626792,443460925,
      747932389,190656701,112052892,838906526,456638239,308815925,
      314952227,615497055,197589208,851292505,114291449,749275465
    },

    {
      508763882,863822867,579200240,341866102,1020791415,952571345,
      739394084,867999885,527833410,840385800,558226716,581629303,
      539677350,61386309,344675504,571043943,679597514,131099271,
      983978597,401123019,825124467,1023625675,887540408,932091587,
      134304604,538330537,662740073,203901635,988267545,346153028
    },

    {
      621391692,190216341,240288833,47344896,848418599,660064937,
      724208392,17567804,947270235,290260854,65728614,661215123,
      409991240,708141568,308244599,200143731,276033077,525109062,
      395293500,464968947,618144500,326508566,615516113,557986837,
      45672459,435884540,701004880,63546760,785500141,862516802
    },

    {
      943528403,7424764,444055955,410453694,92186155,283874613,
      17575750,133964545,995073250,957980469,277245753,188272488,
      587777482,95089141,581464409,824556694,437316241,640484097,
      696373407,412102094,239622123,522038752,228604636,453088305,
      1071365639,498667903,869393790,449562132,1033358097,817761061
    },

    {
      1023859687,860420336,437777716,959192061,251318774,343778092,
      413416680,259452981,676804407,122487975,95708507,1054678216,
      363422616,466995666,325488046,373888941,515222328,781087025,
      997868815,459829544,41014936,226257344,331113322,597700410,
      963882199,450825004,1020970018,804124363,40896571,505564148
    },

    {
      755137672,742859173,936475141,354522321,797948600,730638890,
      204492398,159152738,471715321,123471289,839297694,642348689,
      750583992,76497646,744562819,577560611,139554140,626389783,
      521865008,377281746,344637867,286741651,324330987,196287246,
      804548810,417370381,309802854,984272094,97243419,1025617989
    },

    {
      510369304,236761812,13110549,616873707,521953617,792634244,
      531230079,775306128,407484179,747791618,694386176,331508215,
      818343003,781023012,308300789,805796298,944521705,592032883,
      269937303,405771211,145315315,973429799,145186174,1009885150,
      723080138,412171449,770686949,674735092,953203371,379728005
    },

    {
      1070788282,363613908,401556515,737472501,916971534,741851784,
      15970839,575437740,308634173,456365136,610081937,655056737,
      566499647,146062168,11265212,228034960,802031966,1016657337,
      869981249,137320736,78544281,829883313,783146120,970701867,
      916026634,195792316,317440031,798963377,662445144,432575740
    },

    {
      512569269,413297986,617992871,699943554,930550099,360306596,
      942139075,636897660,837676953,680872486,958145976,99814098,
      1030494902,481157406,7043250,631984833,693125308,721326947,
      543308348,845963501,280625890,62529592,1069504905,437525669,
      15289504,1055367702,1030656218,1046177956,480251288,524111444
    },

    {
      992641068,539842223,236658812,663216381,130221500,781554959,
      707555422,697175772,211358555,98635777,101893269,685395385,
      212974975,754694168,348080315,797978796,610517654,353789515,
      740861326,561595503,738001445,148519155,1012540275,866018055,
      814844210,985429376,726145203,572582243,242483828,104550108
    },

    {
      1011761701,257012084,545603242,377614867,1001660721,374400720,
      330700341,414707616,712426496,517978447,955401637,627264514,
      163257057,560991681,704985699,647379336,71773275,710543791,
      1023151414,22605165,301433606,348091287,20570843,637748854,
      761625262,348250326,509470220,66005223,199807360,904227117
    },

    {
      30303575,619474536,775105061,606619342,1020249710,1007526750,
      1028531643,325064071,547200869,136261058,931704242,479402249,
      970006018,326468943,265514350,545747566,716265983,1066574193,
      925563822,652793902,970474175,832205384,982810047,1048889944,
      439489154,226597362,888136689,138618060,814046664,550833263
    },

    {
      342919228,684064825,70595553,821444772,1067772687,228365088,
      875508409,523521168,568857749,27801078,14205486,54170645,
      762348034,442111606,779018993,91884115,96334799,834145058,
      517198826,311189041,351655502,129132263,71651785,69478110,
      877457343,69399165,663085682,432724522,901588315,1036418771
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_15[k][l];
  j++;

  // DIMENSION dim=16,  generator matrices column encodings:

  int C_16[16][30]= { {

      122028500,773796640,79828983,881800358,514905835,983783335,
      1036076530,244094596,778726717,102223501,595753535,877977665,
      293940545,1052797592,998015535,1049706940,961932202,479425179,
      1008703202,146074807,699208338,545769971,783872744,241400761,
      394095793,318234454,691017632,107693692,912226258,921650147
    },

    {
      905747129,125669856,745948482,56213465,962367329,1013342451,
      33005913,1037353434,377933797,863814511,170368760,150537961,
      291553302,143356869,937198823,724066463,163851778,956426595,
      141337432,918194684,406318084,215609507,806253540,443391870,
      94009789,903116998,921416576,514601025,842089657,754668748
    },

    {
      144510343,719814161,764739186,360776817,8475867,170405149,
      322931744,825103235,715522888,308466584,767859270,603415209,
      282275890,370575710,799840954,206934734,378632555,443071164,
      748066014,190526844,111925134,838907447,456500383,309079437,
      315082098,615099103,197455650,851153078,113882241,749580267
    },

    {
      508694523,864018879,579267946,342582253,1020795729,953030849,
      739794289,867481205,527701809,840844607,558748283,581697196,
      539742130,61383864,344610789,571041724,679594220,131096939,
      983843120,400800768,824994954,1023559165,888130186,931957442,
      134307115,538930046,662758996,203788510,988761297,346276074
    },

    {
      621389988,190147614,240354817,47413693,849070421,659871401,
      724277355,17702764,947205554,289740834,65137593,660818834,
      409403056,708667867,308049550,199617242,275965133,524584535,
      395099063,465555703,618149148,326449146,616170625,558444702,
      45281988,435823313,700583419,63138695,785407875,862058847
    },

    {
      1024840886,860416911,437581387,960368629,251385071,342867483,
      414589371,259388234,675558717,122625022,95841389,1054808264,
      363428799,466082259,325425371,374938769,514306939,782072930,
      997935942,459831567,39967148,226387339,331250224,597702156,
      963817961,450635098,1021034056,805254254,39948930,505476103
    },

    {
      943528130,6377416,442877338,409279699,92188770,283745105,
      18625358,132919617,995069163,957055041,278160177,187351096,
      588960203,94958252,582645249,825737187,437319588,640489793,
      696376298,411055566,240803234,520987896,228602589,454136105,
      1070180458,498802178,868346947,449441731,1033212799,816755189
    },

    {
      755131930,742733924,936471700,354519697,797680387,730378704,
      204496645,158891697,471453056,123212969,839163918,642739562,
      750322088,76238277,744564601,577303792,139553741,626517950,
      522000409,377015466,344773034,286611154,324463697,196282908,
      804678176,417369934,309669814,984410809,97361422,1025588420
    },

    {
      509774441,236305777,13111597,617457706,521501292,792079340,
      530609804,775923580,406928336,748418988,694905554,332067769,
      818309335,780466121,308368901,805729970,943997647,591997436,
      270497271,405771136,144758335,973980081,145641057,1009847578,
      723049798,412204818,770093665,675196944,953790072,380349499
    },

    {
      1071829370,363611179,400021605,737468978,915884092,739197071,
      12782863,578052546,308135759,453717337,611167682,655090680,
      570175856,143441666,10178396,228528946,804652632,1016134480,
      869977617,137349722,78541999,829917227,782058450,971718430,
      916580062,195768456,316897570,799508767,662448934,435202045
    },

    {
      514439921,411228727,618086450,697847559,930424358,358576228,
      942010818,636808815,835672190,682739891,958268136,99823447,
      1028538355,480855611,7031412,632123921,693133855,721035685,
      543473531,843998686,278569684,62554824,1067281322,439356997,
      13233123,1053001636,1028403740,1046301216,480255387,524286967
    },

    {
      992744004,539739079,236795785,663216675,130224767,781522855,
      707424299,696815199,211785248,98668673,101863869,685563385,
      212940913,754823102,347687424,797980002,610518958,353825990,
      741254507,562021180,737903974,148684693,1012708501,865658034,
      814815730,985298406,726535512,572583637,242483404,104681103
    },

    {
      410584653,1014456740,859507316,528229087,21041245,451335112,
      273376174,370447156,145225655,100856626,216137927,880988295,
      989591832,498229517,752029644,692203068,378212597,210534553,
      1006542675,571190962,414912293,605262338,705870606,973888734,
      652138388,898974866,574612904,349338863,639882306,1002077865
    },

    {
      1011833690,256942639,546060759,378076728,1002288904,374437599,
      331255010,414611228,713020731,517449415,955993684,627269219,
      163250620,561092153,705439834,647441968,71777603,710053662,
      1023152908,22606250,301863081,347532963,20017716,637718071,
      761555051,348747465,509441421,65484519,199315376,904785068
    },

    {
      342919423,684098882,70628500,821483431,1067777746,228200923,
      875147916,523525120,569279558,28225579,14565382,54301638,
      762352194,442241525,778625199,92020179,96204354,834571042,
      517199569,311086769,351618653,129164929,71684553,69476856,
      877292194,69366427,663054204,432857041,901460654,1036450973
    },

    {
      32268518,619642010,775359809,604787067,1020096905,1009492987,
      1030488316,323225398,549425125,136161349,934053943,479673465,
      972140115,324513836,267614303,547742461,716557290,1068906183,
      925792527,650693947,972834248,834331210,982702238,1051098972,
      437684209,226568545,887965313,138875768,816273500,553069006
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_16[k][l];
  j++;

  // DIMENSION dim=17,  generator matrices column encodings:

  int C_17[17][30]= { {

      129971866,403034142,42953265,456052922,529723536,39622889,
      1022405131,495125695,205765660,236811057,61590560,562939695,
      1009383060,28432675,166824630,956664215,988888538,657242377,
      1023579691,437431770,927205331,608312395,1072853475,655700129,
      27075613,636512015,837235128,226005552,875432949,795592105
    },

    {
      240208052,897491798,957022978,171139102,93849129,244950800,
      344159584,111102217,678504858,574516126,250455084,391388795,
      133277104,489333922,140361892,410735402,619253688,1031862789,
      335691611,322432037,806111734,694909702,349940897,602532064,
      148117500,111729200,457404274,557714256,1040149279,403498667
    },

    {
      432640305,1037156381,314925778,372080644,940221914,572058354,
      1038338338,426569089,455769817,491120620,1057021567,302721707,
      246475132,705152316,499194294,113265561,426137972,136848280,
      562415570,787892679,852991603,351729970,96778882,304416578,
      86282104,1036217109,157944466,954042255,414571869,870959303
    },

    {
      971287756,89414420,204713973,378635447,401797896,163681059,
      751966264,537449399,801535112,1071633688,146052774,3986810,
      636727079,697417592,1006763522,383199535,663223922,144535570,
      201016558,746200892,854342532,328874258,155896922,499850531,
      936254650,7138453,1071466390,61395452,636873816,304460150
    },

    {
      328557410,777876265,101075537,435479976,525041988,303555745,
      350113747,339410275,684034311,889925340,401693876,1060788096,
      113705201,1015366269,1065286982,10076601,952361191,167950482,
      435129959,233185538,781449455,798849169,428671146,124812670,
      188788880,15299783,859915815,1015278018,326831435,1018648256
    },

    {
      754655229,72701120,953293325,631599840,89857942,554011778,
      47073811,835760886,265073629,127620500,308658367,867391607,
      203498672,894076239,524964741,312496838,313578784,269715110,
      705844852,823949536,1027242087,59395526,569377093,394836031,
      343701223,384460815,943904084,699855722,421835371,270791893
    },

    {
      745907108,944952526,82284004,699536358,361313725,905488055,
      1023155197,882057901,746645466,119817566,452638563,1550867,
      1031436690,776155790,394503730,685193551,228707474,911100838,
      964828725,508995720,1026359451,428405268,817946254,828929109,
      313584437,871581416,426427170,160301328,365965375,220499879
    },

    {
      1056914698,521647651,221143510,117107134,150943194,408928603,
      480271744,848194951,172435885,473080053,275662818,324152353,
      59283147,131089463,834982831,778766731,122554353,88292162,
      763906051,129869508,637389308,823280981,671656942,66520354,
      308342240,325794549,6816340,371091997,621547052,25705907
    },

    {
      352064385,40386260,880715911,677695864,1005975671,853795757,
      574407480,530348942,601261381,103248825,975989152,739827654,
      863177279,460530535,159413890,688421859,1044972575,795098007,
      801093227,1001062741,926378401,139434982,834760968,204950615,
      639313518,920958373,495164640,91682281,69551285,433426629
    },

    {
      830376956,810354778,986692688,439214662,955852394,611884049,
      928650736,168559105,1064592932,151376416,433454075,1011126901,
      608575983,128899860,905698379,606430135,179348441,821647220,
      998454680,127121984,795384066,600759059,199794554,295176814,
      939420101,914617710,1018471342,111811433,331709918,221179210
    },

    {
      425733878,658229029,912424615,386431078,862251738,293702507,
      1043902477,877189394,48926307,261568902,462823467,217704510,
      305034830,297130225,26429996,343060209,894475863,17832215,
      1065385292,231895796,449755847,855378457,567620577,152049881,
      814862897,582773843,514674944,282536321,434990576,459545057
    },

    {
      665803357,392676405,1034814418,748287515,958635983,4816387,
      1034522664,915407014,57358263,113520872,253432907,18329885,
      1870034,67039950,583108703,159185799,130709561,976056560,
      448806026,846991206,686503884,308824271,436305887,194443459,
      777220336,1039366896,793594637,928986568,632103016,946360556
    },

    {
      626519650,572986777,596911660,888597461,756498052,392128719,
      36970690,1052917571,313118633,522168466,842712620,894477415,
      1055471970,11136187,252204038,40861408,975737116,212304344,
      965813895,726672608,429270148,457534820,770940172,116146229,
      243261641,1010332968,788194608,944433533,826152594,747969338
    },

    {
      1071582690,774188573,739663870,52487898,1048372764,921959408,
      547457522,429672649,490994041,25062518,805601797,589439943,
      496185778,639115556,433359194,327519661,726063918,256775538,
      973277956,33829588,639364507,351093493,73945571,631973049,
      882014952,196800376,470141183,501641249,527537127,1041746568
    },

    {
      635515840,789421408,903280967,937233682,987061368,245651243,
      162035928,13403621,749035327,131961461,612384091,371767111,
      944735537,346467471,292218132,1012693329,199650456,41027984,
      707150170,430658317,13393404,392035786,595614074,557315511,
      666890400,868924491,243135067,323841161,1027036232,622805502
    },

    {
      449543741,1068764199,919033015,422588592,252659057,380512547,
      538163030,67298058,638240573,691740025,524238634,479950745,
      602605950,1001285057,908196893,633023435,307774415,725379120,
      956739667,806388387,886237605,205177588,348918954,334053401,
      738869676,213168145,1010252086,29197916,769372946,291583312
    },

    {
      741738850,1073019312,151484689,963933996,549606967,851354389,
      277323389,157310630,427601638,817151586,491424124,737041173,
      736095612,646603916,295404110,614293779,110322022,227217880,
      596496751,455560392,177091156,799938318,212552277,239995158,
      942974955,977308770,319620009,456207171,829064434,160328806
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_17[k][l];
  j++;

  // DIMENSION dim=18,  generator matrices column encodings:

  int C_18[18][30]= { {

      129971866,403034142,42953265,456052922,529723536,39622889,
      1022405131,495125695,205765660,236811057,61590560,562939695,
      1009383060,28432675,166824630,956664215,988888538,657242377,
      1023579691,437431770,927205331,608312395,1072853475,655700129,
      27075613,636512015,837235128,226005552,875432949,795592105
    },

    {
      240208052,897491798,957022978,171139102,93849129,244950800,
      344159584,111102217,678504858,574516126,250455084,391388795,
      133277104,489333922,140361892,410735402,619253688,1031862789,
      335691611,322432037,806111734,694909702,349940897,602532064,
      148117500,111729200,457404274,557714256,1040149279,403498667
    },

    {
      432640305,1037156381,314925778,372080644,940221914,572058354,
      1038338338,426569089,455769817,491120620,1057021567,302721707,
      246475132,705152316,499194294,113265561,426137972,136848280,
      562415570,787892679,852991603,351729970,96778882,304416578,
      86282104,1036217109,157944466,954042255,414571869,870959303
    },

    {
      971287756,89414420,204713973,378635447,401797896,163681059,
      751966264,537449399,801535112,1071633688,146052774,3986810,
      636727079,697417592,1006763522,383199535,663223922,144535570,
      201016558,746200892,854342532,328874258,155896922,499850531,
      936254650,7138453,1071466390,61395452,636873816,304460150
    },

    {
      328557410,777876265,101075537,435479976,525041988,303555745,
      350113747,339410275,684034311,889925340,401693876,1060788096,
      113705201,1015366269,1065286982,10076601,952361191,167950482,
      435129959,233185538,781449455,798849169,428671146,124812670,
      188788880,15299783,859915815,1015278018,326831435,1018648256
    },

    {
      754655229,72701120,953293325,631599840,89857942,554011778,
      47073811,835760886,265073629,127620500,308658367,867391607,
      203498672,894076239,524964741,312496838,313578784,269715110,
      705844852,823949536,1027242087,59395526,569377093,394836031,
      343701223,384460815,943904084,699855722,421835371,270791893
    },

    {
      745907108,944952526,82284004,699536358,361313725,905488055,
      1023155197,882057901,746645466,119817566,452638563,1550867,
      1031436690,776155790,394503730,685193551,228707474,911100838,
      964828725,508995720,1026359451,428405268,817946254,828929109,
      313584437,871581416,426427170,160301328,365965375,220499879
    },

    {
      1056914698,521647651,221143510,117107134,150943194,408928603,
      480271744,848194951,172435885,473080053,275662818,324152353,
      59283147,131089463,834982831,778766731,122554353,88292162,
      763906051,129869508,637389308,823280981,671656942,66520354,
      308342240,325794549,6816340,371091997,621547052,25705907
    },

    {
      352064385,40386260,880715911,677695864,1005975671,853795757,
      574407480,530348942,601261381,103248825,975989152,739827654,
      863177279,460530535,159413890,688421859,1044972575,795098007,
      801093227,1001062741,926378401,139434982,834760968,204950615,
      639313518,920958373,495164640,91682281,69551285,433426629
    },

    {
      830376956,810354778,986692688,439214662,955852394,611884049,
      928650736,168559105,1064592932,151376416,433454075,1011126901,
      608575983,128899860,905698379,606430135,179348441,821647220,
      998454680,127121984,795384066,600759059,199794554,295176814,
      939420101,914617710,1018471342,111811433,331709918,221179210
    },

    {
      425733878,658229029,912424615,386431078,862251738,293702507,
      1043902477,877189394,48926307,261568902,462823467,217704510,
      305034830,297130225,26429996,343060209,894475863,17832215,
      1065385292,231895796,449755847,855378457,567620577,152049881,
      814862897,582773843,514674944,282536321,434990576,459545057
    },

    {
      665803357,392676405,1034814418,748287515,958635983,4816387,
      1034522664,915407014,57358263,113520872,253432907,18329885,
      1870034,67039950,583108703,159185799,130709561,976056560,
      448806026,846991206,686503884,308824271,436305887,194443459,
      777220336,1039366896,793594637,928986568,632103016,946360556
    },

    {
      626519650,572986777,596911660,888597461,756498052,392128719,
      36970690,1052917571,313118633,522168466,842712620,894477415,
      1055471970,11136187,252204038,40861408,975737116,212304344,
      965813895,726672608,429270148,457534820,770940172,116146229,
      243261641,1010332968,788194608,944433533,826152594,747969338
    },

    {
      1071582690,774188573,739663870,52487898,1048372764,921959408,
      547457522,429672649,490994041,25062518,805601797,589439943,
      496185778,639115556,433359194,327519661,726063918,256775538,
      973277956,33829588,639364507,351093493,73945571,631973049,
      882014952,196800376,470141183,501641249,527537127,1041746568
    },

    {
      635515840,789421408,903280967,937233682,987061368,245651243,
      162035928,13403621,749035327,131961461,612384091,371767111,
      944735537,346467471,292218132,1012693329,199650456,41027984,
      707150170,430658317,13393404,392035786,595614074,557315511,
      666890400,868924491,243135067,323841161,1027036232,622805502
    },

    {
      449543741,1068764199,919033015,422588592,252659057,380512547,
      538163030,67298058,638240573,691740025,524238634,479950745,
      602605950,1001285057,908196893,633023435,307774415,725379120,
      956739667,806388387,886237605,205177588,348918954,334053401,
      738869676,213168145,1010252086,29197916,769372946,291583312
    },

    {
      741738850,1073019312,151484689,963933996,549606967,851354389,
      277323389,157310630,427601638,817151586,491424124,737041173,
      736095612,646603916,295404110,614293779,110322022,227217880,
      596496751,455560392,177091156,799938318,212552277,239995158,
      942974955,977308770,319620009,456207171,829064434,160328806
    },

    {
      751747817,315669162,1072571913,627839530,710228931,164629564,
      195148759,354677796,438058535,197451362,893770061,655242823,
      933207855,1034931553,1008083539,186134820,228361620,632147422,
      1051983424,579762179,835229716,490118527,935155568,444896042,
      190533305,524870331,248465921,840689687,498889988,302936823
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_18[k][l];
  j++;

  // DIMENSION dim=19,  generator matrices column encodings:

  int C_19[19][30]= { {

      129971866,403034142,42953265,456052922,529723536,39622889,
      1022405131,495125695,205765660,236811057,61590560,562939695,
      1009383060,28432675,166824630,956664215,988888538,657242377,
      1023579691,437431770,927205331,608312395,1072853475,655700129,
      27075613,636512015,837235128,226005552,875432949,795592105
    },

    {
      240208052,897491798,957022978,171139102,93849129,244950800,
      344159584,111102217,678504858,574516126,250455084,391388795,
      133277104,489333922,140361892,410735402,619253688,1031862789,
      335691611,322432037,806111734,694909702,349940897,602532064,
      148117500,111729200,457404274,557714256,1040149279,403498667
    },

    {
      432640305,1037156381,314925778,372080644,940221914,572058354,
      1038338338,426569089,455769817,491120620,1057021567,302721707,
      246475132,705152316,499194294,113265561,426137972,136848280,
      562415570,787892679,852991603,351729970,96778882,304416578,
      86282104,1036217109,157944466,954042255,414571869,870959303
    },

    {
      971287756,89414420,204713973,378635447,401797896,163681059,
      751966264,537449399,801535112,1071633688,146052774,3986810,
      636727079,697417592,1006763522,383199535,663223922,144535570,
      201016558,746200892,854342532,328874258,155896922,499850531,
      936254650,7138453,1071466390,61395452,636873816,304460150
    },

    {
      328557410,777876265,101075537,435479976,525041988,303555745,
      350113747,339410275,684034311,889925340,401693876,1060788096,
      113705201,1015366269,1065286982,10076601,952361191,167950482,
      435129959,233185538,781449455,798849169,428671146,124812670,
      188788880,15299783,859915815,1015278018,326831435,1018648256
    },

    {
      754655229,72701120,953293325,631599840,89857942,554011778,
      47073811,835760886,265073629,127620500,308658367,867391607,
      203498672,894076239,524964741,312496838,313578784,269715110,
      705844852,823949536,1027242087,59395526,569377093,394836031,
      343701223,384460815,943904084,699855722,421835371,270791893
    },

    {
      745907108,944952526,82284004,699536358,361313725,905488055,
      1023155197,882057901,746645466,119817566,452638563,1550867,
      1031436690,776155790,394503730,685193551,228707474,911100838,
      964828725,508995720,1026359451,428405268,817946254,828929109,
      313584437,871581416,426427170,160301328,365965375,220499879
    },

    {
      1056914698,521647651,221143510,117107134,150943194,408928603,
      480271744,848194951,172435885,473080053,275662818,324152353,
      59283147,131089463,834982831,778766731,122554353,88292162,
      763906051,129869508,637389308,823280981,671656942,66520354,
      308342240,325794549,6816340,371091997,621547052,25705907
    },

    {
      352064385,40386260,880715911,677695864,1005975671,853795757,
      574407480,530348942,601261381,103248825,975989152,739827654,
      863177279,460530535,159413890,688421859,1044972575,795098007,
      801093227,1001062741,926378401,139434982,834760968,204950615,
      639313518,920958373,495164640,91682281,69551285,433426629
    },

    {
      830376956,810354778,986692688,439214662,955852394,611884049,
      928650736,168559105,1064592932,151376416,433454075,1011126901,
      608575983,128899860,905698379,606430135,179348441,821647220,
      998454680,127121984,795384066,600759059,199794554,295176814,
      939420101,914617710,1018471342,111811433,331709918,221179210
    },

    {
      425733878,658229029,912424615,386431078,862251738,293702507,
      1043902477,877189394,48926307,261568902,462823467,217704510,
      305034830,297130225,26429996,343060209,894475863,17832215,
      1065385292,231895796,449755847,855378457,567620577,152049881,
      814862897,582773843,514674944,282536321,434990576,459545057
    },

    {
      665803357,392676405,1034814418,748287515,958635983,4816387,
      1034522664,915407014,57358263,113520872,253432907,18329885,
      1870034,67039950,583108703,159185799,130709561,976056560,
      448806026,846991206,686503884,308824271,436305887,194443459,
      777220336,1039366896,793594637,928986568,632103016,946360556
    },

    {
      626519650,572986777,596911660,888597461,756498052,392128719,
      36970690,1052917571,313118633,522168466,842712620,894477415,
      1055471970,11136187,252204038,40861408,975737116,212304344,
      965813895,726672608,429270148,457534820,770940172,116146229,
      243261641,1010332968,788194608,944433533,826152594,747969338
    },

    {
      1071582690,774188573,739663870,52487898,1048372764,921959408,
      547457522,429672649,490994041,25062518,805601797,589439943,
      496185778,639115556,433359194,327519661,726063918,256775538,
      973277956,33829588,639364507,351093493,73945571,631973049,
      882014952,196800376,470141183,501641249,527537127,1041746568
    },

    {
      635515840,789421408,903280967,937233682,987061368,245651243,
      162035928,13403621,749035327,131961461,612384091,371767111,
      944735537,346467471,292218132,1012693329,199650456,41027984,
      707150170,430658317,13393404,392035786,595614074,557315511,
      666890400,868924491,243135067,323841161,1027036232,622805502
    },

    {
      449543741,1068764199,919033015,422588592,252659057,380512547,
      538163030,67298058,638240573,691740025,524238634,479950745,
      602605950,1001285057,908196893,633023435,307774415,725379120,
      956739667,806388387,886237605,205177588,348918954,334053401,
      738869676,213168145,1010252086,29197916,769372946,291583312
    },

    {
      741738850,1073019312,151484689,963933996,549606967,851354389,
      277323389,157310630,427601638,817151586,491424124,737041173,
      736095612,646603916,295404110,614293779,110322022,227217880,
      596496751,455560392,177091156,799938318,212552277,239995158,
      942974955,977308770,319620009,456207171,829064434,160328806
    },

    {
      751747817,315669162,1072571913,627839530,710228931,164629564,
      195148759,354677796,438058535,197451362,893770061,655242823,
      933207855,1034931553,1008083539,186134820,228361620,632147422,
      1051983424,579762179,835229716,490118527,935155568,444896042,
      190533305,524870331,248465921,840689687,498889988,302936823
    },

    {
      231782923,564441136,313799900,266020321,107576271,637714276,
      276715238,1041069275,782157799,490431753,619629874,682769192,
      84396062,1057857132,297378685,291285635,788347610,651201612,
      634291722,566191297,848749961,222420643,105329531,619183894,
      916439463,89124187,439251173,359317962,570853204,769320185
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_19[k][l];
  j++;

  // DIMENSION dim=20,  generator matrices column encodings:

  int C_20[20][30]= { {

      129971866,403034142,42953265,456052922,529723536,39622889,
      1022405131,495125695,205765660,236811057,61590560,562939695,
      1009383060,28432675,166824630,956664215,988888538,657242377,
      1023579691,437431770,927205331,608312395,1072853475,655700129,
      27075613,636512015,837235128,226005552,875432949,795592105
    },

    {
      240208052,897491798,957022978,171139102,93849129,244950800,
      344159584,111102217,678504858,574516126,250455084,391388795,
      133277104,489333922,140361892,410735402,619253688,1031862789,
      335691611,322432037,806111734,694909702,349940897,602532064,
      148117500,111729200,457404274,557714256,1040149279,403498667
    },

    {
      432640305,1037156381,314925778,372080644,940221914,572058354,
      1038338338,426569089,455769817,491120620,1057021567,302721707,
      246475132,705152316,499194294,113265561,426137972,136848280,
      562415570,787892679,852991603,351729970,96778882,304416578,
      86282104,1036217109,157944466,954042255,414571869,870959303
    },

    {
      971287756,89414420,204713973,378635447,401797896,163681059,
      751966264,537449399,801535112,1071633688,146052774,3986810,
      636727079,697417592,1006763522,383199535,663223922,144535570,
      201016558,746200892,854342532,328874258,155896922,499850531,
      936254650,7138453,1071466390,61395452,636873816,304460150
    },

    {
      328557410,777876265,101075537,435479976,525041988,303555745,
      350113747,339410275,684034311,889925340,401693876,1060788096,
      113705201,1015366269,1065286982,10076601,952361191,167950482,
      435129959,233185538,781449455,798849169,428671146,124812670,
      188788880,15299783,859915815,1015278018,326831435,1018648256
    },

    {
      754655229,72701120,953293325,631599840,89857942,554011778,
      47073811,835760886,265073629,127620500,308658367,867391607,
      203498672,894076239,524964741,312496838,313578784,269715110,
      705844852,823949536,1027242087,59395526,569377093,394836031,
      343701223,384460815,943904084,699855722,421835371,270791893
    },

    {
      745907108,944952526,82284004,699536358,361313725,905488055,
      1023155197,882057901,746645466,119817566,452638563,1550867,
      1031436690,776155790,394503730,685193551,228707474,911100838,
      964828725,508995720,1026359451,428405268,817946254,828929109,
      313584437,871581416,426427170,160301328,365965375,220499879
    },

    {
      1056914698,521647651,221143510,117107134,150943194,408928603,
      480271744,848194951,172435885,473080053,275662818,324152353,
      59283147,131089463,834982831,778766731,122554353,88292162,
      763906051,129869508,637389308,823280981,671656942,66520354,
      308342240,325794549,6816340,371091997,621547052,25705907
    },

    {
      352064385,40386260,880715911,677695864,1005975671,853795757,
      574407480,530348942,601261381,103248825,975989152,739827654,
      863177279,460530535,159413890,688421859,1044972575,795098007,
      801093227,1001062741,926378401,139434982,834760968,204950615,
      639313518,920958373,495164640,91682281,69551285,433426629
    },

    {
      830376956,810354778,986692688,439214662,955852394,611884049,
      928650736,168559105,1064592932,151376416,433454075,1011126901,
      608575983,128899860,905698379,606430135,179348441,821647220,
      998454680,127121984,795384066,600759059,199794554,295176814,
      939420101,914617710,1018471342,111811433,331709918,221179210
    },

    {
      425733878,658229029,912424615,386431078,862251738,293702507,
      1043902477,877189394,48926307,261568902,462823467,217704510,
      305034830,297130225,26429996,343060209,894475863,17832215,
      1065385292,231895796,449755847,855378457,567620577,152049881,
      814862897,582773843,514674944,282536321,434990576,459545057
    },

    {
      665803357,392676405,1034814418,748287515,958635983,4816387,
      1034522664,915407014,57358263,113520872,253432907,18329885,
      1870034,67039950,583108703,159185799,130709561,976056560,
      448806026,846991206,686503884,308824271,436305887,194443459,
      777220336,1039366896,793594637,928986568,632103016,946360556
    },

    {
      626519650,572986777,596911660,888597461,756498052,392128719,
      36970690,1052917571,313118633,522168466,842712620,894477415,
      1055471970,11136187,252204038,40861408,975737116,212304344,
      965813895,726672608,429270148,457534820,770940172,116146229,
      243261641,1010332968,788194608,944433533,826152594,747969338
    },

    {
      1071582690,774188573,739663870,52487898,1048372764,921959408,
      547457522,429672649,490994041,25062518,805601797,589439943,
      496185778,639115556,433359194,327519661,726063918,256775538,
      973277956,33829588,639364507,351093493,73945571,631973049,
      882014952,196800376,470141183,501641249,527537127,1041746568
    },

    {
      635515840,789421408,903280967,937233682,987061368,245651243,
      162035928,13403621,749035327,131961461,612384091,371767111,
      944735537,346467471,292218132,1012693329,199650456,41027984,
      707150170,430658317,13393404,392035786,595614074,557315511,
      666890400,868924491,243135067,323841161,1027036232,622805502
    },

    {
      449543741,1068764199,919033015,422588592,252659057,380512547,
      538163030,67298058,638240573,691740025,524238634,479950745,
      602605950,1001285057,908196893,633023435,307774415,725379120,
      956739667,806388387,886237605,205177588,348918954,334053401,
      738869676,213168145,1010252086,29197916,769372946,291583312
    },

    {
      741738850,1073019312,151484689,963933996,549606967,851354389,
      277323389,157310630,427601638,817151586,491424124,737041173,
      736095612,646603916,295404110,614293779,110322022,227217880,
      596496751,455560392,177091156,799938318,212552277,239995158,
      942974955,977308770,319620009,456207171,829064434,160328806
    },

    {
      751747817,315669162,1072571913,627839530,710228931,164629564,
      195148759,354677796,438058535,197451362,893770061,655242823,
      933207855,1034931553,1008083539,186134820,228361620,632147422,
      1051983424,579762179,835229716,490118527,935155568,444896042,
      190533305,524870331,248465921,840689687,498889988,302936823
    },

    {
      231782923,564441136,313799900,266020321,107576271,637714276,
      276715238,1041069275,782157799,490431753,619629874,682769192,
      84396062,1057857132,297378685,291285635,788347610,651201612,
      634291722,566191297,848749961,222420643,105329531,619183894,
      916439463,89124187,439251173,359317962,570853204,769320185
    },

    {
      676629042,890605142,1002839327,523688601,94150929,345382736,
      917470425,448046156,11104351,421774295,1003535548,690755745,
      1072393822,110502200,1046710617,863643895,501940838,1005596262,
      177889070,894474600,1024283517,536108296,511361182,597326621,
      33078708,897620545,435132721,213718759,160790581,74282284
    }

  };

  // copy the values into genMats[j]
  for(int k=0;k<j;k++)
    for(int l=0;l<30;l++) genMats[j-4][k][l]=C_20[k][l];
  j++;

  // end genMats

  // array of current nx-integers,
  // shift d->d-4 in generator matrix index
  for(int j=0;j<dim;j++)x_int[j]=genMats[d-4][j][0];

}                                                 // end constructor


/*******************************************************************************
 *
 *                     SOBOL SEQUENCE
 *
 ******************************************************************************/

void
Sobol::
restart() {
  index=1;
  // return the integer vector to the initial state
  for(int k=0;k<dim;k++)x_int[k]=v[k][0];
}


const RealArray1D&
Sobol::
nextPoint() {
  // find the rightmost zero bit of index
  int j=0, n=index;
  while(n%2==1){ n=n>>1; j++; }

  for(int k=0;k<dim;k++) {
    x_int[k]^=v[k][j];
    x[k]=((Real)x_int[k])/N;
  }

  index++;
  return x;
}


// CONSTRUCTOR

void
Sobol::
read_prim_pol(int k, int n, int d) {
  int j=0; p[k][d]=1;
  while(n>0){ j++; p[k][d-j]=n%2; n=n/2; }
  p[k][0]=1;
}


void
Sobol::
printInitialization() {
  int n=min(dim-1,30);
  cout << endl << "Sobol sequence, v[j][k]: ";
  for(int j=0;j<n;j++) {

    cout << endl << endl;
    for(int k=0;k<6;k++)
      cout << endl << "v["<<j<<"]["<<k<<"]="<<v[j][k];
  }
}                                                 // end printInitialization


// RL CHANGE: move definition of N to cc file to prevent a linker error with gcc 4 on Tiger

const Real Sobol::N = 4294967296.0;               // 2^32 to big for int

Sobol::
Sobol(int dim) :
LowDiscrepancySequence(dim),
v(dim,bits),
p(dim,13),                                        // 13 > max primpol degree
g(dim),
x_int(dim),
index(1) {
  if(dim>3500) {

    std::cout << endl
      << "Sobol sequence: dimension at most 3500. Aborting.";
    exit(0);
  }

  // 360 primitive polynomials,
  // first coordinate is the encoding,
  // second coordinate is the degree
  int pp[][2]= { {
                                                  // read_primpol handles this correctly
      0,0
    },
    {0,1},
    {1,2},
    {1,3},
    {2,3},
    {1,4},
    {4,4},
    {2,5},
    {4,5},
    {7,5},
    {11,5},
    {13,5},
    {14,5},
    {1,6},
    {13,6},
    {16,6},
    {19,6},
    {22,6},
    {25,6},
    {1,7},
    {4,7},
    {7,7},
    {8,7},
    {14,7},
    {19,7},
    {21,7},
    {28,7},
    {31,7},
    {32,7},
    {37,7},
    {41,7},
    {42,7},
    {50,7},
    {55,7},
    {56,7},
    {59,7},
    {62,7},
    {14,8},
    {21,8},
    {22,8},
    {38,8},
    {47,8},
    {49,8},
    {50,8},
    {52,8},
    {56,8},
    {67,8},
    {70,8},
    {84,8},
    {97,8},
    {103,8},
    {115,8},
    {122,8},
    {8,9},
    {13,9},
    {16,9},
    {22,9},
    {25,9},
    {44,9},
    {47,9},
    {52,9},
    {55,9},
    {59,9},
    {62,9},
    {67,9},
    {74,9},
    {81,9},
    {82,9},
    {87,9},
    {91,9},
    {94,9},
    {103,9},
    {104,9},
    {109,9},
    {122,9},
    {124,9},
    {137,9},
    {138,9},
    {143,9},
    {145,9},
    {152,9},
    {157,9},
    {167,9},
    {173,9},
    {176,9},
    {181,9},
    {182,9},
    {185,9},
    {191,9},
    {194,9},
    {199,9},
    {218,9},
    {220,9},
    {227,9},
    {229,9},
    {230,9},
    {234,9},
    {236,9},
    {241,9},
    {244,9},
    {253,9},
    {4,10},
    {13,10},
    {19,10},
    {22,10},
    {50,10},
    {55,10},
    {64,10},
    {69,10},
    {98,10},
    {107,10},
    {115,10},
    {121,10},
    {127,10},
    {134,10},
    {140,10},
    {145,10},
    {152,10},
    {158,10},
    {161,10},
    {171,10},
    {181,10},
    {194,10},
    {199,10},
    {203,10},
    {208,10},
    {227,10},
    {242,10},
    {251,10},
    {253,10},
    {265,10},
    {266,10},
    {274,10},
    {283,10},
    {289,10},
    {295,10},
    {301,10},
    {316,10},
    {319,10},
    {324,10},
    {346,10},
    {352,10},
    {361,10},
    {367,10},
    {382,10},
    {395,10},
    {398,10},
    {400,10},
    {412,10},
    {419,10},
    {422,10},
    {426,10},
    {428,10},
    {433,10},
    {446,10},
    {454,10},
    {457,10},
    {472,10},
    {493,10},
    {505,10},
    {508,10},
    {2,11},
    {11,11},
    {21,11},
    {22,11},
    {35,11},
    {49,11},
    {50,11},
    {56,11},
    {61,11},
    {70,11},
    {74,11},
    {79,11},
    {84,11},
    {88,11},
    {103,11},
    {104,11},
    {112,11},
    {115,11},
    {117,11},
    {122,11},
    {134,11},
    {137,11},
    {146,11},
    {148,11},
    {157,11},
    {158,11},
    {162,11},
    {164,11},
    {168,11},
    {173,11},
    {185,11},
    {186,11},
    {191,11},
    {193,11},
    {199,11},
    {213,11},
    {214,11},
    {220,11},
    {227,11},
    {236,11},
    {242,11},
    {251,11},
    {256,11},
    {259,11},
    {265,11},
    {266,11},
    {276,11},
    {292,11},
    {304,11},
    {310,11},
    {316,11},
    {319,11},
    {322,11},
    {328,11},
    {334,11},
    {339,11},
    {341,11},
    {345,11},
    {346,11},
    {362,11},
    {367,11},
    {372,11},
    {375,11},
    {376,11},
    {381,11},
    {385,11},
    {388,11},
    {392,11},
    {409,11},
    {415,11},
    {416,11},
    {421,11},
    {428,11},
    {431,11},
    {434,11},
    {439,11},
    {446,11},
    {451,11},
    {453,11},
    {457,11},
    {458,11},
    {471,11},
    {475,11},
    {478,11},
    {484,11},
    {493,11},
    {494,11},
    {499,11},
    {502,11},
    {517,11},
    {518,11},
    {524,11},
    {527,11},
    {555,11},
    {560,11},
    {565,11},
    {569,11},
    {578,11},
    {580,11},
    {587,11},
    {589,11},
    {590,11},
    {601,11},
    {607,11},
    {611,11},
    {614,11},
    {617,11},
    {618,11},
    {625,11},
    {628,11},
    {635,11},
    {641,11},
    {647,11},
    {654,11},
    {659,11},
    {662,11},
    {672,11},
    {675,11},
    {682,11},
    {684,11},
    {689,11},
    {695,11},
    {696,11},
    {713,11},
    {719,11},
    {724,11},
    {733,11},
    {734,11},
    {740,11},
    {747,11},
    {749,11},
    {752,11},
    {755,11},
    {762,11},
    {770,11},
    {782,11},
    {784,11},
    {787,11},
    {789,11},
    {793,11},
    {796,11},
    {803,11},
    {805,11},
    {810,11},
    {815,11},
    {824,11},
    {829,11},
    {830,11},
    {832,11},
    {841,11},
    {847,11},
    {849,11},
    {861,11},
    {871,11},
    {878,11},
    {889,11},
    {892,11},
    {901,11},
    {908,11},
    {920,11},
    {923,11},
    {942,11},
    {949,11},
    {950,11},
    {954,11},
    {961,11},
    {968,11},
    {971,11},
    {973,11},
    {979,11},
    {982,11},
    {986,11},
    {998,11},
    {1001,11},
    {1010,11},
    {1012,11},
    {41,12},
    {52,12},
    {61,12},
    {62,12},
    {76,12},
    {104,12},
    {117,12},
    {131,12},
    {143,12},
    {145,12},
    {157,12},
    {167,12},
    {171,12},
    {176,12},
    {181,12},
    {194,12},
    {217,12},
    {236,12},
    {239,12},
    {262,12},
    {283,12},
    {286,12},
    {307,12},
    {313,12},
    {319,12}

  };                                              // end pp

  // decode polynomial coefficient arrays from pp
  for(int k=0;k<dim;k++) read_prim_pol(k,pp[k][0],pp[k][1]);

  // initialize the array of direction integers
  for(int j=0;j<bits;j++)v[0][j]=(1L<<(bits-j-1));

  if(dim>1)
    v[1][0]=(1L<<bits-1);

  if(dim>2) {
    v[2][0]=(1L<<bits-1);
    v[2][1]=(1L<<bits-2);
  }

  if(dim>3) {
    v[3][0]=(1L<<bits-1);
    v[3][1]=(3L<<bits-2);
    v[3][2]=(7L<<bits-3);
  }

  if(dim>4) {
    v[4][0]=(1L<<bits-1);
    v[4][1]=(1L<<bits-2);
    v[4][2]=(5L<<bits-3);
  }

  if(dim>5) {
    v[5][0]=(1L<<bits-1);
    v[5][1]=(3L<<bits-2);
    v[5][2]=(1L<<bits-3);
    v[5][3]=(1L<<bits-4);
  }

  if(dim>6) {
    v[6][0]=(1L<<bits-1);
    v[6][1]=(1L<<bits-2);
    v[6][2]=(3L<<bits-3);
    v[6][3]=(7L<<bits-4);
  }

  if(dim>7) {
    v[7][0]=(1L<<bits-1);
    v[7][1]=(3L<<bits-2);
    v[7][2]=(3L<<bits-3);
    v[7][3]=(9L<<bits-4);
    v[7][4]=(9L<<bits-5);
  }

  if(dim>8) {
    v[8][0]=(1L<<bits-1);
    v[8][1]=(3L<<bits-2);
    v[8][2]=(7L<<bits-3);
    v[8][3]=(7L<<bits-4);
    v[8][4]=(21L<<bits-5);
  }

  if(dim>9) {
    v[9][0]=(1L<<bits-1);
    v[9][1]=(1L<<bits-2);
    v[9][2]=(5L<<bits-3);
    v[9][3]=(11L<<bits-4);
    v[9][4]=(27L<<bits-5);
  }

  if(dim>10) {
    v[10][0]=(1L<<bits-1);
    v[10][1]=(1L<<bits-2);
    v[10][2]=(7L<<bits-3);
    v[10][3]=(3L<<bits-4);
    v[10][4]=(29L<<bits-5);
  }

  if(dim>11) {
    v[11][0]=(1L<<bits-1);
    v[11][1]=(3L<<bits-2);
    v[11][2]=(7L<<bits-3);
    v[11][3]=(13L<<bits-4);
    v[11][4]=(3L<<bits-5);
  }

  if(dim>12) {
    v[12][0]=(1L<<bits-1);
    v[12][1]=(3L<<bits-2);
    v[12][2]=(5L<<bits-3);
    v[12][3]=(1L<<bits-4);
    v[12][4]=(15L<<bits-5);
  }

  if(dim>13) {
    v[13][0]=(1L<<bits-1);
    v[13][1]=(1L<<bits-2);
    v[13][2]=(1L<<bits-3);
    v[13][3]=(9L<<bits-4);
    v[13][4]=(23L<<bits-5);
    v[13][5]=(37L<<bits-6);
  }

  if(dim>14) {
    v[14][0]=(1L<<bits-1);
    v[14][1]=(1L<<bits-2);
    v[14][2]=(3L<<bits-3);
    v[14][3]=(13L<<bits-4);
    v[14][4]=(11L<<bits-5);
    v[14][5]=(7L<<bits-6);
  }

  if(dim>15) {
    v[15][0]=(1L<<bits-1);
    v[15][1]=(3L<<bits-2);
    v[15][2]=(3L<<bits-3);
    v[15][3]=(5L<<bits-4);
    v[15][4]=(19L<<bits-5);
    v[15][5]=(33L<<bits-6);
  }

  if(dim>16) {
    v[16][0]=(1L<<bits-1);
    v[16][1]=(1L<<bits-2);
    v[16][2]=(7L<<bits-3);
    v[16][3]=(13L<<bits-4);
    v[16][4]=(25L<<bits-5);
    v[16][5]=(5L<<bits-6);
  }

  if(dim>17) {
    v[17][0]=(1L<<bits-1);
    v[17][1]=(1L<<bits-2);
    v[17][2]=(1L<<bits-3);
    v[17][3]=(13L<<bits-4);
    v[17][4]=(15L<<bits-5);
    v[17][5]=(39L<<bits-6);
  }

  if(dim>18) {
    v[18][0]=(1L<<bits-1);
    v[18][1]=(3L<<bits-2);
    v[18][2]=(5L<<bits-3);
    v[18][3]=(11L<<bits-4);
    v[18][4]=(7L<<bits-5);
    v[18][5]=(11L<<bits-6);
  }

  if(dim>19) {
    v[19][0]=(1L<<bits-1);
    v[19][1]=(3L<<bits-2);
    v[19][2]=(1L<<bits-3);
    v[19][3]=(7L<<bits-4);
    v[19][4]=(3L<<bits-5);
    v[19][5]=(23L<<bits-6);
    v[19][6]=(79L<<bits-7);
  }

  if(dim>20) {
    v[20][0]=(1L<<bits-1);
    v[20][1]=(3L<<bits-2);
    v[20][2]=(1L<<bits-3);
    v[20][3]=(15L<<bits-4);
    v[20][4]=(17L<<bits-5);
    v[20][5]=(63L<<bits-6);
    v[20][6]=(13L<<bits-7);
  }

  if(dim>21) {
    v[21][0]=(1L<<bits-1);
    v[21][1]=(3L<<bits-2);
    v[21][2]=(3L<<bits-3);
    v[21][3]=(3L<<bits-4);
    v[21][4]=(25L<<bits-5);
    v[21][5]=(17L<<bits-6);
    v[21][6]=(115L<<bits-7);
  }

  if(dim>22) {
    v[22][0]=(1L<<bits-1);
    v[22][1]=(3L<<bits-2);
    v[22][2]=(7L<<bits-3);
    v[22][3]=(9L<<bits-4);
    v[22][4]=(31L<<bits-5);
    v[22][5]=(29L<<bits-6);
    v[22][6]=(17L<<bits-7);
  }

  if(dim>23) {
    v[23][0]=(1L<<bits-1);
    v[23][1]=(1L<<bits-2);
    v[23][2]=(3L<<bits-3);
    v[23][3]=(15L<<bits-4);
    v[23][4]=(29L<<bits-5);
    v[23][5]=(15L<<bits-6);
    v[23][6]=(41L<<bits-7);
  }

  if(dim>24) {
    v[24][0]=(1L<<bits-1);
    v[24][1]=(3L<<bits-2);
    v[24][2]=(1L<<bits-3);
    v[24][3]=(9L<<bits-4);
    v[24][4]=(5L<<bits-5);
    v[24][5]=(21L<<bits-6);
    v[24][6]=(119L<<bits-7);
  }

  if(dim>25) {
    v[25][0]=(1L<<bits-1);
    v[25][1]=(1L<<bits-2);
    v[25][2]=(5L<<bits-3);
    v[25][3]=(5L<<bits-4);
    v[25][4]=(1L<<bits-5);
    v[25][5]=(27L<<bits-6);
    v[25][6]=(33L<<bits-7);
  }

  if(dim>26) {
    v[26][0]=(1L<<bits-1);
    v[26][1]=(1L<<bits-2);
    v[26][2]=(3L<<bits-3);
    v[26][3]=(1L<<bits-4);
    v[26][4]=(23L<<bits-5);
    v[26][5]=(13L<<bits-6);
    v[26][6]=(75L<<bits-7);
  }

  if(dim>27) {
    v[27][0]=(1L<<bits-1);
    v[27][1]=(1L<<bits-2);
    v[27][2]=(7L<<bits-3);
    v[27][3]=(7L<<bits-4);
    v[27][4]=(19L<<bits-5);
    v[27][5]=(25L<<bits-6);
    v[27][6]=(105L<<bits-7);
  }

  if(dim>28) {
    v[28][0]=(1L<<bits-1);
    v[28][1]=(3L<<bits-2);
    v[28][2]=(5L<<bits-3);
    v[28][3]=(5L<<bits-4);
    v[28][4]=(21L<<bits-5);
    v[28][5]=(9L<<bits-6);
    v[28][6]=(7L<<bits-7);
  }

  if(dim>29) {
    v[29][0]=(1L<<bits-1);
    v[29][1]=(1L<<bits-2);
    v[29][2]=(1L<<bits-3);
    v[29][3]=(15L<<bits-4);
    v[29][4]=(5L<<bits-5);
    v[29][5]=(49L<<bits-6);
    v[29][6]=(59L<<bits-7);
  }

  if(dim>30) {
    v[30][0]=(1L<<bits-1);
    v[30][1]=(3L<<bits-2);
    v[30][2]=(5L<<bits-3);
    v[30][3]=(15L<<bits-4);
    v[30][4]=(17L<<bits-5);
    v[30][5]=(19L<<bits-6);
    v[30][6]=(21L<<bits-7);
  }

  if(dim>31) {
    v[31][0]=(1L<<bits-1);
    v[31][1]=(1L<<bits-2);
    v[31][2]=(7L<<bits-3);
    v[31][3]=(11L<<bits-4);
    v[31][4]=(13L<<bits-5);
    v[31][5]=(29L<<bits-6);
    v[31][6]=(3L<<bits-7);
  }

  // random initialization in dimension bigger than 32

  for(int k=32;k<dim;k++) {
    int degree_k=pp[k][1];
    for(int l=0;l<degree_k;l++) {
      Real u=Random::U01();
      unsigned long f=(1L<<l+1), n=(int)(f*u);
      while(n%2==0){ u=Random::U01(); n=(int)(f*u); }

      v[k][l]=(n<<(bits-l-1));
    }
  }                                               // end direction integer initialization

  // computation of direction integer v_kl for k>=degree[k]
  for(int k=1;k<dim;k++) {

    int degree_k=pp[k][1];
    for(int l=degree_k;l<bits;l++) {
      unsigned long n=(v[k][l-degree_k]>>degree_k);
      for(int j=1;j<=degree_k;j++) if(p[k][j]!=0) n=n^v[k][l-j];
      v[k][l]=n;
    }
  }                                               // end for k

  // initialize the vector of Sobol integers and Sobol points
  index=1;
  for(int k=0;k<dim;k++) x_int[k]=v[k][0];
  for(int k=0;k<dim;k++) x[k]=((Real)x_int[k])/N;

}                                                 // end constructor


MTGL_END_NAMESPACE(Martingale)
