
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*! \file curve.cpp
    \brief yield curve building, swaption volatility matrix, caplet volatility bootstrap
*/

#include "curve.h"

//-----------------------------------------------------------------------------
Curve::Curve( std::string ccy_, XML::list<RL::MarketPoint> *mkt_ ) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::Curve" );

  init ();

  mkt = mkt_;

  setCcy (ccy_);

  setDate (Settings::instance().evaluationDate());

  generate ();
}

//-----------------------------------------------------------------------------
void Curve::init () {
//-----------------------------------------------------------------------------
  currency = "";
  asofdate = QL::Date::minDate ();
}

//-----------------------------------------------------------------------------
int Curve::setCcy (std::string ccy_) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::setCcy" );

  if (currency != ccy_) {
    log.msg (NOTICE, "currency change %s -> %s",
             currency.c_str(), ccy_.c_str());
    previousCurrency = currency;
    currency         = ccy_;
  }

  return 0;
}

//-----------------------------------------------------------------------------
int Curve::setDate (std::string asof_) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::setDate" );

  QL::Date date = parseDate( asof_.c_str() );

  if (date.weekday() == Sunday || date.weekday() == Saturday) {
    log.msg (ERROR, "only non-weekend days allowed, exit");
    exit(-1);
  }

  if (asofdate != date) {
    log.msg (NOTICE, "date change %s -> %s",
	     DateFormatter::toString( asofdate, DateFormatter::ISO ).c_str(),
	     DateFormatter::toString( date, DateFormatter::ISO ).c_str());
    asofdate = date;
  }

  return 0;
}

//-----------------------------------------------------------------------------
int Curve::setDate (QL::Date asof_) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::setDate" );

  if (asof_.weekday() == Sunday || asof_.weekday() == Saturday) {
    log.msg (ERROR, "only non-weekend days allowed, exit");
    exit(-1);
  }

  if (asofdate != asof_) {
    log.msg (NOTICE, "date change %s -> %s",
	     DateFormatter::toString( asofdate, DateFormatter::ISO ).c_str(),
	     DateFormatter::toString( asof_, DateFormatter::ISO ).c_str());
    asofdate = asof_;
  }

  return 0;
}

//-----------------------------------------------------------------------------
double Curve::disc( QL::Date date ) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::disc" );

  if (isUpToDate() == false) {
    log.msg (NOTICE, "curve is not up to date, generate...");
    generate();
  }
//  else log.msg (NOTICE, "curve is up-to-date");

  return ts->discount( date );
}

//-----------------------------------------------------------------------------
double Curve::zero( QL::Date date ) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::zero" );

  if (isUpToDate() == false) {
    log.msg (NOTICE, "curve is not up to date, generate...");
    generate();
  }
//  else log.msg (NOTICE, "curve is up-to-date");

  return ts->zeroRate (date, Actual365Fixed(), Continuous);
}

//-----------------------------------------------------------------------------
double Curve::fwdvol (QL::Date date) {
//-----------------------------------------------------------------------------
  // linear interpolation of caplet volatilities
  static Debug log( "Curve::fwdvol" );

  QL::Date aDate = QL::Date::minDate();
  QL::Date bDate = QL::Date::minDate();

  double aVol = 0, bVol = 0;

  if (date <= caplets.begin()->start) return caplets.begin()->vol;

  for (std::list<Caplet>::iterator i=caplets.begin(); i != caplets.end(); i++){
    bDate = i->start;
    bVol  = i->vol;

    if (bDate >= date) {
      return aVol + (bVol - aVol)
	* (date.serialNumber() - aDate.serialNumber())
	/ (bDate.serialNumber() - aDate.serialNumber());
    }

    aDate = bDate;
    aVol  = bVol;
  }

  return aVol;
}

//-----------------------------------------------------------------------------
double Curve::capletPrice (QL::Date start, double strike) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::capletPrice" );

  Default def( "Swap", ccy(), ts );

  Period  tenor  = PeriodParser::parse( def.floatingTenor );
  QL::Date startDate = start;
  QL::Date endDate   = startDate + tenor;

  // caplet schedule, one period
  QL::Schedule schedule = MakeSchedule (def.calendar, startDate, endDate,
                                        def.floatFreq, def.roll);

  std::vector<boost::shared_ptr<CashFlow> >
    leg = FloatingRateCouponVector (schedule, def.roll,
                                    std::vector<double>(1,1),
                                    def.index,
                                    def.fixingDays,
                                    std::vector<double>(0,1) );

  if (leg.size() != 1)
    log.msg (ALERT, "caplet leg size %d, exit", leg.size() );

  double vol = fwdvol (startDate);

  boost::shared_ptr<CapFloor> cap = makeCapFloor (CapFloor::Cap,
                                                  leg, strike, vol);

  return cap->NPV();
}

//-----------------------------------------------------------------------------
double Curve::capletPrice (CapFloor::Type type,
			   const QL::Date& startDate,
			   const QL::Date& endDate,
			   Calendar calendar,
			   double nominal,
			   double strike,
			   boost::shared_ptr<Xibor> index) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::capletPrice" );

  boost::shared_ptr<CapFloor> capfloor =
    makeCapFloorLet (type, startDate, endDate, calendar,
                     nominal, strike, index);

  return capfloor->NPV();
}

//-----------------------------------------------------------------------------
int Curve::generate () {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::generate" );

  double rate = 0.05;

  if (getenv (ENV_FLAT_CURVE)) {
    rate = atof (getenv(ENV_FLAT_CURVE));
    log.msg( NOTICE, "environment variable %s set to %s",
	     ENV_FLAT_CURVE, getenv(ENV_FLAT_CURVE) );
    log.msg( NOTICE, "flat rate set to %.6f A360", rate );
  }
  else {
    log.msg( WARNING, "environment variable %s not set", ENV_FLAT_CURVE );
    log.msg( WARNING, "using default flat rate %.6f A360", rate );
  }

  log.msg (NOTICE, "build flat curve ccy %s asof %s",
	   currency.c_str(),
           DateFormatter::toString (asofdate, DateFormatter::ISO).c_str());

  tsFlat = buildFlatCurve( currency, asofdate, rate, mapDayCounter("A360") );

  // perform basic quick tests on flat curve ...
  double z, d;
  try {
    z = tsFlat->zeroRate (asofdate + PeriodParser::parse ("1Y"),
                          Actual365Fixed(), Continuous);
    d = tsFlat->discount (asofdate + PeriodParser::parse ("1Y"));
    log.msg (NOTICE, "zero %.6f discount %.6f", z, d);
  }
  CATCH_ALERT;

  try {
    ts.linkTo (tsFlat);
    log.msg (NOTICE, "linking curve ok");
    z = ts->zeroRate (asofdate + PeriodParser::parse ("1Y"),
                      Actual365Fixed(), Continuous);
    d = ts->discount (asofdate + PeriodParser::parse ("1Y"));
    log.msg (NOTICE, "zero %.6f discount %.6f", z, d);
  }
  CATCH_ALERT;

  if ( mkt->size() == 0 ) {
    log.msg( ERROR, "cannot build curves from empty market list" );
    exit(1);
  }

  log.msg (NOTICE, "build depo/swap curve ccy %s asof %s",
	   currency.c_str(),
           DateFormatter::toString (asofdate, DateFormatter::ISO).c_str());

  tsDepoSwap = buildDepoSwapCurve (ccy(), asof());

  log.msg (NOTICE, "build swaption vola matrix ccy %s asof %s",
	   currency.c_str(),
           DateFormatter::toString (asofdate, DateFormatter::ISO).c_str());

  volMatrix  = buildSwaptionVolMatrix (ccy(), asof());

  previousCurrency = currency;

  log.msg (NOTICE, "flat curve ccy %s asof %s",
	   currency.c_str(),
	   DateFormatter::toString (tsFlat->referenceDate(),
                                    DateFormatter::ISO).c_str());

  log.msg (NOTICE, "depo/swap curve ccy %s asof %s",
	   currency.c_str(),
	   DateFormatter::toString (tsDepoSwap->referenceDate(),
                                    DateFormatter::ISO).c_str());

  // link to deposwap curve and do basic tests
  link(1);

  test (3);

  log.msg (NOTICE, "build caplets ccy %s asof %s",
	   currency.c_str(),
           DateFormatter::toString (asofdate, DateFormatter::ISO).c_str());

  buildCaplets (ccy(), asof(), asof());

  log.msg (NOTICE, "curves generated" );

  return 0;
}

//-----------------------------------------------------------------------------
void Curve::test (int N) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::test" );

  Date date;
  Default def( "Deposit", "EUR", ts );
  FILE *f = fopen ("curvetest.txt", "w");

  if (!f)
    log.msg (ALERT, "error opening file");

  for (int n = 0; n <= N; n++) {
    link(0);
    for( int i=1; i<=20; i++ ) {
      date = def.calendar.advance( asofdate, i, Years );
      fprintf (f, "%2d discount(%s,%ld) = %.4f (%.8f)\n",
               n,
               DateFormatter::toString( date, DateFormatter::ISO ).c_str(),
               date.serialNumber() - asofdate.serialNumber(),
               disc( date ),
               zero( date ) );
    }
    link(1);
    for( int i=1; i<=20; i++ ) {
      date = def.calendar.advance( asofdate, i, Years );
      fprintf (f, "%2d discount(%s,%ld) = %.4f (%.8f)\n",
               n,
               DateFormatter::toString( date, DateFormatter::ISO ).c_str(),
               date.serialNumber() - asofdate.serialNumber(),
               disc( date ),
               zero( date ) );
    }
  }

  log.msg (NOTICE, "completed");

  fclose (f);
}

//-----------------------------------------------------------------------------
bool Curve::isUpToDate () {
//-----------------------------------------------------------------------------

//   if (tsFlat->todaysDate() == asofdate &&
//       tsDepoSwap->todaysDate() == asofdate &&
//       currency == previousCurrency)
//  if (tsFlat->referenceDate() == asofdate &&
//      tsDepoSwap->referenceDate() == asofdate &&
//      currency == previousCurrency)
  if (currency == previousCurrency)
    return true;
  else
    return false;
}

//-----------------------------------------------------------------------------
boost::shared_ptr<YieldTermStructure>
Curve::buildFlatCurve (std::string ccy, QL::Date asof, double rate,
                       DayCounter dc) {
//-----------------------------------------------------------------------------
  static Debug log( "buildFlatCurve" );

  log.msg( DEBUG, "building flat curve" );

  Default def( "Deposit", ccy, ts );

  QL::Date settlementDate = def.calendar.advance (asof, def.settlementDays, Days);

  boost::shared_ptr<YieldTermStructure>
    myTermStructure(new FlatForward(asof, rate, dc));

  return myTermStructure;
}

//-----------------------------------------------------------------------------
bool Curve::getRate (double *rate,
		     const char *type, const char *ccy, const char *term) {
//-----------------------------------------------------------------------------
  std::list<RL::MarketPoint>::iterator i;

  for( i = mkt->begin(); i != mkt->end(); i++ ) {
    if ( strcmp(i->type.c_str(), type)==0 &&
	 strcmp(i->ccy.c_str(),  ccy )==0 &&
	 strcmp(i->term.c_str(), term)==0 ) {
      *rate = i->rate;
      return true;
    }
  }

  return false;
}

//-----------------------------------------------------------------------------
bool Curve::getVol (double *rate, const char *type, const char *ccy,
		    const char *term, const char *expiry) {
//-----------------------------------------------------------------------------
  std::list<RL::MarketPoint>::iterator i;

  for( i = mkt->begin(); i != mkt->end(); i++ ) {
    if ( strcmp(i->type.c_str(),   type)==0 &&
	 strcmp(i->ccy.c_str(),    ccy )==0 &&
	 strcmp(i->term.c_str(),   term)==0 &&
	 strcmp(i->expiry.c_str(), expiry)==0 ) {
      *rate = i->rate;
      return true;
    }
  }

  return false;
}

//-----------------------------------------------------------------------------
bool Curve::getCapVol (double *rate, double *strike,
		       const char *ccy, const char *term, const char *expiry) {
//-----------------------------------------------------------------------------
  std::list<RL::MarketPoint>::iterator i;

  //  printf( "find %s %s %s\n", ccy, term, expiry );

  for( i = mkt->begin(); i != mkt->end(); i++ ) {
    if ( strcmp(i->type.c_str(),   "CAP")==0 &&
	 strcmp(i->ccy.c_str(),    ccy )==0 &&
	 strcmp(i->term.c_str(),   term)==0 &&
	 strcmp(i->expiry.c_str(), expiry)==0 ) {
      *rate   = i->rate;
      *strike = i->strike;
      return true;
    }
  }

  return false;
}

//-----------------------------------------------------------------------------
boost::shared_ptr<YieldTermStructure>
Curve::buildDepoSwapCurve (std::string ccy, QL::Date asof) {
//-----------------------------------------------------------------------------
  static Debug log( "buildCurve" );

  double  rate;
  char    mktDate[11], gdate[3], product[20];
  Period  p;
  std::vector<boost::shared_ptr<RateHelper> > instruments;

  log.msg( DEBUG, "building deposit/swap curve" );

  // convert asof date to ISO string format for xml file look up
  //  cout << DateFormatter::toString( asof, DateFormatter::ISO ) << endl;
  sprintf( mktDate, DateFormatter::toString( asof,
                                             DateFormatter::ISO ).c_str() );

  log.msg( NOTICE, "asof is %s", DateFormatter::toString(asof).c_str() );

  // build deposit rate section ...............................................

  // get deposit defaults
  Default depoDef( "Deposit", ccy, ts );

  log.msg (NOTICE, "number of instruments = %d", instruments.size() );

  sprintf( product, "DEPOSIT" );
  // week rates
  for( int i=1; i<=3; i++ ) {
    sprintf( gdate, "%dW", i );
    p = PeriodParser::parse (gdate);
    if (getRate (&rate, product, ccy.c_str(), gdate) == true) {
      boost::shared_ptr<Quote> depositRate (new SimpleQuote (rate * 0.01));
      boost::shared_ptr<RateHelper> depositHelper (new DepositRateHelper(
             Handle<Quote>(depositRate),
             p.length(), p.units(),
             depoDef.settlementDays, // default
             depoDef.calendar,       // default
             depoDef.roll,           // default
             depoDef.mmDayCount));   // default
      instruments.push_back(depositHelper);
      log.msg (DEBUG,  "%d deposit %s = %.3f", instruments.size(), gdate,rate);
    }
  }
  // month rates
  for( int i=1; i<=12; i++ ) {
    sprintf( gdate, "%dM", i );
    p = PeriodParser::parse (gdate);
    if ( getRate (&rate, product, ccy.c_str(), gdate) == true ) {
      boost::shared_ptr<Quote> depositRate (new SimpleQuote (rate*0.01));
      boost::shared_ptr<RateHelper> depositHelper( new DepositRateHelper(
	     Handle<Quote>(depositRate), p.length(), p.units(),
             depoDef.settlementDays, // default
             depoDef.calendar,       // default
             depoDef.roll,           // default
             depoDef.mmDayCount));     // default
      instruments.push_back(depositHelper);
      log.msg (DEBUG, "%d deposit %s = %.3f", instruments.size(), gdate, rate);
    }
  }

  log.msg (NOTICE, "number of depos = %d", instruments.size() );

  // build swap rate section ..................................................

  // get swap defaults
  Default swapDef( "Swap", ccy, ts );

  sprintf( product, "SWAP" );
  for( int i=2; i<=50; i++ ) {
    sprintf( gdate, "%dY", i );
    p = PeriodParser::parse (gdate);
    if (getRate (&rate, product, ccy.c_str(), gdate) == true) {
      boost::shared_ptr<Quote> swapRate (new SimpleQuote (rate*0.01));
      boost::shared_ptr<RateHelper> swapHelper (new SwapRateHelper(
             Handle<Quote>(swapRate), p.length(), p.units(),
             swapDef.settlementDays,    // default
             swapDef.calendar,          // default
             swapDef.fixedFreq,         // default
             swapDef.fixedRoll,         // default
             swapDef.cmDayCount,        // default
             swapDef.floatFreq,
             ModifiedFollowing));       // FIXME
      instruments.push_back(swapHelper);
      log.msg (DEBUG, "%d swap %s = %.3f", instruments.size(), gdate, rate );
    }
  }

  log.msg (NOTICE, "number of depos/swaps = %d", instruments.size() );

  QL::Date settlementDate = depoDef.calendar.advance (asof,
                                                      depoDef.settlementDays,
                                                      Days );

  if (instruments.size() < 2) {
    log.msg(ALERT, "number of instruments %d too small", instruments.size() );
  }

  double tolerance = 1e-15;
  boost::shared_ptr<YieldTermStructure> myTermStructure(new
			 PiecewiseFlatForward(settlementDate, instruments,
					      depoDef.mmDayCount, tolerance));

  return myTermStructure;
}

//-----------------------------------------------------------------------------
SwaptionVolatilityMatrix *
Curve::buildSwaptionVolMatrix (std::string ccy, QL::Date asof) {
//-----------------------------------------------------------------------------
  static Debug log( "buildSwaptionVolMatrix" );

  // these are the exercises and terms where I would very optimistically
  // expect to find market data
  // the market file is scanned for these market points

  log.msg( DEBUG, "building swaption volatility matrix" );

  char * exercises[]={"1M",  "2M",  "3M",  "6M",  "9M",
		      "1Y",  "2Y",  "3Y",  "4Y",  "5Y",  "6Y",  "7Y",  "8Y",
                      "9Y", "10Y", "12Y", "15Y", "20Y", "25Y", "30Y" };

  char * lengths[] ={ "1Y",  "2Y",  "3Y",  "4Y",  "5Y",  "6Y",  "7Y",  "8Y",
                      "9Y", "10Y", "12Y", "15Y", "20Y", "25Y", "30Y", "35Y",
                      "40Y", "45Y", "50Y" };

  // convert asof date to ISO string format for xml file look up
  char mktDate[11];
  sprintf( mktDate,
           DateFormatter::toString( asof, DateFormatter::ISO ).c_str() );

  // let us check the borders by scanning all swaptions vol's in the dom tree
  // individually and identifying the borders here.
  // it works, but this should be done in a more clever way using the xalan api - FIXME

  QL::Date minExercise = Date::maxDate();
  QL::Date maxExercise = Date::minDate();
  QL::Date tmpExercise;
  QL::Date minLength = Date::maxDate();
  QL::Date maxLength = Date::minDate();
  QL::Date tmpLength;
  int i, j;
  int iMin=-1, iMax=-1, jMin=-1, jMax=-1;
  double rate;

  for( i=0; i<(int)LENGTH(exercises); i++ ) { // rows
    for( j=0; j<(int)LENGTH(lengths); j++ ) { // columns
      if ( getVol (&rate, "SWAPTION", ccy.c_str(), lengths[j], exercises[i])
           == true ) {
	tmpExercise = asof + PeriodParser::parse(exercises[i]);
	tmpLength   = asof + PeriodParser::parse(lengths[j]);
	if (tmpExercise < minExercise) { minExercise = tmpExercise; iMin = i; }
	if (tmpExercise > maxExercise) { maxExercise = tmpExercise; iMax = i; }
	if (tmpLength < minLength) { minLength = tmpLength; jMin = j; }
	if (tmpLength > maxLength) { maxLength = tmpLength; jMax = j; }
      }
    }
  }

  if ( iMin < 0 || iMax < 0 || jMin < 0 || jMax < 0 ) {
    log.msg( ERROR, "no swaption vols found, exit" );
    exit(-1);
  }

  log.msg( DEBUG, "Exercise range %s - %s, dimension %d",
	 exercises[iMin], exercises[iMax], iMax-iMin+1 );
  log.msg( DEBUG, "Length range %s - %s, dimension %d",
	 lengths[jMin], lengths[jMax], jMax-jMin+1 );


  std::vector<QL::Date> exerciseDates;
  std::vector<Period>   periods;
  std::vector<int>      exerciseIndices, periodIndices;

  // set the exercise date and length vectors
  for( i = iMin; i <= iMax; i++ ) { // rows
    int count=0;
    for( j = jMin; j <= jMax; j++ ) { // columns
      if ( getVol (&rate, "SWAPTION", ccy.c_str(), lengths[j], exercises[i])
           == true ) {
	count++;
      }
    }
    if (count>0) {
      exerciseIndices.push_back (i);
      exerciseDates.push_back (asof + PeriodParser::parse (exercises[i]));
    }
  }

  for( j = jMin; j <= jMax; j++ ) { // columns
    int count=0;
    for( i = iMin; i <= iMax; i++ ) { // rows
      if ( getVol (&rate, "SWAPTION", ccy.c_str(), lengths[j], exercises[i])
           == true ) {
	count++;
      }
    }
    if (count>0) {
      periodIndices.push_back( j );
      periods.push_back( PeriodParser::parse( lengths[j] ) );
    }
  }

  log.msg( DEBUG, "Exercise dates size %d",  exerciseDates.size() );
  log.msg( DEBUG, "Length size %d", periods.size() );

  if ( exerciseDates.size() == 0 ||
       periods.size() == 0 ) {
    log.msg( "error reading exercises or periods, exit" );
    exit(-1);
  }

  // seem to need allocating a quadratic matrix ? bug ? FIXME
  int dim;
  if ( exerciseDates.size() > periods.size() )
    dim = exerciseDates.size();
  else
    dim = periods.size();

  // exercises in rows (1st arg)
  // Math::Matrix volatilities(iMax-iMin+2,jMax-jMin+2,0);
  QL::Matrix volatilities(dim,dim,0);

  log.msg( DEBUG, "Matrix dimension (rows x cols) = %d x %d",
	 volatilities.rows(), volatilities.columns() );

  // get swap defaults for the given ccy
  Default swapDef( "Swap", ccy, ts );

  // scan the dom tree once again
  for( unsigned int iIndex = 0; iIndex < exerciseIndices.size(); iIndex++ ) {
    i = exerciseIndices[iIndex];
    for( unsigned int jIndex = 0; jIndex < periodIndices.size(); jIndex++ ) {
      j = periodIndices[jIndex];

      if ( getVol (&rate, "SWAPTION", ccy.c_str(), lengths[j], exercises[i])
           == true ) {
	volatilities[jIndex][iIndex] = rate;
	log.msg( DEBUG, "added vol (expiry x term = %s x %s) = %.2f",
	       exercises[i], lengths[j], rate );
      }
      else {
	volatilities[jIndex][iIndex] = 0.0; // undefined ...
	log.msg( WARNING,
                 "missing vol, setting (expiry x term = %s x %s) to zero" );
	log.msg( WARNING, "this will spoil the bilinear interpolation" );
      }
    }
  }

  SwaptionVolatilityMatrix *
    matrix = new SwaptionVolatilityMatrix (asof, exerciseDates, periods,
                                           volatilities,
                                           swapDef.cmDayCount);

  log.msg( DEBUG, "new swaption matrix created" );

  return matrix;
}

//-----------------------------------------------------------------------------
std::vector<boost::shared_ptr<CashFlow> >
Curve::makeLeg(const QL::Date& startDate, int length, Default& def) {
//-----------------------------------------------------------------------------
  QL::Date endDate = def.calendar.advance (startDate, length, Years, def.roll);

  QL::Schedule schedule = MakeSchedule (def.calendar, startDate, endDate,
					def.floatFreq, def.roll); //, true);

  return FloatingRateCouponVector (schedule, def.roll,
                                   std::vector<double>(1,1),
                                   def.index, def.fixingDays,
                                   std::vector<double>(1,0));
}

//-----------------------------------------------------------------------------
std::vector<boost::shared_ptr<CashFlow> >
Curve::makeLeg(const QL::Date& startDate, const QL::Date& endDate,
               Default& def) {
//-----------------------------------------------------------------------------
  QL::Schedule schedule = MakeSchedule (def.calendar, startDate, endDate,
					def.floatFreq, def.roll); //, true);

  return FloatingRateCouponVector (schedule, def.roll,
                                   std::vector<double>(1,1),
                                   def.index, def.fixingDays,
                                   std::vector<double>(1,0));
}

//-----------------------------------------------------------------------------
boost::shared_ptr<PricingEngine>
Curve::makeEngine(double volatility) {
//-----------------------------------------------------------------------------
  Handle<Quote> vol(boost::shared_ptr<Quote>(new SimpleQuote(volatility)));
  boost::shared_ptr<BlackModel> model(new BlackModel(vol, ts));
  return boost::shared_ptr<PricingEngine>(new BlackCapFloorEngine(model));
}

//-----------------------------------------------------------------------------
boost::shared_ptr<CapFloor>
Curve::makeCapFloor (CapFloor::Type type,
                     const std::vector<boost::shared_ptr<CashFlow> >& leg,
                     Rate strike,
                     double volatility) {
//-----------------------------------------------------------------------------
  switch (type) {
  case CapFloor::Cap:
    return boost::shared_ptr<CapFloor>(
      new Cap(leg, std::vector<Rate>(1, strike),
              ts, makeEngine(volatility)));
  case CapFloor::Floor:
    return boost::shared_ptr<CapFloor>(
      new Floor(leg, std::vector<Rate>(1, strike),
                ts, makeEngine(volatility)));
  default:
    QL_FAIL("unknown cap/floor type");
  }
}

//-----------------------------------------------------------------------------
boost::shared_ptr<CapFloor>
Curve::makeCapFloorLet (CapFloor::Type type,
                        const QL::Date& startDate,
                        const QL::Date& endDate,
                        Calendar calendar,
                        double nominal,
                        double strike,
                        boost::shared_ptr<Xibor> index) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::makeCapFloorLet" );

  Default def( "Swap", ccy(), ts );

  QL::Schedule schedule = MakeSchedule (calendar, startDate, endDate,
					def.floatFreq, def.roll); //, true);

  std::vector<boost::shared_ptr<CashFlow> >
    leg = FloatingRateCouponVector (schedule, def.roll,
                                    std::vector<double>(1,nominal),
                                    index, def.fixingDays,
                                    std::vector<double>(1,0));

  if (leg.size() != 1) {
    log.msg (ERROR, "caplet leg size %d, exit", leg.size() );
    exit(-1);
  }

  double vol = fwdvol (startDate);

  boost::shared_ptr<CapFloor> capfloor = makeCapFloor (type, leg, strike, vol);

  return capfloor;
}

//-----------------------------------------------------------------------------
double
Curve::flatVol (QL::Date date, int mode) {
//-----------------------------------------------------------------------------
  if (date <= caps.begin()->maturity) return caps.begin()->vol;

  std::vector<double> t, v;
  QL::Date cut = Date::minDate();
  std::list<FlatCapFloor>::iterator j=caps.begin();

  for (std::list<FlatCapFloor>::iterator i=caps.begin(); i != caps.end(); i++){
    t.push_back ((double)i->maturity.serialNumber());
    v.push_back (i->vol);
    // identify the date at the maximum of the volatility "curve"
    if (i->vol < j->vol && cut == Date::minDate()) cut = j->maturity;
    j = i;
  }

  // 1: linear
  // 2: cubic spline
  int order;

  if (date > cut) order = 2; // use cubic spline interpolation beyond "hump"
  else            order = 1; // linear interpolation before

  return interpolate1D (t, v, (double)date.serialNumber(), order, true);
}

//-----------------------------------------------------------------------------
double
Curve::atmStrike (std::vector<boost::shared_ptr<CashFlow> > leg, double vol) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::atmStrike" );

  double strike    = 0.05;
  double step      = 0.01;
  int    maxCount  = 200;
  double minStep   = 1e-8;
  double precision = 1e-5;
  double c, f;
  int    i = 0;
  int    sign=1, old_sign=1;

  do {
    boost::shared_ptr<CapFloor>
      cap = makeCapFloor (CapFloor::Cap, leg, strike, vol);
    boost::shared_ptr<CapFloor>
      flo = makeCapFloor (CapFloor::Floor, leg, strike, vol);
    c = cap->NPV();
    f = flo->NPV();
    if (c > f) sign =  1;
    else       sign = -1;
    if (sign * old_sign == -1) step /= 2;
    strike += sign * step;
    old_sign = sign;
    //printf( "%d step %.6f cap/floor %.4f/%.4f\n", i, step, c, f);
    i++;
  }
  while (fabs(c-f) > precision && i < maxCount && step > minStep);

  if (i == maxCount) {
    log.msg (ERROR, "%d step %.6f cap/floor %.4f/%.4f", i, step, c, f);
    log.msg (ERROR, "maximum number of steps %d reached, exit", i);
    exit(-1);
  }

  if (step <= minStep) {
    log.msg (ERROR, "%d step %.6f cap/floor %.4f/%.4f", i, step, c, f);
    log.msg (ERROR, "minimum step size %.6f reached, exit", step);
    exit(-1);
  }

  //printf( "%d step %.6f cap/floor %.4f/%.4f strike %.5f\n", i, step, c, f, strike);

  return strike;
}

//-----------------------------------------------------------------------------
int Curve::buildCaplets (std::string ccy, QL::Date asof, QL::Date start) {
//-----------------------------------------------------------------------------
  static Debug log( "Curve::buildCaplets" );

  Default def( "Swap", ccy, ts );

  QL::Date startDate = def.calendar.advance (start, def.settlementDays, Days);

  // convert asof date to ISO string format for xml file look up
  char mktDate[11];
  sprintf( mktDate, DateFormatter::toString( asof,
                                             DateFormatter::ISO ).c_str() );

  // potential cap maturities, searched in market xml file
  char * lengths[] ={ "1Y",   "2Y",  "3Y",  "4Y",  "5Y",  "6Y",  "7Y",  "8Y",
                      "9Y",  "10Y", "12Y", "15Y", "20Y", "25Y", "30Y", "35Y",
                      "40Y", "45Y", "50Y" };

  log.msg (NOTICE, "read flat vols from market");

// read flat vols from market
  double vol, strike;
  FlatCapFloor *cap;
  QL::Date finalMaturity;
  caps.clear();
  for( unsigned int j=0; j<LENGTH(lengths); j++ ) {
    if ( getCapVol (&vol,
                    &strike,
                    ccy.c_str(),
                    def.floatingTenor.c_str(),
                    lengths[j]) == true ) {
      cap = new FlatCapFloor ("Cap",
                              startDate,
                              ccy, def.floatingTenor,
                              lengths[j],
                              strike, vol);
      caps.push_back (*cap);
      finalMaturity = cap->maturity;
    }
  }

  if (caps.size() == 0) {
    log.msg( ERROR, "no flat cap vols found, exit" );
    exit(-1);
  }

  // bootstrap caplets
  Period   tenor = PeriodParser::parse (caps.begin()->tenor);
  QL::Date capStartDate = startDate + tenor;
  QL::Date endDate;
  Caplet*  caplet;
  double   lastFlatPrice = 0;
  double   lastFlatVol   = 0;
  int      i = 1;
  std::vector<boost::shared_ptr<CashFlow> > leg;

  FILE* f = fopen ("CapletBootstrap.txt", "w");

  fprintf (f, "%-2s %-10s %-10s %-8s %-8s %-8s %-8s %-8s %-8s %-8s %-8s\n",
	   "#", "From", "To", "FlatVol", "FwdVol", "Flat", "Price", "Strike",
           "Fwd", "P(strike)", "P(fwd)");

  do {
    startDate = startDate + tenor;
    endDate   = startDate + tenor;

    startDate = def.calendar.adjust (startDate);
    endDate   = def.calendar.adjust (endDate);

    caplet = new Caplet (caps.begin()->ccy, startDate, endDate);

    // caplet schedule, one period
    Schedule schedule = MakeSchedule (def.calendar, startDate, endDate,
                                      def.floatFreq, def.roll); //, true);

    caplet->leg = FloatingRateCouponVector (schedule, def.roll,
                                            std::vector<double>(1,1),
                                            def.index, def.fixingDays,
                                            std::vector<double>(1,0));

    if (caplet->leg.size() != 1)
      log.msg (ALERT, "caplet leg size %d, exit", caplet->leg.size() );

    // atm caplet strike = forward rate
    boost::shared_ptr<FloatingRateCoupon>
      coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(caplet->leg[0]);
    caplet->strike = coupon->rate();

    caplet->start   = coupon->accrualStartDate();
    caplet->end     = coupon->accrualEndDate();

    // next flat vol (0) or linear inerpolation (1)
    caplet->flatVol = flatVol (caplet->end, 1);

    ///////////////////////////////////////////////////////////////////////////
    //
    // price the atm cap with maturity = current caplet maturity
    //
    // cap schedule with maturity = caplet->end
    schedule = MakeSchedule (def.calendar, capStartDate, endDate,
			     def.floatFreq, def.roll); //, true);

    // cap leg
    leg = FloatingRateCouponVector (schedule, def.roll,
                                    std::vector<double>(1,1),
                                    def.index, def.fixingDays,
                                    std::vector<double>(1,0));

    // flat atm cap strike
    caplet->strike = atmStrike (leg, caplet->flatVol);

    boost::shared_ptr<CapFloor> cap = makeCapFloor (CapFloor::Cap,
                                                    leg,
                                                    caplet->strike,
                                                    caplet->flatVol);

    // atm cap price
    caplet->flatPrice = cap->NPV();

    // this is equal to the caplet price in period 1
    if (i == 1) {
      caplet->price = caplet->flatPrice;
      caplet->vol   = caplet->flatVol;
    }
    else {
      /////////////////////////////////////////////////////////////////////////
      // price a cap with
      // - maturity and flat vol of the previous caplet and
      // - strike of the CURRENT caplet

      // cap schedule with maturity = caplet->start
      schedule = MakeSchedule (def.calendar, capStartDate, caplet->start,
			       def.floatFreq, def.roll); //, true);
      // cap leg
      leg = FloatingRateCouponVector (schedule, def.roll,
                                      std::vector<double>(1,1),
                                      def.index, def.fixingDays,
                                      std::vector<double>(1,0));

      boost::shared_ptr<CapFloor> cap = makeCapFloor (CapFloor::Cap,
                                                      leg,
                                                      caplet->strike,
                                                      lastFlatVol);
      // previous cap price
      lastFlatPrice = cap->NPV();

      // the current caplet's price is the difference between
      // current and previous flat cap prices
      caplet->price = caplet->flatPrice - lastFlatPrice;

      if (caplet->price < 0) {
	log.msg (ERROR, "caplet price < 0, exit");
	exit(-1);
      }

      // create the current caplet instrument for imply calculation
      boost::shared_ptr<CapFloor> qlcap = makeCapFloor (CapFloor::Cap,
                                                        caplet->leg,
                                                        caplet->strike,
                                                        caplet->flatVol);

      // the current caplet's implied volatility
      try {
	caplet->vol = qlcap->impliedVolatility (caplet->price, 1e-6, 1000);
      } catch(std::exception& e) {
	std::cout << e.what() << std::endl;
	exit(-1);
      } catch (...) {
	std::cout << "unknown error" << std::endl;
	exit(-1);
      }

    }

    // double check consistency
    boost::shared_ptr<CapFloor> anothercap = makeCapFloor (CapFloor::Cap,
                                                           caplet->leg,
                                                           caplet->strike,
                                                           caplet->vol);
    double anotherprice = anothercap->NPV();

    // keep in mind
    caplets.push_back (*caplet);

    log.msg (NOTICE, "Caplet %s %s %s %7.4f %7.4f %7.4f %7.4f %7.4f %7.4f",
	     caplet->ccy.c_str(),
	     DateFormatter::toString(caplet->start,DateFormatter::ISO).c_str(),
	     DateFormatter::toString(caplet->end,DateFormatter::ISO).c_str(),
	     caplet->strike,
	     caplet->flatVol,
	     caplet->flatPrice,
	     caplet->vol,
	     caplet->price,
	     anotherprice);

    fprintf (f, "%2d %s %s %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f\n",
	     i,
	     DateFormatter::toString(caplet->start,DateFormatter::ISO).c_str(),
	     DateFormatter::toString(caplet->end,DateFormatter::ISO).c_str(),
	     caplet->flatVol,
	     caplet->vol,
	     caplet->flatPrice,
	     caplet->price,
	     caplet->strike,
	     coupon->rate(),
	     capletPrice (caplet->start, caplet->strike),
	     capletPrice (caplet->start, coupon->rate()) );

    // keep in mind as next caplet's predecessor
    lastFlatVol = caplet->flatVol;
    lastFlatPrice = caplet->flatPrice;

    i++;
  }
  while (endDate < finalMaturity);

  fclose (f);

  /////////////////////////////////////////////////////////////////////////////
  // final consistency check between flat market caps and caplets
  // price market caps
  // - with flat volatility and
  // - using caplets with caplet volatilities
  double capPrice0, capPrice1, err = 0;
  for (std::list<FlatCapFloor>::iterator i=caps.begin(); i != caps.end(); i++){
    // flat cap schedule
    QL::Schedule schedule = MakeSchedule (def.calendar, capStartDate,
                                          i->maturity,
					  def.floatFreq, def.roll);
    // flat cap leg
    leg = FloatingRateCouponVector (schedule, def.roll,
                                    std::vector<double>(1,1),
                                    def.index, def.fixingDays,
                                    std::vector<double>(1,0));
    // flat cap's atm strike
    strike = atmStrike (leg, i->vol);

    // flat cap instrument
    boost::shared_ptr<CapFloor> cap = makeCapFloor (CapFloor::Cap,
                                                    leg,      // cap leg
                                                    strike,   // atm cap strike
                                                    i->vol);  // flat cap vol
    // flat cap price
    capPrice0 = cap->NPV();

    // sum of caplet prices (using same strike as in the flat cap)
    capPrice1 = 0;
    for (std::list<Caplet>::iterator j=caplets.begin();
	 j != caplets.end() && j->end <= schedule.endDate(); j++ ) {
      boost::shared_ptr<CapFloor> cap = makeCapFloor (CapFloor::Cap,
                                                      j->leg,  // caplet leg
                                                      strike,  // flat atm strk
                                                      j->vol); // caplet vol
      capPrice1 += cap->NPV();
    }

    log.msg (NOTICE, "cap term %s vs caplet %.8f %.8f",
             i->term.c_str(), capPrice0, capPrice1 );

    err += (capPrice0-capPrice1)/capPrice0;
  }

  err /= caps.size();

  log.msg (NOTICE, "avg. rel. error %.4f", err );

  if (err > 0.1) log.msg (WARNING, "avg. rel. cap error exceeds 0.1");

  log.msg (NOTICE, "caplet volatilities ok");

  return 0;
}

//-----------------------------------------------------------------------------
Default::Default( std::string type_,
		  std::string ccy_,
		  Handle<YieldTermStructure> rhTermStructure )
  : calendar(TARGET()),
    mmDayCount(Actual360()),
    cmDayCount(Thirty360()),
    actDayCount(ActualActual()) {
//-----------------------------------------------------------------------------
  static Debug log( "Default::Default" );

  instrument      = type_;
  ccy             = ccy_;
  nominal         = 1;
  settlementDays  = 2;
  fixingDays      = 2;

  if ( instrument == "Deposit" ) {
    if ( ccy == "EUR" ) {
      calendar        = mapCalendar         ( "TARGET" );
      roll            = mapRollingConvention( "F" );
      fixedRoll       = mapRollingConvention( "F" );
      fixedFreq       = mapFrequency        ( "A" );
      floatFreq       = mapFrequency        ( "S" );
      floatingTenor   = "6M";
      fixedIsAdjusted = mapYorN             ( "Y" );
      mmDayCount      = mapDayCounter       ( "A360" );
      cmDayCount      = mapDayCounter       ( "ACT" );
    }
    else if ( ccy == "USD" ) {
      calendar        = mapCalendar         ( "NY" );
      roll            = mapRollingConvention( "F" );
      fixedRoll       = mapRollingConvention( "F" );
      fixedFreq       = mapFrequency        ( "A" );
      floatFreq       = mapFrequency        ( "S" );
      floatingTenor   = "3M";
      fixedIsAdjusted = mapYorN             ( "Y" );
      mmDayCount      = mapDayCounter       ( "A360" );
      cmDayCount      = mapDayCounter       ( "ACT" );
    }
    else if ( ccy == "GBP" ) {
      calendar        = mapCalendar         ( "LON" );
      roll            = mapRollingConvention( "F" );
      fixedRoll       = mapRollingConvention( "F" );
      fixedFreq       = mapFrequency        ( "A" );
      floatFreq       = mapFrequency        ( "S" );
      floatingTenor   = "3M";
      fixedIsAdjusted = mapYorN             ( "Y" );
      mmDayCount      = mapDayCounter       ( "A365" );
      cmDayCount      = mapDayCounter       ( "ACT" );
    }
    else {
      log.msg( "no defaults defined for instrument %s with ccy %s",
	     instrument.c_str(), ccy.c_str() );
    }
  }
  else if ( instrument == "Swap" ) {
    if ( ccy == "EUR" ) {
      calendar        = mapCalendar         ( "TARGET" );
      roll            = mapRollingConvention( "F" );
      fixedRoll       = mapRollingConvention( "F" );
      fixedFreq       = mapFrequency        ( "A" );
      floatFreq       = mapFrequency        ( "S" );
      floatingTenor   = "6M";
      fixedIsAdjusted = mapYorN             ( "N" );
      mmDayCount      = mapDayCounter       ( "A360" );
      cmDayCount      = mapDayCounter       ( "ACT" );
      index           = mapIndex            ( "EUR", "EURIBOR", "6M",
                                              rhTermStructure );
    }
    else if ( ccy == "USD" ) {
      calendar        = mapCalendar         ( "NY" );
      roll            = mapRollingConvention( "F" );
      fixedRoll       = mapRollingConvention( "F" );
      fixedFreq       = mapFrequency        ( "A" );
      floatFreq       = mapFrequency        ( "S" );
      floatingTenor   = "3M";
      fixedIsAdjusted = mapYorN             ( "N" );
      mmDayCount      = mapDayCounter       ( "A360" );
      cmDayCount      = mapDayCounter       ( "ACT" );
      index           = mapIndex            ( "USD", "LIBOR", "6M",
                                              rhTermStructure );
    }
    else if ( ccy == "GBP" ) {
      calendar        = mapCalendar         ( "LON" );
      roll            = mapRollingConvention( "F" );
      fixedRoll       = mapRollingConvention( "F" );
      fixedFreq       = mapFrequency        ( "A" );
      floatFreq       = mapFrequency        ( "S" );
      floatingTenor   = "3M";
      fixedIsAdjusted = mapYorN             ( "N" );
      mmDayCount      = mapDayCounter       ( "A360" );
      cmDayCount      = mapDayCounter       ( "ACT" );
      index           = mapIndex            ( "GBP", "LIBOR", "3M",
                                              rhTermStructure );
    }
    else {
      log.msg( CRITICAL, "no defaults defined for instrument %s with ccy %s",
	     instrument.c_str(), ccy.c_str() );
      exit(1);
    }
  }
}

//eof
