
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "engine.h"

namespace RL {

using std::cout;
using std::endl;
using namespace Martingale;

//-----------------------------------------------------------------------------
Engine::Engine (XML::list<RL::MarketPoint> *mkt_)
  : curve ("EUR", mkt_) {
//-----------------------------------------------------------------------------
  Debug log ("Engine::Engine");

  mkt   = mkt_;
  asof  = Settings::instance().evaluationDate();
  model = "";

  std::cout << "engine date:  "
	    << DateFormatter::toString( asof, DateFormatter::ISO )
	    << "  (" << DateFormatter::toString( asof) << ")" << std::endl;

  lmmParams.init();

  char* e = NULL;
  if (! (e = getenv("ENV_MODEL"))) {
    log.msg (NOTICE, "environment variable ENV_MODEL not set, use default FWD model");
    model = "FWD";
  }
  else if (!strstr (e, "LMM") && !strstr (e, "FWD")) {
    log.msg (NOTICE, "pricing model %s not recognised, use default FWD model", e);
    model = "FWD";
  }
  else if (strstr (e, "LMM")) {
    log.msg (NOTICE, "pricing model %s, setup LMM", e);
    model = "LMM";

    if ((e = getenv("ENV_LMM_PARAMETER_FILE"))) {
      if (lmmParams.readObject(e) < 0) {
	log.msg (ALERT, "error reading file %s, exit.", e);
	exit(-1);
      }
      else log.msg (NOTICE, "LMM parameters read from file %s", e);

      log.msg (NOTICE, "LMM type        %s", lmmParams.lmmType.c_str());
      log.msg (NOTICE, "LMM lowFactor   %d", lmmParams.lowFactor);
      log.msg (NOTICE, "LMM volSurface  %s", lmmParams.volSurfaceType.c_str());
      log.msg (NOTICE, "LMM correlation %s", lmmParams.correlationType.c_str());
      log.msg (NOTICE, "LMM nVals       %d", lmmParams.nVals);
      log.msg (NOTICE, "LMM nPaths      %d", lmmParams.nPaths);
      log.msg (NOTICE, "LMM volSets     %d", lmmParams.volSurfaces.size());
      log.msg (NOTICE, "LMM corSets     %d", lmmParams.correlations.size());

      // check consistency
      // TODO: move to validate method in obj.h ?
      XML::list<RL::LmmVolSurface>::iterator itvs;
      int vols = 0;
      for (itvs = lmmParams.volSurfaces.begin(); itvs != lmmParams.volSurfaces.end(); itvs++) {
	if (lmmParams.volSurfaceType == itvs->type) vols++;
      }

      XML::list<RL::LmmCorrelations>::iterator itc;
      int cors = 0;
      for (itc = lmmParams.correlations.begin(); itc != lmmParams.correlations.end(); itc++) {
	if (lmmParams.correlationType == itc->type) cors++;
      }

      if (vols == 0) {
	log.msg (ALERT, "initial VolSurface parameters missing for type %s, exit.",
		 lmmParams.volSurfaceType.c_str());
	exit(-1);
      }

      if (cors == 0) {
	log.msg (ALERT, "initial Correlation parameters missing for type %s, exit.",
		 lmmParams.correlationType.c_str());
	exit(-1);
      }
    }
    else {
      log.msg (ALERT, "environment variable RL_LMM_PARAMETER_FILE not set, exit.");
      exit (-1);
    }
  }
  else log.msg (NOTICE, "pricing model %s, use default FWD, do not setup LMM", e);

  if (getenv(ENV_FLAT_CURVE))
    curve.link(0);
  else
    curve.link(1);
}

//-----------------------------------------------------------------------------
int Engine::price (RL::Instrument& inst, LiborMarketModel* lmm_) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::price(Instrument&)");

  if (getenv(ENV_FLAT_CURVE))
    curve.link(0);
  else
    curve.link(1);

  std::string ccy = inst.legs.begin()->ccy;
  if (curve.ccy() != ccy)
    log.msg (ALERT, "curve ccy %s does not match leg ccy %s, exit.",
	     curve.ccy().c_str(), ccy.c_str());

  log.msg (NOTICE, "instrument (%s), structure type (%s)",
	   inst.id.c_str(), inst.structure.type.c_str());

  if (inst.structure.type == "LINEAR" ||
      inst.structure.type == "") {

    log.msg (NOTICE, "number of legs = %d", inst.legs.size() );

    inst.stats.date = DateFormatter::toString (asof, DateFormatter::ISO);

    // get info which model to use from environment
    LiborMarketModel* lmm = NULL;
    if (model == "LMM") {
      if (lmm_) lmm = lmm_;
      else      lmm = setupModel (inst);
    }

    if (lmm)
      pathPricer (inst, lmm);
    //    legPricer (inst, lmm);
    //    else
      legPricer (inst, NULL);

    if (inst.structure.underlying.use_count())
      log.msg (ALERT, "underlying instrument id %s ignored",
	       inst.structure.underlying.get()->id.c_str());
  }
  else {
    // FIXME: non-linear structures using underlyings to be done
    if (inst.structure.underlying.use_count()) {
      log.msg (ALERT, "underlying instrument id %s",
               inst.structure.underlying.get()->id.c_str());
      log.msg (ALERT, "structure/underlying not yet implemented" );
    }
  }

  return 0;
}

//-----------------------------------------------------------------------------
MTG::LiborMarketModel* Engine::setupModel (RL::Instrument& inst) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::setupModel (instrument)");

  // assume ccy is unique, checked below
  Default def ("Swap", inst.legs.begin()->ccy, curve.ts);

  // construct a "reference" float leg for setting up the libor market model
  // the leg spans min(startDate) to max(matDate)
  // frequency of payments is the maximum that occurs amoung all legs
  // FIXME: cal, roll and adjusted are taken arbitrarily from the last
  //        non-fixed leg
  //        if there are only fixed legs, they are set to defaults
  // nominal is set to 1
  // spread is set to 0
  // index is set to the currency's default

  QL::Date              start     = QL::Date::maxDate();
  QL::Date              end       = QL::Date::minDate();
  Frequency             frequency = QL::NoFrequency;
  QL::Calendar          calendar  = QL::NullCalendar();
  QL::BusinessDayConvention roll  = Following;
  bool                  adjusted  = false;

  XML::list<RL::Leg>::iterator it;
  for (it = inst.legs.begin(); it != inst.legs.end(); it++) {
    if (it->ccy != def.ccy) log.msg (ALERT, "ccy is not unique, exit.");
    QL::Date effDate = parseDate (it->index->resetSched.startDate, asof);
    QL::Date matDate = parseDate (it->index->resetSched.endDate, effDate);
    Frequency freq   = mapFrequency (it->index->resetSched.frequency);
    if (effDate < start)  start     = effDate;
    if (matDate > end)    end       = matDate;
    if (freq > frequency) frequency = freq;
    if (it->index->type != "FIXED") {
      calendar = mapCalendar (it->index->resetSched.calendar);
      roll     = mapRollingConvention (it->index->resetSched.bdRule);
      adjusted = mapYorN (it->index->resetSched.adjusted);
    }
  }

  log.msg (NOTICE, "schedule start end freq = %s %s %d",
           DateFormatter::toString (start, DateFormatter::ISO).c_str(),
           DateFormatter::toString (end, DateFormatter::ISO).c_str(),
           frequency);

  QL::Schedule schedule = MakeSchedule(calendar,
                                       start, end,
                                       frequency, roll).longFinalPeriod();

  std::vector<boost::shared_ptr<CashFlow> >
    cashflow = FloatingRateCouponVector (schedule,
                                         def.roll,
                                         std::vector<double>(1,1),
                                         def.index,
                                         def.fixingDays,
                                         std::vector<QL::Spread>(1,0));

  log.msg (NOTICE, "reference floatleg (start / end / coupon date):");

  boost::shared_ptr<FloatingRateCoupon> coupon;
  for (Size i=0; i<cashflow.size(); i++) {
    coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[i]);
    log.msg (NOTICE, "%s %s %s",
	     DateFormatter::toString (coupon->accrualStartDate(),
                                      DateFormatter::ISO).c_str(),
	     DateFormatter::toString (coupon->accrualEndDate(),
                                      DateFormatter::ISO).c_str(),
	     DateFormatter::toString (coupon->date(),
                                      DateFormatter::ISO).c_str());
  }

  MTG::LiborMarketModel* lmm = setupModel (cashflow);

  log.msg (NOTICE, "libor market model set up");

  return lmm;
}

//-----------------------------------------------------------------------------
MTG::LiborMarketModel*
Engine::setupModel (std::vector<boost::shared_ptr<CashFlow> > cashflow) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::setupModel (cashflow)");

  Default def( "Swap", curve.ccy(), curve.ts );

  DayCounter dc = Actual365Fixed();
  boost::shared_ptr<FloatingRateCoupon> coupon;

  int n = cashflow.size();

  RealArray1D T(n+1);        // period start times and final period's end time
  RealArray1D deltas(n);     // year fractions
  RealArray1D c(n);          // volatility scaling factors, k_j
  RealArray1D L0(n);         // initial forward libors
  MTG::Array1D<CapletData*>   caplets(n-1,1);   // caplet calibration inst.
  MTG::Array1D<SwaptionData*> swaptions(n-1,1); // swaption calibration inst.

  log.msg (NOTICE, "cash flow size = %d", cashflow.size() );
  log.msg (NOTICE, "dimension n = %d", n );

  // pick the relevant volatility surface parameters
  RL::LmmVolSurface volSurface;
  for (XML::list<RL::LmmVolSurface>::iterator it = lmmParams.volSurfaces.begin();
       it != lmmParams.volSurfaces.end(); it++) {
    volSurface = *it;
    if (lmmParams.volSurfaceType == it->type) break;
  }

  // set up volatility surface
  VolSurface* vols;
  if (volSurface.type == "M")
    vols = new M_VolSurface (volSurface.a,
                             volSurface.b,
                             volSurface.c,
                             volSurface.d);
  else if (volSurface.type == "JR")
    vols = new JR_VolSurface (volSurface.a,
                              volSurface.b,
                              volSurface.c,
                              volSurface.d);
  else if (volSurface.type == "CONST")
    vols = new CONST_VolSurface (volSurface.a,
                                 volSurface.b,
                                 volSurface.c,
                                 volSurface.d);
  else {
    log.msg (ALERT, "volatility surface type %s not recognised",
             volSurface.type.c_str());
    exit(-1);
  }

  log.msg (NOTICE, "volSurfaceType %s", vols->volSurfaceType().c_str() );
  log.msg (NOTICE, "%s", vols->as_string().c_str() );

  // link to flat or deposit/swap curve
  log.msg (NOTICE, "link to depo/swap curve");
  if (getenv(ENV_FLAT_CURVE))
    curve.link(0);
  else
    curve.link(1);

  // set up times, period lenghts (deltas), scaling factors (c)
  // and initial forward libors (L0)
  log.msg (NOTICE, "init deltas, c, libors");
  for (Size i = 0; i < cashflow.size(); i++) {
    coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[i]);
    try {
      T[i] = dc.yearFraction (asof, coupon->accrualStartDate());
      deltas[i] = dc.yearFraction (coupon->accrualStartDate(),
                                   coupon->accrualEndDate());
      c[i] = 0.2 + 0.1*Random::U01();
      L0[i] = coupon->rate(); //fixing();
    }
    CATCH_ALERT;
    log.msg (NOTICE, "%d %s %s %s %.4f %.4f %.4f %.4f", i,
	     DateFormatter::toString (asof, DateFormatter::ISO).c_str(),
	     DateFormatter::toString (coupon->accrualStartDate(),
                                      DateFormatter::ISO).c_str(),
	     DateFormatter::toString (coupon->accrualEndDate(),
                                      DateFormatter::ISO).c_str(),
	     T[i], deltas[i], c[i], L0[i] );
  }
  T[n] = dc.yearFraction (asof, coupon->accrualEndDate());

  // pick the relevant correlation parameters
  RL::LmmCorrelations correlations;
  for (XML::list<RL::LmmCorrelations>::iterator it = lmmParams.correlations.begin();
       it != lmmParams.correlations.end(); it++) {
    correlations = *it;
    if (lmmParams.correlationType == it->type) break;
  }

  // set up correlations
  Correlations* corrs;
  if (correlations.type == "JR")
    corrs = new JR_Correlations (T, correlations.beta);
  else if (correlations.type == "CS")
    corrs = new CS_Correlations (n, correlations.alpha,
                                 correlations.beta, correlations.rho);
  else {
    log.msg (ALERT, "correlation type %s not recognised",
             correlations.type.c_str());
    exit(-1);
  }

  log.msg (NOTICE, "lmmCorrType %s", corrs->correlationType().c_str() );
  log.msg (NOTICE, "%s", corrs->as_string().c_str() );

  // initialise factor loadings given the vectors set up so far
  log.msg (NOTICE, "construct libor factor loadings");

  LiborFactorLoading* fl = new LiborFactorLoading (L0, deltas, c, vols,
                                                   corrs, T[0]);

  log.msg (NOTICE, "%s", fl->getType()->as_string().c_str() );

  // cap and swaption files,
  // not needed if caplet and swaption arrays are set up, see below
  std::string capletInFile="CapletsIn.txt";
  std::string swaptionInFile="SwaptionsIn.txt";

  // set up caplet calibration instruments
  // divide by final discount factor for the forward price at horizon T_n
  // this value is matched by the calibration routine
  FILE* g = fopen ("CapletsIn.txt", "w");
  fprintf (g, "%-2s %-10s %-10s %-13s %-13s %-13s\n",
	   "#", "From", "To", "Strike", "Price", "FwdPrice");
  boost::shared_ptr<FloatingRateCoupon> finalCoupon
    = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[n-1]);
  double finalDiscount = curve.ts->discount (finalCoupon->date());
  for (Size i = 1; i < cashflow.size(); i++) {
    coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[i]);
    caplets[i] = new CapletData;
    caplets[i]->i            = i;
    caplets[i]->strike       = coupon->rate();
    caplets[i]->forwardPrice = curve.capletPrice (coupon->accrualStartDate(), coupon->rate()) / finalDiscount;

    fprintf (g, "%2ld %s %s %13.8f %13.8f %13.8f\n",
	     i,
	     DateFormatter::toString (coupon->accrualStartDate(),
                                      DateFormatter::ISO).c_str(),
	     DateFormatter::toString (coupon->accrualEndDate(),
                                      DateFormatter::ISO).c_str(),
	     caplets[i]->strike,
	     curve.capletPrice (coupon->accrualStartDate(), coupon->rate()),
	     caplets[i]->forwardPrice);
  }
  fclose (g);

  // set up swaption calibration instruments
  // divide by final discount factor for the forward price at horizon T_n
  // this value is matched by the calibration routine
  g = fopen ("SwaptionsIn.txt", "w");
  fprintf (g, "%-3s %-3s %-10s %-10s %-10s %-10s %-13s %-13s %-10s %-13s\n",
	   "#", "xM", "From", "To", "Fixed", "Fair", "Swap", "Swaption",
           "Vol", "FwdPrice");

  double volatility = 0;
  boost::shared_ptr<SimpleSwap> swap;
  boost::shared_ptr<QL::Swaption>   swaption;
  for (Size i = 1; i < cashflow.size(); i++) {
    Period period ((cashflow.size() - i) * 12 / def.floatFreq, Months);
    coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[i]);
    try {
      if (i <= cashflow.size()-2)
	volatility = 0.01*curve.volMatrix->volatility (coupon->accrualStartDate(), period, 0.0);
      swap = makeSimpleAtmSwap (coupon->accrualStartDate(), period.length(),
                                period.units(), 0, true, "EUR");
      swaption = makeSwaption (swap, coupon->accrualStartDate(), volatility);
    }
    CATCH_ALERT;

    swaptions[i] = new SwaptionData;
    swaptions[i]->p            = i;
    swaptions[i]->q            = n;
    swaptions[i]->strike       = swap->fixedRate();
    swaptions[i]->forwardPrice = swaption->NPV() / finalDiscount;

    fprintf (g, "%3ld %3ld %s %s %10.6f %10.6f %+12.6f %+12.6f %10.6f %+12.6f\n",
	     i, (cashflow.size() - i) * 12 / def.floatFreq,
	     DateFormatter::toString (swap->startDate(),
                                      DateFormatter::ISO).c_str(),
	     DateFormatter::toString (swap->maturity(),
                                      DateFormatter::ISO).c_str(),
	     swap->fixedRate(),
	     swap->fairRate(),
	     swap->NPV(),
	     swaption->NPV(),
	     volatility,
	     swaptions[i]->forwardPrice);
  }
  fclose (g);

  // set up calibrator
  LmmCalibrator* cal;
  if (lmmParams.lmmType == "PC") {
    cal = new PredictorCorrectorLmmCalibrator (fl,
                                               capletInFile.c_str(),
                                               swaptionInFile.c_str(),
					       "CapletsOut.txt",
                                               "SwaptionsOut.txt",
					       caplets, swaptions);
    log.msg (NOTICE, "using PC calibrator");
  }
  else {
    cal = new DriftlessLmmCalibrator (fl,
                                      capletInFile.c_str(),
                                      swaptionInFile.c_str(),
				      "CapletsOut.txt",
                                      "SwaptionsOut.txt",
				      caplets, swaptions);
    log.msg (NOTICE, "using DL calibrator");
  }
  VolSurface*   vol = fl->getVolSurface();
  Correlations* cor = fl->getCorrelations();

  log.msg (NOTICE, "Calibrating factor loadings");
  log.msg (NOTICE, "VolSurface:   %s",
           VolSurface::volSurfaceType(vol->getType()).c_str());
  log.msg (NOTICE, "Correlations: %s",
           Correlations::correlationType(cor->getType()).c_str());
  log.msg (NOTICE, "Dimension:    %d", n);
  log.msg (NOTICE, "Initial values:");
  log.msg (NOTICE, " a      = %+.4f", vol->getA());
  log.msg (NOTICE, " b      = %+.4f", vol->getB());
  log.msg (NOTICE, " c      = %+.4f", vol->getC());
  log.msg (NOTICE, " d      = %+.4f", vol->getD());
  log.msg (NOTICE, " alpha  = %+.4f", cor->getAlpha());
  log.msg (NOTICE, " beta   = %+.4f", cor->getBeta());
  log.msg (NOTICE, " rho_oo = %+.4f", cor->getRho());
  log.msg (NOTICE, " avg rel error = %.2f%%",
           cal->meanRelativeCalibrationError());

  cal->calibrate(lmmParams.nVals);

  log.msg (NOTICE, "Calibration results:");
  log.msg (NOTICE, " a      = %+.4f", vol->getA());
  log.msg (NOTICE, " b      = %+.4f", vol->getB());
  log.msg (NOTICE, " c      = %+.4f", vol->getC());
  log.msg (NOTICE, " d      = %+.4f", vol->getD());
  log.msg (NOTICE, " alpha  = %+.4f", cor->getAlpha());
  log.msg (NOTICE, " beta   = %+.4f", cor->getBeta());
  log.msg (NOTICE, " rho_oo = %+.4f", cor->getRho());
  log.msg (NOTICE, " avg rel error = %.2f%%",
           cal->meanRelativeCalibrationError());

  log.msg (NOTICE, "construct lmm from factor loadings");

  // set up the model
  LiborMarketModel* lmm;
  if (lmmParams.lmmType == "PC")
    lmm = new PredictorCorrectorLMM (fl);
  else if (lmmParams.lmmType == "FPC")
    lmm = new FastPredictorCorrectorLMM (fl);
  else if (lmmParams.lmmType == "LFDL")
    lmm = new LowFactorDriftlessLMM (fl, lmmParams.lowFactor);
  else
    lmm = new DriftlessLMM (fl);

  log.msg (NOTICE, "done");

  return lmm;
}

//-----------------------------------------------------------------------------
MTG::LiborMarketModel* Engine::setupModel () {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::setupModel ()");

  // FIXME: change hard coded file name to reading name from environment
  char* summit_caplets   = "SummitData/summit_atm_caplets.txt";
  char* summit_swaptions = "SummitData/summit_atm_swaptions.txt";
  int capCols      = 10;
  int swaptionCols =9;

  // find the size of the calibration list from reading the 10 column caplet file
  char line[1000], arg[10][100];
  int  nMax = 0, nMin = 10000, col;
  FILE * f = fopen (summit_caplets, "r");
  if (!f) log.msg (ALERT, "error opening file %s", summit_caplets);
  while (fgets (line, 1000, f) != NULL) {
    if (strncmp (line, "#", 1) != 0) { // skip comment lines (e.g. header line)
      strcpy (arg[0], strtok (line, " "));
      for (col = 1; col < capCols; col++) strcpy (arg[col], strtok (NULL, " "));
      //      for (col = 0; col < capCols; col++) printf ("%s ", arg[col]);
      // last argument contains \n
      nMax = max (nMax, atoi (arg[1]));
      nMin = min (nMin, atoi (arg[1]));
    }
  }
  fclose (f);

  printf ("period index is in [%d:%d]\n", nMin, nMax);

  if (nMin != 0)
    log.msg (ALERT, "minimum period index in caplet input file != 0");

  int n = nMax + 1; // number of periods

  //  exit(0);
  //  int n = 19; // TODO read from file

  RealArray1D T(n+1);                           // period start times and final period's end time
  RealArray1D deltas(n);                        // year fractions
  RealArray1D c(n);                             // volatility scaling factors, k_j
  RealArray1D L0(n);                            // initial forward libors
  MTG::Array1D<CapletData*>   caplets(n-1,1);   // caplet calibration instruments
  MTG::Array1D<SwaptionData*> swaptions(n-1,1); // swaption calibration instruments

  log.msg (NOTICE, "dimension n = %d", n );

  // pick the relevant volatility surface parameters
  RL::LmmVolSurface volSurface;
  for (XML::list<RL::LmmVolSurface>::iterator it = lmmParams.volSurfaces.begin();
       it != lmmParams.volSurfaces.end(); it++) {
    volSurface = *it;
    if (lmmParams.volSurfaceType == it->type) break;
  }

  // set up volatility surface
  VolSurface* vols;
  if (volSurface.type == "M")
    vols = new M_VolSurface (volSurface.a, volSurface.b, volSurface.c, volSurface.d);
  else if (volSurface.type == "JR")
    vols = new JR_VolSurface (volSurface.a, volSurface.b, volSurface.c, volSurface.d);
  else if (volSurface.type == "CONST")
    vols = new CONST_VolSurface (volSurface.a, volSurface.b, volSurface.c, volSurface.d);
  else {
    log.msg (ALERT, "volatility surface type %s not recognised", volSurface.type.c_str());
    exit(-1);
  }

  log.msg (NOTICE, "volSurfaceType %s", vols->volSurfaceType().c_str() );
  log.msg (NOTICE, "%s", vols->as_string().c_str() );

  // link to flat or deposit/swap curve
  log.msg (NOTICE, "link to depo/swap curve");
  if (getenv(ENV_FLAT_CURVE))
    curve.link(0);
  else
    curve.link(1);

  // read forward curve from 10 column caplet input file

  int i = 0;
  if (!(f = fopen (summit_caplets, "r")))
    log.msg (ALERT, "error opening file %s", summit_caplets);
  while (fgets (line, 1000, f) != NULL) {
    if (strncmp (line, "#", 1) != 0) { // skip comment lines (e.g. header line)
      strcpy (arg[0], strtok (line, " "));
      for (col  = 1; col < capCols; col++) strcpy (arg[col], strtok (NULL, " "));
      for (col = 0; col < capCols; col++) printf ("%s ", arg[col]);
      if (atoi(arg[1]) != i) log.msg (ALERT, "caplets not ordered in input file");
      T[i]      = atof (arg[4]);
      deltas[i] = atof (arg[5]);
      L0[i]     = atof (arg[6]);
      c[i]      = 0.2 + 0.1*Random::U01();
      printf ("%d %.4f %.4f %.4f %.4f\n", i, T[i], deltas[i], L0[i], c[i]);
      i++;
    }
  }
  T[i] = T[i-1] + deltas[i-1];
  fclose (f);

  printf ("curve data done, Tn = %.4f\n", T[n]);

  // pick the relevant correlation parameters
  RL::LmmCorrelations correlations;
  for (XML::list<RL::LmmCorrelations>::iterator it = lmmParams.correlations.begin();
       it != lmmParams.correlations.end(); it++) {
    correlations = *it;
    if (lmmParams.correlationType == it->type) break;
  }

  // set up correlations
  Correlations* corrs;
  if (correlations.type == "JR")
    corrs = new JR_Correlations (T, correlations.beta);
  else if (correlations.type == "CS")
    corrs = new CS_Correlations (n, correlations.alpha, correlations.beta, correlations.rho);
  else {
    log.msg (ALERT, "correlation type %s not recognised", correlations.type.c_str());
    exit(-1);
  }

  log.msg (NOTICE, "lmmCorrType %s", corrs->correlationType().c_str() );
  log.msg (NOTICE, "%s", corrs->as_string().c_str() );

  // initialise factor loadings given the vectors set up so far
  log.msg (NOTICE, "construct libor factor loadings");

  LiborFactorLoading* fl = new LiborFactorLoading (L0, deltas, c, vols, corrs, T[0]);

  log.msg (NOTICE, "%s", fl->getType()->as_string().c_str() );

  // set up caplet calibration instruments from a 10 column input file


  if (!(f = fopen (summit_caplets, "r")))
    log.msg (ALERT, "error opening file %s", summit_caplets);
  i = 1;
  while (fgets (line, 1000, f) != NULL) {
    if (strncmp (line, "#", 1) != 0) { // skip comment lines (e.g. header line)
      strcpy (arg[0], strtok (line, " "));
      for (col  = 1; col < capCols; col++) strcpy (arg[col], strtok (NULL, " "));
      if (atoi (arg[1]) > 0) {
	caplets[i] = new CapletData;
	caplets[i]->i            = atoi (arg[1]);
	caplets[i]->strike       = atof (arg[6]);
	caplets[i]->forwardPrice = atof (arg[9]);

	if (caplets[i]->i != i) log.msg (ALERT, "caplets not ordered in input file");
	i++;
      }
    }
  }
  fclose (f);

  printf ("caplets doen\n");

  // set up swaption calibration instruments
  // divide by final discount factor to produce the forward price at horizon T_n
  // this value is matched by the calibration routine

  if (!(f = fopen (summit_swaptions, "r")))
    log.msg (ALERT, "error opening file %s", summit_swaptions);
  i = 1;
  while (fgets (line, 1000, f) != NULL) {
    if (strncmp (line, "#", 1) != 0) { // skip comment lines (e.g. header line)
      strcpy (arg[0], strtok (line, " "));
      for (col  = 1; col < swaptionCols; col++) strcpy (arg[col], strtok (NULL, " "));
      if (atoi (arg[1]) > 0) {
	swaptions[i] = new SwaptionData;
	swaptions[i]->p            = atoi (arg[1]);
	swaptions[i]->q            = atoi (arg[2]);
	swaptions[i]->strike       = atof (arg[5]);
	swaptions[i]->forwardPrice = atof (arg[8]);

	if (swaptions[i]->p != i) log.msg (ALERT, "swaptions not ordered in input file");
	i++;
      }
    }
  }
  fclose (f);

  printf ("swaptions done\n");

  // cap and swaption files, not needed if caplet and swaption arrays are set up, see below
  std::string capletInFile="CapletsInput.txt";
  std::string swaptionInFile="SwaptionsInput.txt";

  // set up calibrator
  LmmCalibrator* cal;
  if (lmmParams.lmmType == "PC") {
    cal = new PredictorCorrectorLmmCalibrator (fl, capletInFile.c_str(), swaptionInFile.c_str(),
					       "CapletsOut.txt", "SwaptionsOut.txt",
					       caplets, swaptions);
    log.msg (NOTICE, "using PC calibrator");
  }
  else {
    cal = new DriftlessLmmCalibrator (fl, capletInFile.c_str(), swaptionInFile.c_str(),
				      "CapletsOut.txt", "SwaptionsOut.txt",
				      caplets, swaptions);
    log.msg (NOTICE, "using DL calibrator");
  }
  VolSurface*   vol = fl->getVolSurface();
  Correlations* cor = fl->getCorrelations();

  log.msg (NOTICE, "Calibrating factor loadings");
  log.msg (NOTICE, "VolSurface:   %s", VolSurface::volSurfaceType(vol->getType()).c_str());
  log.msg (NOTICE, "Correlations: %s", Correlations::correlationType(cor->getType()).c_str());
  log.msg (NOTICE, "Dimension:    %d", n);
  log.msg (NOTICE, "Initial values:");
  log.msg (NOTICE, " a      = %+.4f", vol->getA());
  log.msg (NOTICE, " b      = %+.4f", vol->getB());
  log.msg (NOTICE, " c      = %+.4f", vol->getC());
  log.msg (NOTICE, " d      = %+.4f", vol->getD());
  log.msg (NOTICE, " alpha  = %+.4f", cor->getAlpha());
  log.msg (NOTICE, " beta   = %+.4f", cor->getBeta());
  log.msg (NOTICE, " rho_oo = %+.4f", cor->getRho());
  log.msg (NOTICE, " avg rel error = %.2f%%", cal->meanRelativeCalibrationError());

  cal->calibrate(lmmParams.nVals);

  log.msg (NOTICE, "Calibration results:");
  log.msg (NOTICE, " a      = %+.4f", vol->getA());
  log.msg (NOTICE, " b      = %+.4f", vol->getB());
  log.msg (NOTICE, " c      = %+.4f", vol->getC());
  log.msg (NOTICE, " d      = %+.4f", vol->getD());
  log.msg (NOTICE, " alpha  = %+.4f", cor->getAlpha());
  log.msg (NOTICE, " beta   = %+.4f", cor->getBeta());
  log.msg (NOTICE, " rho_oo = %+.4f", cor->getRho());
  log.msg (NOTICE, " avg rel error = %.2f%%", cal->meanRelativeCalibrationError());

  // set up the model
  LiborMarketModel* lmm;
  if (lmmParams.lmmType == "PC")        lmm = new PredictorCorrectorLMM (fl);
  else if (lmmParams.lmmType == "FPC")  lmm = new FastPredictorCorrectorLMM (fl);
  else if (lmmParams.lmmType == "LFDL") lmm = new LowFactorDriftlessLMM (fl, lmmParams.lowFactor);
  else                                  lmm = new DriftlessLMM (fl);

  return lmm;
}

//-----------------------------------------------------------------------------
std::vector<boost::shared_ptr<CashFlow> >
Engine::cashFlowVector (RL::Leg& leg) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::cashFlowVector");

  std::vector<boost::shared_ptr<CashFlow> > cashflow;

  QL::Calendar calendar  = mapCalendar (leg.index->resetSched.calendar);
  QL::Date     startDate = parseDate (leg.index->resetSched.startDate, asof);
  QL::Date     maturity  = parseDate (leg.index->resetSched.endDate, startDate);
  Frequency    frequency = mapFrequency (leg.index->resetSched.frequency);
  QL::BusinessDayConvention roll =
    mapRollingConvention (leg.index->resetSched.bdRule);

//  bool                  adjusted  = mapYorN (leg.index->resetSched.adjusted);

  QL::Schedule schedule = MakeSchedule (calendar,
					startDate,
					maturity,
					frequency,
					roll); //, adjusted);

  double  nominal = leg.notional.begin()->amt;   // FIXME

  if (leg.index->type == "FIXED") {
    log.msg (NOTICE, "build fixed leg id (%s)", leg.id.c_str() );

    QL::DayCounter daycount = mapDayCounter (leg.index->dayCount);
    QL::Rate       rate     = leg.index->rate;

    std::vector<boost::shared_ptr<CashFlow> >
      cashflow = FixedRateCouponVector (schedule, roll, // FIXME roll
                                        std::vector<double>(1,nominal),
                                        std::vector<QL::Rate>(1,rate),
                                        daycount);

    return cashflow;
  }
  else if (leg.index->type == "FLOAT" ||
	   leg.index->type == "FLOATCF" ||
	   leg.index->type == "FORM" ) {
    log.msg (NOTICE, "build float leg id (%s)", leg.id.c_str() );

    boost::shared_ptr<Xibor> index =
      mapIndex (leg.index->ccy, leg.index->name, leg.index->term, curve.ts);
    QL::Spread    spread = leg.index->spread;
    int       fixingdays = 2; // FIXME

    std::vector<boost::shared_ptr<CashFlow> >
      cashflow = FloatingRateCouponVector (schedule, roll,
                                           std::vector<double>(1,nominal),
                                           index,
                                           fixingdays,
                                           std::vector<QL::Spread>(1,spread));
    return cashflow;
  }
  else log.msg (ALERT, "index type %s not yet implemented, exit",
                leg.index->type.c_str());

  return cashflow; // never returned
}

//-----------------------------------------------------------------------------
int Engine::legPricer (RL::Instrument& inst, MTG::LiborMarketModel* lmm) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::legPricer");

  std::string ccy = inst.legs.begin()->ccy;
  if (curve.ccy() != ccy)
    log.msg (ALERT, "curve ccy %s does not match leg ccy %s, exit.",
	     curve.ccy().c_str(), ccy.c_str());

  XML::list<RL::Leg>::iterator it;
  for (it = inst.legs.begin(); it != inst.legs.end(); it++) {

    printf ("pricing instrument %s leg %s ...\n",
            inst.id.c_str(), it->id.c_str());

    // price individual legs
    if (it->ccy == ccy) {
      price (*it, lmm);

      // accumulate stats
      inst.stats.ccy = ccy;
      inst.stats.npv  += it->stats.npv;
      inst.stats.gpv  += it->stats.gpv;
      inst.stats.acc  += it->stats.acc;
      inst.stats.pv01 += it->stats.pv01;
      inst.stats.bpv  += it->stats.bpv;
    }
    else {
      // FIXME: cross ccy
      log.msg (ALERT, "leg ccy is not unique for instrument %s (%s,%s)",
	       inst.id.c_str(),
	       ccy.c_str(),
	       it->stats.ccy.c_str());
      return -1;
    }
  }

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::pathPricer (RL::Instrument& inst, MTG::LiborMarketModel* lmm) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::pathPricer");

  int n = lmm->getDimension();

  int paths = 1000;
  char *e;
  if ((e=getenv("ENV_LMM_PATHS")))
    paths = atoi(e);
  else
    log.msg (ERROR, "ENV_LMM_PATHS not set, using default %d", paths);

  typedef std::vector<boost::shared_ptr<CashFlow> > CFVector;

  // cache for each leg's cash flow vector
  std::list<CFVector>                    cfvlist;
  std::list<CFVector>::iterator          cfv;

  // pay off vectors for each leg
  std::list<RealArray1D>                 edcflist;
  std::list<RealArray1D>::iterator       edcf;

  printf ("pricing instrument %s ...\n", inst.id.c_str());

  // cache QL cash flow and payoff vectors for each leg
  XML::list<RL::Leg>::iterator leg;
  for (leg = inst.legs.begin(); leg != inst.legs.end(); leg++) {
    cfvlist.push_back (cashFlowVector (*leg));
    edcflist.push_back (*(new RealArray1D (n+1)));
  }

  // init payoff vectors for each leg
  for (edcf = edcflist.begin(); edcf != edcflist.end(); edcf++)
    for (int t = 0; t <= n; t++) (*edcf)[t] = 0;

  // outer loop: path generator ===============================================
  for (int p = 1; p <= paths; p++) {
    printf ("paths %5d %3.0f%%\r", p, 100.0*(p+1)/paths);
    fflush(stdout);

    lmm->newPath();

    // inner loop: scan legs and aggregate expected discounted payoffs --------
    for (leg = inst.legs.begin(), cfv = cfvlist.begin(), edcf = edcflist.begin();
	 leg != inst.legs.end(); leg++, cfv++, edcf++) {

      if (leg->index->type == "FIXED")

	pathPricerFixed (*edcf, *leg, *cfv, lmm, p, 1, paths);

      else if (leg->index->type == "FLOAT" ||
	       leg->index->type == "FLOATCF" ||
	       leg->index->type == "FORM" )

	pathPricerFloat (*edcf, *leg, *cfv, lmm, p, 1, paths);

      else

	log.msg (ALERT, "index type %s not recognised, exit.",
                 leg->index->type.c_str());

    } // end of leg loop ------------------------------------------------------
  } // end of path loop =======================================================

  printf ("\n");

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::price (RL::Leg& leg, MTG::LiborMarketModel* lmm) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::price(Leg&)");

  if (leg.type != "INDEX") {
    log.msg (ALERT, "leg type %s not yet implemented, exit", leg.type.c_str());
    exit(-1);
  }

  if (leg.index->type != "FIXED"   &&
      leg.index->type != "FLOAT"   &&
      leg.index->type != "FLOATCF" &&
      leg.index->type != "FORM") {
    log.msg (ALERT, "index type %s not yet implemented, exit", leg.index->type.c_str());
    exit(-1);
  }

  leg.stats.npv = 0;
  leg.flows.init();

  // build a fixed rate coupon vector
  if (leg.index->type == "FIXED") {
    log.msg (NOTICE, "build fixed leg id (%s)", leg.id.c_str() );
    priceFixedLeg (leg);
    if (lmm) priceFixedLegLMM (leg, lmm);
  }
  // build floating rate coupon vector
  else if (leg.index->type == "FLOAT" ||
	   leg.index->type == "FLOATCF" ||
	   leg.index->type == "FORM" ) {
    log.msg (NOTICE, "build float leg id (%s)", leg.id.c_str() );

    priceFloatLeg (leg);
    if (lmm) priceFloatLegLMM (leg, lmm);
  }
  else {
    log.msg (ALERT, "index type %s not yet implemented, exit", leg.index->type.c_str());
    exit(-1);
  }

  leg.stats.date = DateFormatter::toString (asof, DateFormatter::ISO);
  leg.stats.ccy  = leg.ccy;

  log.msg (NOTICE, "leg %s ccy %s npv %.2f", leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::priceFixedLeg (RL::Leg& leg) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::priceFixedLeg");

  std::vector<boost::shared_ptr<CashFlow> > cashflow = cashFlowVector (leg);

  leg.stats.npv = 0;

  boost::shared_ptr<FixedRateCoupon> coupon;
  for (Size i=0; i<cashflow.size(); i++) {
    coupon = boost::dynamic_pointer_cast<FixedRateCoupon>(cashflow[i]); // cast from Handle<CashFlow> to Handle<FloatingRateCoupon>
    leg.stats.npv += coupon->amount() * curve.ts->discount(coupon->date());

    leg.flows.append(new Flow(
      DateFormatter::toString (coupon->accrualStartDate(), DateFormatter::ISO),
      DateFormatter::toString (coupon->accrualEndDate(), DateFormatter::ISO),
      coupon->accrualPeriod(),
      coupon->rate(),
      0,
      coupon->nominal(),
      DateFormatter::toString (coupon->date(), DateFormatter::ISO),
      coupon->amount(),
      leg.ccy,
      curve.ts->zeroRate(coupon->date(), Actual365Fixed(), Continuous),
      curve.ts->discount(coupon->date())));

    log.msg (NOTICE, "leg %s date %s amount %.2f discount %.4f",
             leg.id.c_str(),
	     DateFormatter::toString (coupon->date(), DateFormatter::ISO).c_str(),
	     coupon->amount(),
	     curve.ts->discount(coupon->date()));
  }

  log.msg (NOTICE, "leg %s ccy %s npv %.2f",
	   leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::pathPricerFixed (RealArray1D& EDCF,
			     RL::Leg& leg,
			     std::vector<boost::shared_ptr<CashFlow> >& cfv,
			     LiborMarketModel* lmm,
			     int path, int pmin, int pmax) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::pathPricerFixed");

  std::vector<boost::shared_ptr<CashFlow> > cashflow = cfv;
  boost::shared_ptr<FixedRateCoupon> coupon;

  double      timeDiffTolerance = 0.02; // approx. 1 week
  int         paths = pmax - pmin + 1;
  DayCounter  dc = Actual365Fixed();
  int         n  = lmm->getDimension();
  RealArray1D T  = lmm->getTenorStructure();

  double      finalDiscount;
  boost::shared_ptr<FixedRateCoupon> lastCoupon =
    boost::dynamic_pointer_cast<FixedRateCoupon>(cashflow[cashflow.size() -1]);
  if (fabs (dc.yearFraction (asof, lastCoupon->accrualEndDate()) - T[n])
      < timeDiffTolerance)
    finalDiscount = curve.ts->discount (lastCoupon->accrualEndDate());
  else
    // FIXME: error due to different daycounter used in curve construction,
    // QL problem?
    finalDiscount = curve.ts->discount (T[n]);

  if (path == 1) {
    log.msg (NOTICE, "LMM dimension n = %d", n);
    log.msg (NOTICE, "T[0] = %.4f", T[0]);
    log.msg (NOTICE, "T[n] = %.4f", T[n]);
    log.msg (NOTICE, "final discount = %.4f", finalDiscount);
  }

  // accumulate
  for (Size i = 0; i < cashflow.size(); i++) {
    coupon = boost::dynamic_pointer_cast<FixedRateCoupon>(cashflow[i]);
    double time = dc.yearFraction (asof, coupon->accrualEndDate());
    int    match = 0;

    // find matching LMM grid point, T[0] ... T[n]
    for (int t = 0; t <= n; t++) {
      if (fabs (T[t] - time) < timeDiffTolerance) {
	match++;
	// payoff, transport to horizon, discount, avergae
	EDCF[t] += lmm->H_ii(t) * coupon->amount() * finalDiscount / paths;
      }
    }

    if (match != 1)
      log.msg (ALERT, "leg %s ccy %s: coupon %s %.4f not matched, exit.",
	       leg.id.c_str(),
	       leg.ccy.c_str(),
	       DateFormatter::toString (coupon->date(),
                                        DateFormatter::ISO).c_str(),
	       time);
  }

  // final processing
  if (path == pmax) {

    leg.stats.npv = 0;
    for (Size i = 0; i < cashflow.size(); i++) {
      coupon = boost::dynamic_pointer_cast<FixedRateCoupon>(cashflow[i]);
      // cast from Handle<CashFlow> to Handle<FixedRateCoupon>
      double time     = dc.yearFraction (asof, coupon->accrualEndDate());
      double discount = curve.ts->discount(coupon->date());
      int    match    = 0;

      // find matching LMM grid point, T[0] ... T[n]
      for (int t = 0; t <= n; t++) {
	if (fabs (T[t] - time) < timeDiffTolerance) {
	  match++;
	  leg.stats.npv += EDCF[t];

	  leg.flows.append(new Flow(
              DateFormatter::toString (coupon->accrualStartDate(),
                                       DateFormatter::ISO),
              DateFormatter::toString (coupon->accrualEndDate(),
                                       DateFormatter::ISO),
              coupon->accrualPeriod(),
	      EDCF[t] / coupon->nominal() / coupon->accrualPeriod() / discount,
              0,
              coupon->nominal(),
              DateFormatter::toString (coupon->date(),
                                       DateFormatter::ISO),
              EDCF[t],
              leg.ccy,
              curve.ts->zeroRate(coupon->date(), Actual365Fixed(), Continuous),
              curve.ts->discount(coupon->date())));

	  log.msg (NOTICE, "leg %s date %s amount %.2f discount %.4f",
              leg.id.c_str(),
	      DateFormatter::toString (coupon->date(),
                                       DateFormatter::ISO).c_str(),
	      EDCF[t] / discount,
	      curve.ts->discount(coupon->date()));
	}
      }

      if (match != 1)
	log.msg (ALERT, "leg %s ccy %s: coupon %s %.4f not matched, exit.",
		 leg.id.c_str(),
		 leg.ccy.c_str(),
		 DateFormatter::toString (coupon->date(),
                                          DateFormatter::ISO).c_str(),
		 time);
    }

    log.msg (NOTICE, "leg %s ccy %s npv %.2f",
	     leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);
  }

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::pathPricerFloat (MTG::RealArray1D& EDCF,
			     RL::Leg& leg,
			     std::vector<boost::shared_ptr<CashFlow> >& cfv,
			     MTG::LiborMarketModel* lmm,
			     int path, int pmin, int pmax) {
//-----------------------------------------------------------------------------
// Build a FloatingRateCouponVector providing schedule, dates, forward rates.
// Use this to set up Volatility surface, Correlation matrix and Libor factor loadings

  static Debug log ("Engine::pathPricerFloat");

  double      timeDiffTolerance = 0.02; // approx. 1 week
  int         paths = pmax - pmin + 1;
  DayCounter  dc = Actual365Fixed();
  int         n  = lmm->getDimension();
  RealArray1D T  = lmm->getTenorStructure();

  std::vector<boost::shared_ptr<CashFlow> > cashflow = cfv;
  boost::shared_ptr<FloatingRateCoupon> coupon;
  boost::shared_ptr<FloatingRateCoupon> finalCoupon =
    boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[cashflow.size() - 1]);
  double rate, payoff=0, finalDiscount;

  boost::shared_ptr<FloatingRateCoupon> lastCoupon =
    boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[cashflow.size() - 1]);
  if (fabs (dc.yearFraction (asof, lastCoupon->accrualEndDate()) - T[n]) < timeDiffTolerance)
    finalDiscount = curve.ts->discount (lastCoupon->accrualEndDate());
  else
    // FIXME: error due to different daycounter used in curve construction, QL problem?
    finalDiscount = curve.ts->discount (T[n]);

  if (path == 1) {
    log.msg (NOTICE, "LMM dimension n = %d", n);
    log.msg (NOTICE, "T[0] = %.4f", T[0]);
    log.msg (NOTICE, "T[n] = %.4f", T[n]);
    log.msg (NOTICE, "final discount (%s) = %.4f",
	     DateFormatter::toString (finalCoupon->date(),
                                      DateFormatter::ISO).c_str(),
	     finalDiscount);
  }

  Default def( "Swap", curve.ccy(), curve.ts );

  // accumulate
  for (Size i = 0; i < cashflow.size(); i++) {
    coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[i]);
    double time = dc.yearFraction (asof, coupon->accrualStartDate()); // diff to fixed leg !
    int    match = 0;

    // find matching LMM grid point, T[0] ... T[n]
    for (int t = 0; t <= n; t++) {
      if (fabs (T[t] - time) < timeDiffTolerance) {
	match++;

	// underlying: forward rate spanning period t -> t+1
	rate = lmm->L(t,t);

	// apply formula (ignore cap/floor)
	if (leg.index->formulas.size() > 0) {
	  // find current formula in list
	  XML::list<RL::Formula>::iterator it, apply = leg.index->formulas.begin();
	  for (it = leg.index->formulas.begin(); it != leg.index->formulas.end(); it++) {
	    if (parseDate (it->date) <= coupon->accrualStartDate() &&
		parseDate (it->date) >= parseDate (apply->date))
	      apply = it;
	  }
	  // evaluate formula expression for current index
	  if ((payoff = apply->eval (rate)) == ErrVal) {
	    log.msg (ALERT, "formula: evaluation failed, exit");
	    exit(-1);
	  }
	}
	// apply cap/floor, if no formula is given
	// i.e. here: apply max, min
	else if (leg.index->isCapped == "Y" || leg.index->isFloored == "Y") {
	  if (leg.index->isCapped == "Y" &&
	      leg.index->isFloored == "Y" &&
	      leg.index->cap < leg.index->floor) {
	    log.msg (ALERT, "collar with cap strike < floor strike, exit" );
	    exit(-1);
	  }

	  if (leg.index->isCapped  == "Y")
            payoff = min (rate,   leg.index->cap);
	  if (leg.index->isFloored == "Y")
            payoff = max (payoff, leg.index->floor);
	}
	else payoff = rate;

	// payoff transported to the horizon T_n
	EDCF[t+1] += payoff * coupon->nominal() * coupon->accrualPeriod() \
	             * lmm->H_ii(t+1) * finalDiscount / paths;
      }
    }

    if (match != 1)
      log.msg (ALERT, "leg %s ccy %s: coupon %s %.4f not matched, exit.",
	       leg.id.c_str(),
	       leg.ccy.c_str(),
	       DateFormatter::toString (coupon->date(),
                                        DateFormatter::ISO).c_str(),
	       time);

  }

  // final processing
  if (path == pmax) {

    leg.stats.npv = 0;
    for (Size i = 0; i < cashflow.size(); i++) {
      coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[cashflow.size() - 1]);
      double time     = dc.yearFraction (asof, coupon->accrualEndDate());
      double discount = curve.ts->discount(coupon->date());
      int    match    = 0;

      // find matching LMM grid point, T[0] ... T[n]
      for (int t = 0; t <= n; t++) {
	if (fabs (T[t] - time) < timeDiffTolerance) {
	  match++;
	  leg.stats.npv += EDCF[t];

	  leg.flows.append(new Flow(
	      DateFormatter::toString (coupon->accrualStartDate(),
                                       DateFormatter::ISO),
              DateFormatter::toString (coupon->accrualEndDate(),
                                       DateFormatter::ISO),
              coupon->accrualPeriod(),
              EDCF[t] / coupon->nominal() / coupon->accrualPeriod() / discount,
              coupon->spread(),
              coupon->nominal(),
              DateFormatter::toString (coupon->date(), DateFormatter::ISO),
              EDCF[t],
              leg.ccy,
              curve.ts->zeroRate(coupon->date(), Actual365Fixed(), Continuous),
              discount));

	  log.msg (NOTICE, "leg %s date %s amt %.2f (%.6f) dis %.4f",
	      leg.id.c_str(),
	      DateFormatter::toString (coupon->date(),
                                       DateFormatter::ISO).c_str(),
	      EDCF[t] / discount,
	      EDCF[t] / coupon->nominal() / coupon->accrualPeriod() / discount,
		   discount);
	}
      }

      if (match != 1)
	log.msg (ALERT, "leg %s ccy %s: coupon %s %.4f not matched, exit.",
		 leg.id.c_str(),
		 leg.ccy.c_str(),
		 DateFormatter::toString (coupon->date(),
                                          DateFormatter::ISO).c_str(),
		 time);
    }

    log.msg (NOTICE, "leg %s ccy %s npv %.2f (using average future rates)",
	     leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);
  }

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::priceFixedLegLMM (RL::Leg& leg, LiborMarketModel* lmm) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::priceFixedLegLMM");

  double timeDiffTolerance = 0.02; // approx. 1 week

  leg.stats.npv = 0;

  std::vector<boost::shared_ptr<CashFlow> > cashflow = cashFlowVector (leg);
  boost::shared_ptr<FixedRateCoupon> coupon;

  DayCounter  dc = Actual365Fixed();
  int         n  = lmm->getDimension();
  RealArray1D T  = lmm->getTenorStructure();
  int         paths = 1000; // number of simulated paths, default overridden by environment below

  double      finalDiscount;
  boost::shared_ptr<FixedRateCoupon> lastCoupon
    = boost::dynamic_pointer_cast<FixedRateCoupon>(cashflow[ cashflow.size() - 1 ]);
  if (fabs (dc.yearFraction (asof, lastCoupon->accrualEndDate()) - T[n]) < timeDiffTolerance)
    finalDiscount = curve.ts->discount (lastCoupon->accrualEndDate());
  else
    // FIXME: error due to different daycounter used in curve construction, QL problem?
    finalDiscount = curve.ts->discount (T[n]);

  RealArray1D EDCF(n+1); // expected discounted cashflow, n+1 like T grid points
  RealArray1D ER(n+1);   // expected rate

  log.msg (NOTICE, "LMM dimension n = %d", n);
  log.msg (NOTICE, "T[0] = %.4f", T[0]);
  log.msg (NOTICE, "T[n] = %.4f", T[n]);
  log.msg (NOTICE, "final discount = %.4f", finalDiscount);

  char *e;
  if ((e=getenv("ENV_LMM_PATHS")))
    paths = atoi(e);
  else
    log.msg (ERROR, "ENV_LMM_PATHS not set, using default %d", paths);

  for (int t = 0; t < n; t++) EDCF[t] = ER[t] = 0;

  for (int p = 1; p <= paths; p++) {
    printf ("paths %5d %3.0f%%\r", p, 100.0*(p+1)/paths);
    fflush(stdout);

    lmm->newPath();

    for (Size i = 0; i < cashflow.size(); i++) {
      coupon = boost::dynamic_pointer_cast<FixedRateCoupon>(cashflow[i]); // cast from Handle<CashFlow> to Handle<FixedRateCoupon>
      double time = dc.yearFraction (asof, coupon->accrualEndDate());

      // find matching LMM grid point, T[0] ... T[n]
      for (int t = 0; t <= n; t++) {
	if (fabs (T[t] - time) < timeDiffTolerance)
	  EDCF[t] += lmm->H_ii(t) * coupon->amount() * finalDiscount / paths;
      }
    }
  }

  printf ("\n");

  leg.stats.npv = 0;
  for (Size i = 0; i < cashflow.size(); i++) {
    coupon = boost::dynamic_pointer_cast<FixedRateCoupon>(cashflow[i]); // cast from Handle<CashFlow> to Handle<FixedRateCoupon>
    double time     = dc.yearFraction (asof, coupon->accrualEndDate());
    double discount = curve.ts->discount(coupon->date());
    int    match    = 0;

    // find matching LMM grid point, T[0] ... T[n]
    for (int t = 0; t <= n; t++) {
      if (fabs (T[t] - time) < timeDiffTolerance) {
	match++;
	leg.stats.npv += EDCF[t];

	ER[t] = EDCF[t] / coupon->nominal() / coupon->accrualPeriod() / discount;

	leg.flows.append(new Flow(
          DateFormatter::toString (coupon->accrualStartDate(),
                                   DateFormatter::ISO),
          DateFormatter::toString (coupon->accrualEndDate(),
                                   DateFormatter::ISO),
          coupon->accrualPeriod(),
          ER[t],
          0,
          coupon->nominal(),
          DateFormatter::toString (coupon->date(), DateFormatter::ISO),
          EDCF[t],
          leg.ccy,
          curve.ts->zeroRate(coupon->date(), Actual365Fixed(), Continuous),
          curve.ts->discount(coupon->date())));

	log.msg (NOTICE, "leg %s date %s amount %.2f discount %.4f",
		 leg.id.c_str(),
		 DateFormatter::toString (coupon->date(),
                                          DateFormatter::ISO).c_str(),
		 ER[t] * coupon->nominal() * coupon->accrualPeriod(),
		 curve.ts->discount(coupon->date()));
      }
    }

    if (match != 1)
      log.msg (ALERT, "leg %s ccy %s: coupon %s %.4f not matched, exit.",
	       leg.id.c_str(),
	       leg.ccy.c_str(),
	       DateFormatter::toString (coupon->date(),
                                        DateFormatter::ISO).c_str(),
	       time);
  }

  log.msg (NOTICE, "leg %s ccy %s npv %.2f",
	   leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::priceFloatLeg (RL::Leg& leg) {
//-----------------------------------------------------------------------------
  static Debug log ("Engine::priceFloatLeg");

  if (leg.index->formulas.size() > 0 &&
      (leg.index->isCapped == "Y" || leg.index->isFloored == "Y"))
    log.msg (ERROR, "leg %s ccy %s: caps/floors are ignored/overridden by formula",
	     leg.id.c_str(), leg.ccy.c_str());

  std::vector<boost::shared_ptr<CashFlow> > cashflow = cashFlowVector (leg);
  boost::shared_ptr<FloatingRateCoupon> coupon;

  // needed for caplet / floorlet pricing in QL
  boost::shared_ptr<Xibor> index = mapIndex (leg.index->ccy, leg.index->name,
                                             leg.index->term, curve.ts);
  QL::Calendar  calendar = mapCalendar (leg.index->resetSched.calendar);
  double        nominal  = leg.notional.begin()->amt; // FIXME

  double rate=0, modified, capModified, floorModified;
  double npv = 0;
  double capPrice = 0;
  double floorPrice = 0;
  int    capPriceSign, floorPriceSign;

  if (nominal < 0) { // FIXME
    capPriceSign   = +1;
    floorPriceSign = -1;
  }
  else {
    capPriceSign   = -1;
    floorPriceSign = +1;
  }

  for (Size i=0; i<cashflow.size(); i++) {
    coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[i]);

    try {
      leg.stats.npv += coupon->amount()
                           * curve.ts->discount(coupon->date());
    }
    CATCH_ALERT;

    // retrieves the (past) fixing or (projected) forward rate
    try {
      rate = coupon->rate();
    }
    CATCH_ALERT;

    // apply formula (ignore cap/floor)

    if (leg.index->formulas.size() > 0) {
      // find current formula in list
      XML::list<RL::Formula>::iterator it, apply = leg.index->formulas.begin();
      for (it = leg.index->formulas.begin(); it != leg.index->formulas.end(); it++) {
	if (parseDate (it->date) <= coupon->accrualStartDate() &&
	    parseDate (it->date) >= parseDate (apply->date))
	  apply = it;
      }
      // evaluate formula expression for current index
      if ((modified = apply->eval (rate)) == ErrVal) {
	log.msg (ALERT, "formula: evaluation failed, exit");
	exit(-1);
	modified = rate;
      }
      else {
	log.msg (NOTICE, "formula (%s): index modified from %.4f to %.4f",
		 apply->date.c_str(), rate, modified );
      }
    }
    else {
      // apply cap/floor, if no formula is given
      // price individual caplets/floorlets (Black-Scholes)
      // adjust forward rate using BS prices such that
      // sum of discounted cash flows = price of floater + cap + floor

      if (leg.index->isCapped == "Y" &&
	  leg.index->isFloored == "Y" &&
	  leg.index->cap < leg.index->floor) {
	log.msg (ALERT, "collar with cap strike < floor strike, exit" );
	exit(-1);
      }
      capModified = rate;
      floorModified = rate;

      if (leg.index->isCapped == "Y") {
	double price = curve.capletPrice (CapFloor::Cap,
					  coupon->accrualStartDate(),
					  coupon->accrualEndDate(),
					  calendar,
					  nominal,
					  leg.index->cap,
					  index);

	capModified = rate - price
	  / coupon->nominal()
	  / coupon->accrualPeriod()
	  / curve.ts->discount(coupon->date());

	capPrice += price * capPriceSign;

 	if (fabs((capModified-rate)/rate) > 1e-4)
 	  log.msg (DEBUG, "apply cap: rate %.6f -> %.6f, price %.4f",
 		   rate, capModified, price);
      }

      if (leg.index->isFloored == "Y") {
	double price = curve.capletPrice (CapFloor::Floor,
					  coupon->accrualStartDate(),
					  coupon->accrualEndDate(),
					  calendar,
					  nominal,
					  leg.index->floor,
					  index);
	floorModified = rate + price
	  / coupon->nominal()
	  / coupon->accrualPeriod()
	  / curve.ts->discount(coupon->date());

	floorPrice += price * floorPriceSign;

 	if (fabs((floorModified-rate)/rate) > 1e-4)
 	  log.msg (DEBUG, "apply floor: rate %.6f -> %.6f, price %.4f",
 		   rate, floorModified, price);
      }

      modified = capModified + floorModified - rate;

      if (fabs((modified-rate)/rate) > 1e-4)
	log.msg (DEBUG, "apply capfloor: %.6f -> %.6f", rate, modified);
    }

    leg.flows.append(new Flow(
	DateFormatter::toString (coupon->accrualStartDate(),
                                 DateFormatter::ISO),
        DateFormatter::toString (coupon->accrualEndDate(), DateFormatter::ISO),
        coupon->accrualPeriod(),
        modified,
        coupon->spread(),
        coupon->nominal(),
        DateFormatter::toString (coupon->date(), DateFormatter::ISO),
        coupon->amount() * modified/rate,
        leg.ccy,
        curve.ts->zeroRate(coupon->date(), Actual365Fixed(), Continuous),
        curve.ts->discount(coupon->date())));

    log.msg (NOTICE, "leg %s date %s amt %.2f (%.6f -> %.6f) dis %.4f",
	     leg.id.c_str(),
	     DateFormatter::toString (coupon->date(),
                                      DateFormatter::ISO).c_str(),
	     coupon->amount() * modified/rate,
	     rate,
	     modified,
	     curve.ts->discount(coupon->date()));

    npv += modified / rate * coupon->amount() * curve.ts->discount(coupon->date());

    double amount = coupon->nominal()
      * coupon->rate()
      * coupon->accrualPeriod();

    if (fabs ((amount - coupon->amount()) / amount) > 1e-6)
      log.msg (ALERT, "leg %s: coupon amount inconsistency, exit.", leg.id.c_str());
  }

  // price of the unadjusted floater

  log.msg (NOTICE, "leg %s ccy %s npv %.2f (ignoring formula and cap/floor)",
	   leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);

  // re-derive price using modified rates and adjusted cash flows in flow list

  double check = 0;
  for (XML::list<RL::Flow>::iterator i = leg.flows.begin(); i != leg.flows.end(); i++) {
    check += i->amount * i->discount;
  }

  // check consistency 1: compare the latter to the npv calculated during construction

  if (fabs ((npv - check) / npv) > 1e-6)
    log.msg (ALERT, "leg %s ccy %s, accuracy problem in npv calculation, exit.",
	     leg.id.c_str(), leg.ccy.c_str());

  // check consistency 2: floater + cap + floor vs. discounted adjusted flows

  if (leg.index->formulas.size() == 0 &&
      (leg.index->isCapped == "Y" || leg.index->isFloored == "Y")) {

    log.msg (NOTICE, "cap price   %+.4f", capPrice);
    log.msg (NOTICE, "floor price %+.4f", floorPrice);

    leg.stats.npv += capPrice + floorPrice;

    log.msg (NOTICE, "leg %s ccy %s npv %.2f (floater + cap + floor)",
	     leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);

    if (fabs ((npv - leg.stats.npv) / npv) > 1e-6)
      log.msg (ALERT, "leg %s ccy %s, inconsistent capped/floored floater price, exit.",
	       leg.id.c_str(), leg.ccy.c_str());
  }

  leg.stats.npv = npv;
  log.msg (NOTICE, "leg %s ccy %s npv %.2f (using adjusted flows)",
	   leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::priceFloatLegLMM (RL::Leg& leg, MTG::LiborMarketModel* lmm) {
//-----------------------------------------------------------------------------
// Build a FloatingRateCouponVector providing schedule, dates, forward rates.
// Use this to set up Volatility surface, Correlation matrix and Libor factor loadings

  static Debug log ("Engine::priceFloatLegLMM");

  Default def( "Swap", curve.ccy(), curve.ts );

  if (leg.index->formulas.size() > 0 &&
      (leg.index->isCapped == "Y" || leg.index->isFloored == "Y"))
    log.msg (ERROR, "leg %s ccy %s: caps/floors are ignored/overridden by formula",
	     leg.id.c_str(), leg.ccy.c_str());

  std::vector<boost::shared_ptr<CashFlow> > cashflow = cashFlowVector (leg);

  boost::shared_ptr<FloatingRateCoupon> coupon;
  boost::shared_ptr<FloatingRateCoupon> finalCoupon =
    boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[cashflow.size() - 1]);
   double discount, finalDiscount = curve.ts->discount (finalCoupon->date());

  log.msg (NOTICE, "final discount %s %.6f",
	   DateFormatter::toString (finalCoupon->date(),
                                    DateFormatter::ISO).c_str(),
	   finalDiscount);

  int paths = 10; // number of simulated paths, default
  int print = 0;  // do not print, default
  int test_fwd = 0;      // perform consistency check forwards vs. expected future rates, default
  int test_caplet = 0;   // test caplet prices
  int test_swaption = 0; // test swaption prices

  char *e;
  if ((e=getenv("ENV_LMM_PATHS")))
    paths = atoi(e);
  else
    log.msg (ERROR, "ENV_LMM_PATHS not set, using default %d", paths);

  if ((e=getenv("ENV_LMM_PRINT")))
    print = atoi(e);
  else
    log.msg (ERROR, "ENV_LMM_PRINT not set, using default %d", print);

  if ((e=getenv("ENV_LMM_TEST_FWD")))
    test_fwd = atoi(e);
  else
    log.msg (ERROR, "ENV_LMM_TEST_FWD not set, using default %d", test_fwd);

  if ((e=getenv("ENV_LMM_TEST_CAPLET")))
    test_caplet = atoi(e);
  else
    log.msg (ERROR, "ENV_LMM_TEST_CAPLET not set, using default %d", test_caplet);

  if ((e=getenv("ENV_LMM_TEST_SWAPTION")))
    test_swaption = atoi(e);
  else
    log.msg (ERROR, "ENV_LMM_TEST_SWAPTION not set, using default %d", test_swaption);

  int n = lmm->getDimension();

  // tests ////////////////////////////////////////////////////////////////////
  if (test_fwd) {
    log.msg (NOTICE, "testing forwards against expected future rates");

    double      error=0;
    RealArray1D EL(n), L1(n), L2(n), L3(n);

    for (int l = 0; l < n; l++) EL[l] = 0;

    for (int p = 1; p <= paths; p++) {
      printf ("paths %5d %3.0f%%\r", p, 100.0*(p+1)/paths);
      fflush(stdout);

      lmm->newPath();

      if (print) {
	printf( "p=%2d   ", p );
	for (int l = 0; l < n; l++) printf( "%5d ", l );
	printf( "\n" );
      }

      for (int t = 0; t < n; t++) {
	coupon   = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[t]);
	discount = curve.ts->discount (coupon->date());

	double increment = lmm->L(t,t) * lmm->H_ii(t+1) * finalDiscount / discount / paths;

	EL[t] += increment;
	// sample forward curves
	if (p == 1)      L1[t] = lmm->L(t,t);
	else if (p == 2) L2[t] = lmm->L(t,t);
	else if (p == 3) L3[t] = lmm->L(t,t);

	if (print) {
	  printf( "  t=%2d ", t );
	  for (int l = 0; l < t; l++)   printf( "----- " );
	  for (int l = t; l < n-1; l++) printf( "%5.2f ", 100.0 * lmm->L(l,t));
	  printf( "%5.2f\n", 100.0 * lmm->L(n-1,t) );
	}
      }
    }
    printf ("\n");

    // check expectation against current forward
    for (int t = 0; t < n; t++) error += pow ((lmm->L(t,0) - EL[t]) / lmm->L(t,0), 2);
    error = sqrt (error / n);

    FILE* f=fopen ("libor.txt", "w");
    fprintf (f, "# paths: %d\n", paths);
    fprintf (f, "# error: %.4f percent\n", error * 100.0);
    for (int t = 0; t < n; t++)
      fprintf (f, "%d %.6f %.6f %.6f %.6f %.6f %.6f\n",
	       t, lmm->L(t,0), EL[t], fabs(lmm->L(t,0) - EL[t]) / lmm->L(t,0),
	       L1[t], L2[t], L3[t]);

    fclose (f);
  } // end of test

  // pricing //////////////////////////////////////////////////////////////////
  //  log.msg (NOTICE, "pricing");

  RealArray1D ER(n);   // expected rate
  RealArray1D EDCF(n); // expected discounted cashflow
  RealArray1D DF(n);   // expected discount factor

  int bins=100;
  RealArray1D R(bins+1), M(bins+1);
  double rMax = 0.1;
  int bin;

  // test analytical and monte carlo caplet prices
  if (test_caplet) {
    if (leg.index->isCapped  == "Y") {
      double s_mc=0, s_an=0;
      for (int t = 1; t < n; t++) {
	coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[t]);
	printf( "t = %d\n", t ); fflush(stdout);

	MTG::Caplet caplet (t, leg.index->cap, lmm);
	double mc = caplet.monteCarloForwardPrice(paths) * finalDiscount * coupon->nominal();
	double an = caplet.analyticForwardPrice() * finalDiscount * coupon->nominal();
	s_mc += mc;
	s_an += an;

	log.msg (NOTICE, "cap price %d %.6f %.6f %.6f %.6f", t, mc, an, s_mc, s_an);
      }
    }
  }

  if (test_swaption) {
    // test analytical and monte carlo european swaption prices
    for (int t = 4; t <= 10; t+=2) {
      coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[t]);
      Period period ((n - t) * 12 / def.floatFreq, Months);
      double strike = 0.05;

      MTG::Swaption swaption(t, n, t, strike, lmm);

      try {
	double vol = 0.01*curve.volMatrix->volatility (coupon->accrualStartDate(), period, 0.0);
	double an = swaption.analyticForwardPrice() * finalDiscount * coupon->nominal();
	double mc = swaption.monteCarloForwardPrice(paths) * finalDiscount * coupon->nominal();
	boost::shared_ptr<SimpleSwap> swap =
          makeSimpleSwap (coupon->accrualStartDate(),
                          period.length(), period.units(),
                          coupon->nominal(), strike, 0,
                          true, "EUR");
	boost::shared_ptr<QL::Swaption> qlswaption =
          makeSwaption (swap,
                        coupon->accrualStartDate(),
                        vol);
	log.msg (NOTICE, "Swaption p=%2d q=%2d t=p AN %.2f MC %.2f QL %.2f",
		 t, n, an, mc, qlswaption->NPV());
      }
      CATCH_ALERT;
    }
  }

  for (int t = 0; t < n; t++) ER[t] = EDCF[t] = 0;

  for (int i = 0; i <= bins; i++) R[i] = M[i] = 0;

  double rate = 0, payoff = 0;
  for (int p = 1; p <= paths; p++) {
    printf ("paths %5d %3.0f%%\r", p, 100.0*(p+1)/paths);
    fflush(stdout);

    lmm->newPath();

    for (int t = 0; t < n; t++) {
      // n == cashflow.size() !
      coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[t]);

      // underlying: forward rate spanning period t -> t+1
      rate = lmm->L(t,t);

      // collect historam for the forward rate distribution at half the horizon
      if (t == 0.5*n) {
	if (rate > rMax) R[bins]++;
	else {
	  bin = (int) floor (rate / rMax * bins);
	  R[bin] ++;
	}
      }

      // apply formula (ignore cap/floor)
      if (leg.index->formulas.size() > 0) {
	// find current formula in list
	XML::list<RL::Formula>::iterator it, apply = leg.index->formulas.begin();
	for (it = leg.index->formulas.begin(); it != leg.index->formulas.end(); it++) {
	  if (parseDate (it->date) <= coupon->accrualStartDate() &&
	      parseDate (it->date) >= parseDate (apply->date))
	    apply = it;
	}
	// evaluate formula expression for current index
	if ((payoff = apply->eval (rate)) == ErrVal) {
	  log.msg (ALERT, "formula: evaluation failed, exit");
	  exit(-1);
	}
      }
      // apply cap/floor, if no formula is given
      // i.e. here: apply max, min
      else if (leg.index->isCapped == "Y" || leg.index->isFloored == "Y") {
	if (leg.index->isCapped == "Y" &&
	    leg.index->isFloored == "Y" &&
	    leg.index->cap < leg.index->floor) {
	  log.msg (ALERT, "collar with cap strike < floor strike, exit" );
	  exit(-1);
	}

	if (leg.index->isCapped  == "Y") payoff = min (rate,   leg.index->cap);
	if (leg.index->isFloored == "Y") payoff = max (payoff, leg.index->floor);
      }
      else payoff = rate;

      // collect historam for the payoff distribution at half the horizon
      if (t == 0.5*n) {
	if (payoff > rMax) M[bins]++;
	else {
	  bin = (int) floor (payoff / rMax * bins);
	  M[bin] ++;
	}
      }

      ER[t] += payoff;

      // discounted future cashflow (rate is multiplied below with nominal and accrualPeriod)
      // FIXME: check carefully discount factor at T[0]

      // failed trials ...
      //EDCF[t] += lmm->B0(t+1) * rate;
      //EDCF[t] += discount * rate;

      // forward transporting factor, take care at the horizon (H_ii=0 !),
      // bug ? fixed in martingale
      // double f = lmm->H_ii(t+1);
      // if (t < n-1) f = lmm->H_ii(t+1);
      // else         f = 1.0;

      // payoff transported to the horizon T_n
      EDCF[t] += lmm->H_ii(t+1) * payoff;
    }
  }

  printf ("\n");

  leg.stats.npv = 0;
  for (int t = 0; t < n; t++) {
    // n == cashflow.size() !
    coupon = boost::dynamic_pointer_cast<FloatingRateCoupon>(cashflow[t]);

    // normalise => expected future rate
    ER[t]   /= paths;

    // apply nominal and accrual period, discount from horizon T_n to t=0, normalise
    // => expected discounted payoff
    EDCF[t] *= finalDiscount * coupon->nominal() * coupon->accrualPeriod();
    EDCF[t] /= paths;

    leg.stats.npv += EDCF[t];

    leg.flows.append(new Flow(
	DateFormatter::toString (coupon->accrualStartDate(),
                                 DateFormatter::ISO),
        DateFormatter::toString (coupon->accrualEndDate(), DateFormatter::ISO),
        coupon->accrualPeriod(),
        ER[t],
        coupon->spread(),
        coupon->nominal(),
        DateFormatter::toString (coupon->date(), DateFormatter::ISO),
        EDCF[t] / curve.ts->discount(coupon->date()),
        leg.ccy,
        curve.ts->zeroRate(coupon->date(), Actual365Fixed(), Continuous),
        curve.ts->discount(coupon->date())));

    log.msg (NOTICE, "leg %s date %s amt %.2f (%.6f) dis %.4f",
	     leg.id.c_str(),
	     DateFormatter::toString (coupon->date(),
                                      DateFormatter::ISO).c_str(),
	     EDCF[t] / curve.ts->discount(coupon->date()),
	     ER[t],
	     curve.ts->discount(coupon->date()));
  }

  char name[50];
  sprintf (name, "dist.%s", leg.id.c_str());
  FILE* f = fopen (name, "w");
  for (int i=0; i<=bins; i++)
    fprintf (f, "%d %.4f %.6f %.6f\n", i, rMax * i / bins, R[i]/paths, M[i]/paths);
  fclose (f);

  log.msg (NOTICE, "leg %s ccy %s npv %.2f (using average future rates)",
	   leg.id.c_str(), leg.ccy.c_str(), leg.stats.npv);

  return 0;
}

double Engine::max(double a, double b) { return a > b ? a : b; }
double Engine::min(double a, double b) { return a < b ? a : b; }
int    Engine::max(int a, int b)       { return a > b ? a : b; }
int    Engine::min(int a, int b)       { return a < b ? a : b; }
double Engine::fak(int n)              { return n <= 0 ? 1.0 : fak(n-1)*n; }
double Engine::binko(int n, int k)     { return fak(n) / (fak(k) * fak(n-k)); }

//-----------------------------------------------------------------------------
int Engine::binomialPrice (double *res, int type,
                           double flt, double fix, double vol,
                           double mat, int iMat, double *expect,
                           int iTime, int iNode) {
//-----------------------------------------------------------------------------

  // CoxRossRubinstein / JarrowRudd "model"
  // i.e. Black Scholes model on a recombining binomial tree
  // applied to european swaptions
  // allowing modified payoff at maturity: max ( intrinsic, expected )

  // type   - 0: CRR, 1: JR
  // flt    - present value of the floating leg of the underlying (fwd) swap
  // fix    - present value of the fixed    leg of the underlying (fwd) swap
  // vol    - black scholes volatility for the given european swaption
  // mat    - time from valuation date to expiry of the swaption (1 year = 1.0)
  // iMat   - total number of time steps for the lattice until maturity
  // expect - vector of expected value of future exercises
  //          for each node at maturity (dimension steps+1)
  //          if set to NULL, zero expectation or final maturity is assumed
  // iTime  - time step at which the rolled-back price is to be determined
  //          (0 ... iMat)
  // iNode  - node number at time step iTime (0 ... iTime)

  static Debug log( "binomialPrice" );

  *res = 0;

  if ( flt * fix > 0 ) {
    log.msg("flt and fix should have opposite signs");
    exit( 1 );
  }

  if ( iTime > iMat || iTime < 0 ) {
    log.msg( "0 <= iTime %d <= iMat %d required", iTime, iMat );
    exit( 1 );
  }

  if ( iNode > iTime || iNode < 0 ) {
    log.msg( "0 <= iNode %d <= iTime %d required", iNode, iTime );
    exit( 1 );
  }

  double u, d; // multiplies float leg pv during up or down move
  double q, p; // probabilities for up or down move
  double dt = mat / iMat;

  if (type == 0) {               // CoxRossRubinstein
    u = exp( + vol * sqrt( dt ) );
    d = exp( - vol * sqrt( dt ) );
    q = ( 1.0 - d ) / ( u - d );
    p = 1.0 - q;
  }
  else if ( type == 1 ) {        // JarrowRudd
    u = exp( + vol * sqrt( dt ) - 0.5 * vol*vol * dt );
    d = exp( - vol * sqrt( dt ) - 0.5 * vol*vol * dt );
    p = q = 0.5;
  }
  else {
    log.msg("type 0 or 1 required");
    return 1;
  }

  FILE   *f = NULL;
  double  weight, intrinsic, sum=0, price=0;

  if (getenv("DEBUG_BINTREE")) {
    f = fopen("bintree.txt","a");
    if (!f) log.msg("error opening file");
  }

  for ( int j = iTime; j <= iMat; j++ ) {
    weight = binko(iMat-iTime,j-iTime) * pow(q,j-iTime) * pow(p,iMat-j);
    if (!expect)
      intrinsic = max( pow(u,j-iTime+iNode) * pow(d,iMat-iNode-(j-iTime)) * flt + fix, 0.0 );
    else
      intrinsic  = max( pow(u,j-iTime+iNode) * pow(d,iMat-iNode-(j-iTime)) * flt + fix, expect[j-iTime+iNode] );
    price     += weight * intrinsic;
    sum       += weight;

    if (f)
      fprintf( f, "%d  %.8f  %.8f  %.8f  %.8f\n",
               j, weight, sum, intrinsic, price );
  }

  if (f) fclose(f);

  *res = price;

  return 0;
}

//-----------------------------------------------------------------------------
boost::shared_ptr<SimpleSwap>
Engine::makeSimpleSwap(const QL::Date& start, int length, QL::TimeUnit unit,
		       double nominal,
		       QL::Rate fixedRate,
		       QL::Spread floatingSpread,
		       bool payFixed,
		       const char * ccy,
		       const char * name ) {
//-----------------------------------------------------------------------------
  static Debug d( "Trade::makeSimpleSwap" );

  Default def( "Swap", ccy, curve.ts );

  QL::Date maturity = def.calendar.advance (start, length, unit, def.roll);

  QL::Schedule fixedSchedule (def.calendar, start, maturity,
                              def.fixedFreq, def.fixedRoll);
  QL::Schedule floatSchedule (def.calendar, start, maturity,
                              def.floatFreq, def.roll);

  return boost::shared_ptr<SimpleSwap>(new SimpleSwap(
                       payFixed,
                       nominal,
                       fixedSchedule, fixedRate, def.cmDayCount,
                       floatSchedule, def.index, def.fixingDays,
                       floatingSpread,
                       curve.ts));
}

//-----------------------------------------------------------------------------
boost::shared_ptr<SimpleSwap>
Engine::makeSimpleAtmSwap (const QL::Date& start, int length,
                           QL::TimeUnit units,
                           QL::Spread floatingSpread,
                           bool payFixed,
                           const char * ccy,
                           const char * name) {
//-----------------------------------------------------------------------------
  static Debug d( "Trade::makeSimpleAtmSwap" );

  Default def( "Swap", ccy, curve.ts );

  QL::Date maturity = def.calendar.advance (start, length, units, def.roll);

  QL::Schedule fixedSchedule (def.calendar, start, maturity,
                              def.fixedFreq, def.fixedRoll);
  QL::Schedule floatSchedule (def.calendar, start, maturity,
                              def.floatFreq, def.roll);

  QL::Rate fixedRate = 0.05;

  boost::shared_ptr<SimpleSwap> swap =
    boost::shared_ptr<SimpleSwap>(new SimpleSwap(
                       payFixed,
                       def.nominal,
                       fixedSchedule, fixedRate, def.cmDayCount,
                       floatSchedule, def.index, def.fixingDays,
                       floatingSpread,
                       curve.ts));

  QL::Rate fairRate = swap->fairRate();

  d.msg( DEBUG, "fair rate %.6f", fairRate * 100 );

  return boost::shared_ptr<SimpleSwap>(new SimpleSwap(
                       payFixed,
                       def.nominal,
                       fixedSchedule, fairRate, def.cmDayCount,
                       floatSchedule, def.index, def.fixingDays,
                       floatingSpread,
                       curve.ts));
}

//-----------------------------------------------------------------------------
boost::shared_ptr<QL::Swaption> Engine::makeSwaption(
  const boost::shared_ptr<SimpleSwap>& swap,
  const QL::Date& exercise,
  double volatility) {
//-----------------------------------------------------------------------------
  static Debug d( "Trade::makeSwaption" );

  // volatility should be amendable after creating the swaption
  // such that the instrument can be repriced based on that volatility
  // FIXME
  boost::shared_ptr<Quote> vol_me(new SimpleQuote(volatility));
  Handle<Quote> vol_rh(vol_me);
  boost::shared_ptr<BlackModel> model(new BlackModel(vol_rh,curve.ts));
  boost::shared_ptr<PricingEngine> engine(new BlackSwaptionEngine(model));

  boost::shared_ptr<QL::Swaption> swaption = boost::shared_ptr<QL::Swaption>(new QL::Swaption(
                       swap,
		       boost::shared_ptr<Exercise>(new EuropeanExercise(exercise)),
                       curve.ts,
                       engine));

 d.msg (DEBUG, "swap %.2f swaption %.2f vol %.4f", swap->NPV(), swaption->NPV(), volatility );

 return swaption;
}

//-----------------------------------------------------------------------------
double Engine::getBinomialEuropeanSwaptionPrice (
  int type,
  const boost::shared_ptr<SimpleSwap>& swap,
  const QL::Date& exercise,
  double volatility,
  int    steps ) {
//-----------------------------------------------------------------------------
  static Debug d("Trade::getBinomialSwaptionPrice");

  double npv    = 0;
  int    steps_ = 100; // default

  if (steps) steps_ = steps;

//   d.msg( DEBUG, "try swap valuation" );
//   d.msg( DEBUG, "fix/flt npv = %.2f %.2f %.2f",
// 	 swap->floatingLegNPV(), swap->fixedLegNPV(), swap->NPV() );
//   d.msg( DEBUG, "1st/2nd npv = %.2f %.2f %.2f",
// 	 swap->firstLegNPV(), swap->secondLegNPV(), swap->NPV() );

  // The following is commented out for the time being because QuantLib's swap
  // does not provide separate floatingLeg and fixedLeg (or firstLeg and
  // secondLeg) NPVs.
  // I have added related members and functions to my local QuantLib version,
  // however, this distribution of MTM should rely on standard QuantLib
  // functionality only.
  // Todo: Submit a patch to QuantLib for this purpose.

/*
  if ( Engine::binomialPrice(&npv, type,
		       swap->floatingLegNPV(), swap->fixedLegNPV(), volatility,
		       1.0*(exercise-asof)/365, steps_, NULL, 0, 0 ) ) {
    d.msg(ERROR, "error in binomialPrice");
    return 0;
  }
*/
  return npv;
}

//-----------------------------------------------------------------------------
double Engine::getBinomialBermudanSwaptionPrice(
  int type,
  std::vector<boost::shared_ptr<SimpleSwap> >& swaps,
  std::vector<QL::Date> & exercises,
  std::vector<double> vols,
  int    steps,
  Default &def ) {
//-----------------------------------------------------------------------------

  static Debug d("Trades::getBinomialBermudanSwaptionPrice");

  //  Default def( "Swap", def.ccy, curve->ts );
  //  fprintf( pof, "\t\t<EmbeddedEuropeanSwaptions>\n" );

  for( unsigned int i=0; i<exercises.size(); i++ ) {
    boost::shared_ptr<QL::Swaption> swaption = makeSwaption (swaps[i], exercises[i], vols[i]);

    d.msg(DEBUG, "Bermudan Exp %s Mat %s vol %.2f FwdSwap %.2f European %.2f",
	  DateFormatter::toString( exercises[i], DateFormatter::ISO ).c_str(),
	  DateFormatter::toString( swaps[i]->maturity(), DateFormatter::ISO ).c_str(),
	  vols[i]*100,
	  swaps[i]->NPV(),
	  swaption->NPV() );
  }

  double tMax = def.actDayCount.yearFraction(asof,exercises[exercises.size()-1]);
  double dt = 0.1;
  int    matSteps; //  = floor( tMax / dt + 0.5 );
  int    evalSteps;

  //  if (steps) steps_ = steps;

  int     maxNodes = (int) floor( tMax / dt + 0.5 ); // exercises.size() * steps_ + 1;
  double  timeMat, timeEval;
  double *pExpected = NULL;
  double *expected = new double [ maxNodes ];
  double *node     = new double [ maxNodes ];

  for(int i=0; i<maxNodes; i++ ) {
    expected[i]=0;
    node[i] = 0;
  }

  d.msg(DEBUG, "tMax %.2f  dt %.2f  nodes %d", tMax, dt, maxNodes );

  for( unsigned int i=exercises.size()-1; i>=0; i-- ) {
    timeMat = def.actDayCount.yearFraction( asof, exercises[i] );
    if (i>0) timeEval = def.actDayCount.yearFraction( asof, exercises[i-1] );
    else     timeEval = 0;

    matSteps  = (int) floor( timeMat / dt + 0.5 );
    evalSteps = (int) floor( timeEval / dt + 0.5 );

    d.msg(DEBUG,  "current timeMat %.2f timeEval %.2f  matSteps %d  evalSteps %d",
	  timeMat, timeEval, matSteps, evalSteps );

    if ( i==exercises.size()-1 ) pExpected = NULL;
    else                         pExpected = expected;// + j;

    //    for( int j=0; j<=i*steps_; j++ ) {
    for( int j=0; j<=evalSteps; j++ ) {
      /*
      Engine::binomialPrice( &(node[j]), 0,
			   swaps[i]->floatingLegNPV(),
			   swaps[i]->fixedLegNPV(),
			   vols[i],
			   timeMat,
			   matSteps,
			   pExpected,
			   //		     i*steps_,
			   evalSteps,
			   j);
      */
      d.msg(DEBUG,  "i=%d node[%d]=%.2f", i, j, node[j] );
    }
    for( int j=0; j<maxNodes; j++ ) { expected[j] = node[j]; node[j]=0; }
  }

  return expected[0];
}

// ----------------------------------------------------------------------------
int Engine::testCurve () {
// ----------------------------------------------------------------------------
  static Debug log("testCurve");

  try {

    log.msg( NOTICE, "--- test curve ---" );

    QL::Calendar cal = mapCalendar( "FRA" );

    log.msg( NOTICE, "cal %s", cal.name().c_str() );

    log.msg( NOTICE, "link to flat curve" );

    curve.link(0);

    QL::Date date = cal.advance( asof, 2, Days );
    log.msg( NOTICE, "discount(%s,%d) = %.4f (%.8f)",
	   DateFormatter::toString( date, DateFormatter::ISO ).c_str(),
	   date.serialNumber() - asof.serialNumber(),
	   curve.disc( date ),
	   curve.zero( date ) );

    for( int i=1; i<=10; i++ ) {
      date = cal.advance( asof, i, Years );
      log.msg( NOTICE, "discount(%s,%d) = %.4f (%.8f)",
	     DateFormatter::toString( date, DateFormatter::ISO ).c_str(),
	     date.serialNumber() - asof.serialNumber(),
	     curve.disc( date ),
	     curve.zero( date ) );
    }

    log.msg( NOTICE, "link to deposit/swap curve" );
    curve.link(1);

    date = cal.advance( asof, 2, Days );
    log.msg( NOTICE, "discount(%s,%d) = %.4f (%.8f)",
	   DateFormatter::toString( date, DateFormatter::ISO ).c_str(),
	   date.serialNumber() - asof.serialNumber(),
	   curve.disc( date ),
	   curve.zero( date ) );

    for( int i=1; i<=10; i++ ) {
      date = cal.advance( asof, i, Years );
      double dis = curve.disc (date);
      double zet = curve.zero (date);
      log.msg( NOTICE, "discount(%s,%d) = %.4f (%.8f)",
             DateFormatter::toString( date, DateFormatter::ISO ).c_str(),
             date.serialNumber() - asof.serialNumber(),
             dis, zet);
    }

    boost::shared_ptr<Xibor> index = boost::shared_ptr<Xibor>(new Euribor( 6, Months, curve.ts ));

    log.msg( NOTICE, "index name: %s", index->name().c_str() );

  }
  CATCH_ERROR;

  log.msg( NOTICE, "finished" );

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::testSwap () {
//-----------------------------------------------------------------------------
  static Debug log ("testSwap");
  boost::shared_ptr<SimpleSwap> swap;
  double rate;

  Default def( "Swap", "EUR", curve.ts );
  QL::Date start = def.calendar.advance( asof, 2, Years );

  try {
    rate = 0.04;
    swap = makeSimpleSwap (start, 10, Years, 100, rate, 0, true, "EUR","SWAP");
    log.msg (NOTICE, "swap(%.4f,%.4f) = %.2f, fair rate %.4f spread %.4f",
	     swap->fixedRate(), swap->spread(),
	     swap->NPV(), swap->fairRate(), swap->fairSpread() );

    swap = makeSimpleSwap (start, 10, Years, 100, rate, 0, false,"EUR","SWAP");
    log.msg (NOTICE, "swap(%.4f,%.4f) = %.2f, fair rate %.4f spread %.4f",
	     swap->fixedRate(), swap->spread(),
	     swap->NPV(), swap->fairRate(), swap->fairSpread() );

  } catch (std::exception& e) {
    std::cout << e.what() << std::endl;
    log.msg( ERROR, "%s", e.what() );
    log.msg( ERROR, "exiting..." );
    exit(-1);

  } catch (...) {
    std::cout << "unknown error" << std::endl;
    log.msg( ERROR, "unknown error, exiting ..." );
    exit(-1);
  }

  log.msg( NOTICE, "finished" );

  return 0;
}

//-----------------------------------------------------------------------------
int Engine::testBinomial () {
//-----------------------------------------------------------------------------

  // TODO check volatility dependence

  static Debug debug("testBinomial");

  debug.msg (NOTICE, "starting");

  QL::Rate strikes[] = { 0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08 };
  int  exercises[]   = { 1, 2, 3, 5, 7, 10, 15, 20 };
  int  lengths[]     = { 1, 2, 3, 5, 7, 10 };
  bool payFixed[]    = { false, true };

  double npv0, npv1, ratio;
  int    steps = 50;

  double volatility = 0.2;

  Default def( "Swap", "EUR", curve.ts );

  QL::Date discountDate = def.calendar.advance(asof, 10, Years);
  double discount = curve.ts->discount ( discountDate, true );
  debug.msg( "discount %.6f", discount );

  FILE *f = fopen("binomialtest.txt","w");
  if (!f) {
    debug.msg("error opening file");
    return 1;
  }

  if (f) {
    fprintf( f, "# swaption volatility: %.4f\n", volatility );
    fprintf( f, "# nodes at maturity:   %d\n", steps );
    fprintf( f, "#\n" );
    fprintf( f, "#Exercise Len P/R  Strike       Swap   Swaption        CRR         JR rErr(CRR)  rErr(JR)\n" );
  }

  debug.msg( "start test cases" );

  for (unsigned int i=0; i<LENGTH(exercises); i++) {
    QL::Date exerciseDate = def.calendar.advance( asof ,exercises[i], Years );
    QL::Date startDate    = def.calendar.advance( exerciseDate,
                                                  def.settlementDays, Days );

    for (unsigned int j=0; j<LENGTH(lengths); j++) {
      for (unsigned int k=0; k<LENGTH(payFixed); k++) {

	for (unsigned int l=0; l<LENGTH(strikes); l++) {

	  boost::shared_ptr<SimpleSwap> swap = makeSimpleSwap( startDate,
						    lengths[j], Years,
						    100,
                                                    strikes[l],
						    0,
						    payFixed[k],
						    "EUR" );


	  boost::shared_ptr<QL::Swaption> swaption =
            Engine::makeSwaption( swap, exerciseDate, volatility);

	  npv0 = getBinomialEuropeanSwaptionPrice(0, swap, exerciseDate,
                                                  volatility, 50);
	  npv1 = getBinomialEuropeanSwaptionPrice(1, swap, exerciseDate,
                                                  volatility, 50);

	  ratio = fabs (swaption->NPV()/swap->NPV());
	  if (f) fprintf( f, "%8.2f %4d %3d %7.4f %+10.4f %10.4f %10.4f %10.4f %+.2e %+.2e\n",
			  1.0*(exerciseDate-asof)/365,
			  lengths[j],
			  payFixed[k],
			  strikes[l],
			  swap->NPV(),
			  swaption->NPV(),
			  npv0,
			  npv1,
			  ratio > 1e-6 ? (npv0 - swaption->NPV()) / swaption->NPV() : 0,
			  ratio > 1e-6 ? (npv1 - swaption->NPV()) / swaption->NPV() : 0);

	}
      }
    }
  }

  if (f) fclose(f);

  // test dependence on number of steps

  QL::Date exerciseDate = def.calendar.advance( asof, 2, Years );
  QL::Date startDate    = def.calendar.advance( exerciseDate,
                                                def.settlementDays, Days );
  QL::Rate strike       = 0.04653;
  int  length       = 5;
  bool payer        = true;

  steps = 100;

  boost::shared_ptr<SimpleSwap> swap = makeSimpleSwap( startDate,
					    length, Years,
					    100,
					    strike,
					    0.0,
					    payer,
					    "EUR" );

  boost::shared_ptr<QL::Swaption> swaption =
    makeSwaption( swap, exerciseDate, volatility );

  double blacknpv = swaption->NPV();

  if ( !(f = fopen("binomialteststeps.txt","w")) ) {
    debug.msg("error opening file");
    return 1;
  }

  if (f) {
    fprintf( f, "# swap nominal : %.2f\n", swap->nominal() );
    fprintf( f, "# fixed rate   : %.6f\n", strike );
    fprintf( f, "# today's date : %d/%d/%d\n",
	     asof.dayOfMonth(), asof.month(), asof.year() );
    fprintf( f, "# start date   : %d/%d/%d\n",
	     startDate.dayOfMonth(), startDate.month(), startDate.year() );
    fprintf( f, "# lenght       : %d\n", length );
    fprintf( f, "# expiry date  : %d/%d/%d\n",
	     exerciseDate.dayOfMonth(), exerciseDate.month(),
             exerciseDate.year() );
    fprintf( f, "# volatility   : %.4f\n", volatility );
    fprintf( f, "# fair rate    : %.6f\n", swap->fairRate() );
    fprintf( f, "# swap NPV     : %.6f\n", swap->NPV() );
    fprintf( f, "# black NPV    : %.6f\n", blacknpv );
    fprintf( f, "#\n" );
    fprintf( f, "# steps  rErr(CRR)  rErr(JR)\n" );
  }

  for( int i=1; i<=steps; i++ ) {

    npv0 = getBinomialEuropeanSwaptionPrice( 0, swap, exerciseDate,
                                             volatility, i );

    npv1 = getBinomialEuropeanSwaptionPrice( 1, swap, exerciseDate,
                                             volatility, i );

    if (f) fprintf( f, "%3d  %+10.6f  %+10.6f\n",
		    i, (npv0-blacknpv)/blacknpv, (npv1-blacknpv)/blacknpv );

  }

  if (f) fclose(f);

  debug.msg (NOTICE, "done");

  return 0;
}

//-----------------------------------------------------------------------------
void Engine::testLmm() {
//-----------------------------------------------------------------------------
  static Debug log("testLmm");

  int n         = 20;   // terms, dimension
  double tstart = 0.0;  // year fraction of start date
  double delta  = 0.25; // year fractions
  int lmmType   = 0;    // < 4
  int volType   = 0;    // < 3
  int corrType  = 0;    // < 2
  int lowFactor = 3;
  int paths     = 3;
  int nVals     = 500;  // calibration

  // set up volatility surface

  VolSurface* vols; //  = VolSurface::sample (volType);
  switch (volType) {
  case VolSurface::M :
    vols = new M_VolSurface (1.5, 0.0, 0.0, 2.0);
    break;
  case VolSurface::JR :
    vols = new JR_VolSurface (-0.05, 0.5, 1.5, 0.15);
    break;
  case VolSurface::CONST :
    vols = new CONST_VolSurface (0, 0, 0, 0);
    break;
  default :
    log.msg (ALERT, "volatility surface type %d not recognised", volType );
    exit(-1);
  }

  log.msg (NOTICE, "volSurfaceType %s", vols->volSurfaceType().c_str() );
  log.msg (NOTICE, "%s", vols->as_string().c_str() );

  // set up correlations

  Correlations* corrs; // = Correlations::sample (n, corrType);
  switch (corrType) {
  case Correlations::JR :
    {
      RealArray1D T(n+1);
      T[0] = tstart;
      for (int i=0; i<n; i++) T[i+1] = T[i] + delta;
      MTG::Real beta_ = 0.1;
      corrs = new JR_Correlations (T, beta_);
      break;
    }
  case Correlations::CS :
    {
      MTG::Real alpha_ = 1.8;
      MTG::Real beta_  = 0.1;
      MTG::Real r_oo_  = 0.4;
      corrs = new CS_Correlations (n, alpha_, beta_, r_oo_);
      break;
    }
  default :
    {
      log.msg (ALERT, "correlation type %d not recognised", corrType );
      exit(-1);
    }
  }

  log.msg (NOTICE, "corrType %s", corrs->correlationType().c_str() );
  log.msg (NOTICE, "%s", corrs->as_string().c_str() );

  // set up libor factor loadings

  RealArray1D deltas(n);   // year fractions
  RealArray1D c(n);        // volatility scaling factors, k_j
  RealArray1D L0(n);       // initial forward libors

  // link to deposit/swap curve

  log.msg (NOTICE, "link to flat or depo/swap curve");
  if (getenv(ENV_FLAT_CURVE))
    curve.link(0);
  else
    curve.link(1);

  log.msg (NOTICE, "init deltas, c, libors");
  for (int i=0; i<n; i++) {
    log.msg (NOTICE, "i = %d", i );
    deltas[i] = delta;
    c[i] = 0.2 + 0.1*Random::U01();
    L0[i] = 0.04;
    //L0[i] = curve.ts->forward (tstart + delta * i, tstart + delta * (i+1));
    log.msg (NOTICE, "%d %.2f %.2f %.4f", i, deltas[i], c[i], L0[i] );
  }

  log.msg (NOTICE, "construct libor factor loadings");
  LiborFactorLoading* fl = new LiborFactorLoading (L0, deltas, c, vols, corrs);
  log.msg (NOTICE, "%s", fl->getType()->as_string().c_str() );

  // cap and swaption market data input for calibration
  Int_t Dim(n);
  std::string s_dim   = Dim.toString();
  std::string s_lmm   = LiborMarketModel::lmmType(lmmType);
  std::string s_vols  = VolSurface::volSurfaceType(volType);
  std::string s_corrs = Correlations::correlationType(corrType);
  std::string capletInFile="SyntheticData/CapletsIn-"+s_lmm+"-dim"+s_dim+"-"+s_vols+"-"+s_corrs+".txt";
  std::string swaptionInFile="SyntheticData/SwaptionsIn-"+s_lmm+"-dim"+s_dim+"-"+s_vols+"-"+s_corrs+".txt";

  // set up calibrator

  LmmCalibrator* cal;
  switch (lmmType) {
  case LiborMarketModel::PC  :
    {
      cal = new PredictorCorrectorLmmCalibrator (fl, capletInFile.c_str(), swaptionInFile.c_str(),
						 "CapletsOut.txt", "SwaptionsOut.txt");
      std::cout << "\n\n\nCalibrating predictor-corrector LMM:"
		<< "\nVolSurface: " << s_vols
		<< "\nCorrelations: " << s_corrs
		<< "\nDimension: " << s_dim
		<< "\nto data "+capletInFile+", "+swaptionInFile;
      break;
    }
  default :
    {
      cal = new DriftlessLmmCalibrator (fl, capletInFile.c_str(), swaptionInFile.c_str(),
					"CapletsOut.txt", "SwaptionsOut.txt");
      std::cout << "\n\n\nCalibrating driftless LMM:"
		<< "\nVolSurface: " << s_vols
		<< "\nCorrelations: " << s_corrs
		<< "\nDimension: " << s_dim
		<< "\nto data "+capletInFile+", "+swaptionInFile;
    }
  }
  cal->calibrate(nVals);

  LiborMarketModel* lmm; //=LiborMarketModel::sample (n, lmmType, volType, corrType);
  switch(lmmType){
  case LiborMarketModel::PC :
    lmm = new PredictorCorrectorLMM (fl);
    break;
  case LiborMarketModel::FPC :
    lmm = new FastPredictorCorrectorLMM (fl);
    break;
  case LiborMarketModel::LFDL :
    lmm = new LowFactorDriftlessLMM (fl, lowFactor);
    break;
  default :
    lmm = new DriftlessLMM (fl);
  }

  log.msg (NOTICE, "%s", lmm->getType()->as_string().c_str() );
  //log.msg (NOTICE, "%s", lmm->as_string().c_str() );

  for(int p = 0; p < paths; p++) {
    lmm->newPath();

    printf( "p=%2d   ", p );
    for(int l=0; l<n; l++) printf( "%5d ", l );
    printf( "\n" );

    for(int t=0; t<n; t++) {
      /*
      printf( "  t=%d ", t );
      for( int l=0; l<t; l++ )   printf( "----- " );
      for( int l=t; l<n-1; l++ ) printf( "%.2f ", 100.0 * lmm->XL(l,t));
      printf( "%.2f\n", 100.0 * lmm->XL(n-1,t) );
    */
      printf( "  t=%2d ", t );
      for( int l=0; l<t; l++ )   printf( "----- " );
      for( int l=t; l<n-1; l++ ) printf( "%5.2f ", 100.0 * lmm->L(l,t));
      printf( "%5.2f\n", 100.0 * lmm->L(n-1,t) );
    }
  }
}

//-----------------------------------------------------------------------------
LiborFactorLoading* Engine::testLmmCalibration() {
//-----------------------------------------------------------------------------
  StandardLmmCalibrator::writeSyntheticDataSample();

  // CALIBRATION to caplets (exact) and coterminal swaptions (approximate)
  // needs the files Capletsin.txt, SwaptionsIn.txt containing the synthetic caplet and swaption prices

  int nVals=500,         // number of evaluations of the objective function
    n=20,                // 20,30,40,50 dimension in which we calibrate
    dataLmmType=LiborMarketModel::DL, // type of LMM the data came from: DL,PC
    dataVolType=VolSurface::JR,
    dataCorrType=Correlations::CS,
    lmmType=LiborMarketModel::DL,     // type of LMM to be calibrated
    volType=VolSurface::JR,           // type of VolSurface to be calibrated
    corrType=Correlations::CS;        // type of correlations to be calibrated

  return StandardLmmCalibrator::testCalibration
    (nVals,n,dataLmmType,dataVolType,dataCorrType,lmmType,volType,corrType);
}

//-----------------------------------------------------------------------------
void Engine::testLiborFactorLoading(int n) {
//-----------------------------------------------------------------------------
  MTG::LiborFactorLoading* fl=0;

  for(int volType=0;volType<3;volType++)
    for(int corrType=0;corrType<2;corrType++){

      Timer watch;
      watch.start();
      fl=MTG::LiborFactorLoading::sample(n,volType,corrType);
      fl->selfTest();
      watch.stop();
      watch.report(" ");
    }

} // end testFactorLoading

//-----------------------------------------------------------------------------
void Engine::testLiborFactorLoadingFactorization(int n, int r) {
//-----------------------------------------------------------------------------
  LiborFactorLoading* fl=0;

  for(int volType=0;volType<3;volType++)
    for(int corrType=0;corrType<2;corrType++){

      printStars();
      cout << "\n\nTesting Libor factor loading,"
	   << "\ncorrelation matrices low rank factorization:"
	   << "\nVolatility surface type = VolSurface::" << volType
	   << "\nCorrelation type = Correlations::" << corrType;
      Timer watch;
      watch.start();
      fl=LiborFactorLoading::sample(n,volType,corrType);
      for(int t=0;t<n-2;t++){

	const UTRRealMatrix& CV=fl->logLiborCovariationMatrix(t);
	CV.testFactorization(r);
      }
      watch.stop();
      watch.report(" ");
    }
} // end factorizationTest

//-----------------------------------------------------------------------------
void Engine::testLmmPaths(int n) {
//-----------------------------------------------------------------------------
  for(int lmmType=0;lmmType<4;lmmType++)
    for(int volType=0;volType<3;volType++)
      for(int corrType=0;corrType<2;corrType++){

	LiborMarketModel* lmm=LiborMarketModel::sample(n,lmmType,volType,corrType);
	printStars();
	cout << "\n\n5 Libor paths, LMM type: " << *(lmm->getType())
	     << endl << endl;
	for(int path=0;path<5;path++){

	  lmm->newPath();
	  for(int t=0;t<n-5;t++) std::cout << lmm->XL(n-4,t) << ", ";
	  cout << lmm->XL(n-4,n-5) << endl;
	}
      }
}

//-----------------------------------------------------------------------------
void Engine::testSwaptionPrice(int lmmType, int volType, int corrType) {
//-----------------------------------------------------------------------------
  int do_again=1;
  // main loop
  while(do_again==1){

    printStars();
    cout << "\n\nPricing swaption along [T_p,T_n] "
	 << "in LMM of dimension n."
	 << "\n\nEnter p = ";
    int p; cin >> p;
    cout << "Enter n = ";
    int n; cin >> n;

    Timer watch; watch.start();
    MTG::Swaption* swpn=MTG::Swaption::sample(p,n,lmmType,volType,corrType);
    swpn->testPrice(8192);
    watch.stop(); watch.report(" ");

    cout << "\n\nDo another run (yes = 1, no = 0) do_again = ";
    cin >> do_again;
  }
}

//-----------------------------------------------------------------------------
void Engine::testCapletPrice(int lmmType, int volType, int corrType) {
//-----------------------------------------------------------------------------
  int do_again=1;
  // main loop
  while(do_again==1){

    printStars();
    cout << "\n\nPricing caplet at i=n/3."
	 << "\nEnter dimension n of Libor process: n = ";
    int n; cin >> n;

    Timer watch; watch.start();
    LiborDerivative* cplt=MTG::Caplet::sample(n,lmmType,volType,corrType);
    cplt->testPrice(8192);
    watch.report(" ");

    cout << "\n\nDo another run (yes = 1, no = 0) do_again = ";
    cin >> do_again;
  }
}

//-----------------------------------------------------------------------------
void Engine::testCallOnBondPrice(int lmmType, int volType, int corrType) {
//-----------------------------------------------------------------------------
  int do_again=1;
  // main loop
  while(do_again==1){

    printStars();
    cout << "\n\nPricing at the money call on bond along [T_p,T_q] "
	 << "with random coupons c_j in [0.5,1.5]"
	 << "\n\nEnter p = ";
    int p; cin >> p;
    cout << "Enter q = ";
    int q; cin >> q;

    Timer watch; watch.start();
    BondCall* bc=BondCall::sample(p,q,lmmType,volType,corrType);
    bc->testPrice(8192);
    watch.report(" ");

    cout << "\n\nDo another run (yes = 1, no = 0) do_again = ";
    cin >> do_again;
  }
}

//-----------------------------------------------------------------------------
void Engine::testCallOnZeroCouponBondPrice(int lmmType, int volType, int corrType) {
//-----------------------------------------------------------------------------
  int do_again=1;
  // main loop
  while(do_again==1){

    printStars();
    cout << "\n\nPricing at the money call expiring at T_{p-1}"
	 << "\non zero coupon bond maturing at T_p"
	 << "\nin LMM of dimension p+3."
	 << "\n\nEnter p = ";
    int p; cin >> p;

    Timer watch; watch.start();
    BondCall*  bc=BondCall::sampleCallOnZeroCouponBond(p,lmmType,volType,corrType);
    bc->testPrice(8192);
    watch.report(" ");

    cout << "\n\nDo another run (yes = 1, no = 0) do_again = ";
    cin >> do_again;
  }
}

//-----------------------------------------------------------------------------
void Engine::testLmmLattice() {
//-----------------------------------------------------------------------------
  int do_again=1;
  // main loop
  while(do_again==1){

    printStars();
    cout << "\n\nBuilding and testing a lattice for the driftless libor market model:"
	 << "\n\nEnter dimension n of Libor process: n = ";
    int n; cin >> n;
    cout << "Enter number r of factors (2 or 3): r = ";
    int r; cin >> r;
    if((r!=2)&&(r!=3))
      cout << "\n\n\nNumber of factors must be two or three.";
    else LmmLattice::test(r,n);
    cout << "\n\nDo another run (yes = 1, no = 0) do_again = ";
    cin >> do_again;
  }
} // end testLmmlattice

//-----------------------------------------------------------------------------
void Engine::testSwaption() {
//-----------------------------------------------------------------------------
  cout << "\nSwap interval: [T_p,T_q]."
       << "\nEnter p = ";
  int p; cin >> p;
  cout << "Enter q = ";
  int q; cin >> q;
  cout << "Enter number of Libor paths: ";
  int nPath; cin >> nPath;

  MTG::Swaption* swpn = MTG::Swaption::sample(p,q);
  swpn->testPrice(nPath);
}

//-----------------------------------------------------------------------------
void Engine::testBermudanSwaption() {
//-----------------------------------------------------------------------------
  cout << "\nSwap interval: [T_p,T_q]."
       << "\nEnter p = ";
  int p; cin >> p;
  cout << "Enter q = ";
  int q; cin >> q;
  cout << "Enter number of Libor paths for Monte Carlo simulation: ";
  int nPath; cin >> nPath;
  cout << "Enter number of training paths for the trigger:  ";
  int paths; cin >> paths;

  bool verbose=false;
  BermudanSwaption* bswpn = BermudanSwaption::sample(p,q,paths,verbose);
  bswpn->testPrice(nPath);
}

//-----------------------------------------------------------------------------
void Engine::testCallOnBond() {
//-----------------------------------------------------------------------------
  cout << "\nBond with random coupons c_j in [0.5,1.5] on [T_p,T_q]."
       << "\n\nEnter p = ";
  int p; cin >> p;
  cout << "Enter q = ";
  int q; cin >> q;
  cout << "Enter number of Libor paths: ";
  int nPath; cin >> nPath;

  BondCall* bondcall = BondCall::sample(p,q);
  bondcall->testPrice(nPath);
}

//-----------------------------------------------------------------------------
void Engine::testCallOnZeroCouponBond() {
//-----------------------------------------------------------------------------
  cout << "\nZero coupon bond matures at T_p:"
       << "\n\nEnter p = ";
  int p; cin >> p;
  cout << "Enter number of Libor paths: ";
  int nPath; cin >> nPath;

  BondCall* bondcall = BondCall::sampleCallOnZeroCouponBond(p);
  bondcall->testPrice(nPath);
}

//-----------------------------------------------------------------------------
void Engine::testCaplet() {
//-----------------------------------------------------------------------------
  cout << "\nCaplet on [T_i,T_{i+1}], i=n/3 in LMM of dimension n."
       << "\n\nEnter n = ";
  int n; cin >> n;
  cout << "Enter number of Libor paths: ";
  int nPath; cin >> nPath;

  MTG::Caplet* cplt = MTG::Caplet::sample(n);
  cplt->testPrice(nPath);
}

//-----------------------------------------------------------------------------
void Engine::testLiborDerivative() {
//-----------------------------------------------------------------------------
  int do_again=1;
  // main loop
  while(do_again==1){

    printStars();
    cout << "\n\nPricing of at the money Libor derivatives"
	 << "\nin the default driftless Libor Market Model."
	 << "\nChoose derivative:"
	 << "\nCaplet (0)"
	 << "\nSwaption (1)"
	 << "\nBermudan swaption (2)"
	 << "\nCall on bond (3)"
	 << "\nCall on zero coupon bond (4)"
	 << "\n\nDerivative = ";

    int derivative; cin>>derivative;
    switch(derivative){

    case 0  : testCaplet(); break;
    case 1  : testSwaption(); break;
    case 2  : testBermudanSwaption(); break;
    case 3  : testCallOnBond(); break;
    case 4  : testCallOnZeroCouponBond(); break;
    default :
      cout << "\nDerivative not recognized. Defaulting to swaption.";
      testSwaption();
    }

    cout << "\n\nDo another run (yes = 1, no = 0) do_again = ";
    cin >> do_again;
  }
} // end testLiborDerivative

}
//eof
