
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <mtm.h>

int  usage                   (char *name);
int  env_usage               (char *name);
void createLmmParameters     ();
void createFloater           (QL::Date asofdate, char *file);
void createPortfolio         (QL::Date asofdate, char *file, int sample);
void writeFactorLoading      (MTG::LiborFactorLoading *lfl);

enum LmmTest {
  lmmNone = 0,
  lmmQ = 1,
  lmmM = 2,
  lmmS = 3
};

//-----------------------------------------------------------------------------
int main( int argc, char **argv ) {
//-----------------------------------------------------------------------------
  int  ret, samples;
  bool sample  = false;
  bool lmmtest = false;
  LmmTest mode = lmmNone;
  char buf[100], ifile[100], ofile[100], mfile[100], cdate[100];

  QuantLib::Calendar cal   = QuantLib::TARGET();
  QuantLib::Date     today = QuantLib::Date::todaysDate();
  QuantLib::Date     asof  = cal.adjust (today, QuantLib::Preceding);
  QuantLib::Date asofdate  = QuantLib::Date();

  if ( getarg (argc, argv, "-H", buf) == 0 ) return usage (argv[0]);

  if ( getarg (argc, argv, "-ENV", buf) == 0 ) return env_usage (argv[0]);

  if ( getarg (argc, argv, "-V", buf) == 0 ) {
    printf( "%s version %s\n", PACKAGE_NAME, PACKAGE_VERSION);
    return 0;
  }

  if ( (ret = getarg (argc, argv, "-E", buf)) >= 0 ) {
    sample = true;
    if (ret == 1) samples = atoi(buf);
    else          samples = 1;
    if ( getarg (argc, argv, "-O", ofile) < 1 ) return usage (argv[0]);
    if ( getarg (argc, argv, "-D", cdate) < 1 ) {
      asofdate = asof;
    } else {
      asofdate = parseDate( cdate );
    }
  }
  else if ( getarg (argc, argv, "-LMM", buf) >= 0) {
    lmmtest = true;
    if (strcmp(buf,"M")==0)      mode = lmmM;
    else if (strcmp(buf,"S")==0) mode = lmmS;
    else                         mode = lmmQ;
    if (mode == lmmQ)
      if ( getarg (argc, argv, "-I", ifile) < 1 ) return usage (argv[0]);
    if ( getarg (argc, argv, "-M", mfile) < 1 ) return usage (argv[0]);
    if ( getarg (argc, argv, "-D", cdate) < 1 ) {
      asofdate = asof;
    } else {
      asofdate = parseDate( cdate );
    }
  }
  else {
    if ( getarg (argc, argv, "-I", ifile) < 1 ) return usage (argv[0]);
    if ( getarg (argc, argv, "-O", ofile) < 1 ) return usage (argv[0]);
    if ( getarg (argc, argv, "-M", mfile) < 1 ) return usage (argv[0]);
    if ( getarg (argc, argv, "-D", cdate) < 1 ) {
      asofdate = asof;
    } else {
      asofdate = parseDate( cdate );
    }
  }

  static Debug log("main",1);

  log.msg( "starting" );

  std::cout << "today's date: "
	    << DateFormatter::toString (today, DateFormatter::ISO)
	    << "  (" << DateFormatter::toString (today) << ")" << std::endl;
  std::cout << "asof date:    "
	    << DateFormatter::toString (asofdate, DateFormatter::ISO)
	    << "  (" << DateFormatter::toString (asofdate) << ")" << std::endl;

  if (asofdate.weekday() == QL::Sunday || asofdate.weekday() == QL::Saturday) {
    std::cout << "please choose a non-weekend day, exit." << std::endl;
    return 1;
  }

// set the global valuation date

  Settings::instance().evaluationDate() = asofdate;

  if (sample > 0) { //---------- create sample instrument xml file ------------
    createFloater (asofdate, ofile);
//    createPortfolio (asofdate, ofile, sample);
    createLmmParameters();

    log.msg (NOTICE, "all done");
    printf ("done\n");

    return 0;
  }

  // --------------------------- common setup ---------------------------------
  XML::list<RL::MarketPoint> market;
  XML::list<RL::Instrument>  instruments;

  // read market data
  log.msg (NOTICE, "read market");
  market.read (mfile);

  // set up engine
  log.msg (NOTICE, "setup engine");
  RL::Engine engine (&market);

  // QuantLib tests
  log.msg (NOTICE, "perform basic tests");
  engine.testCurve();
  engine.testSwap();
  engine.testCurve();
  engine.testBinomial();

  if (mode == lmmM) { // ------- run synthetic data examples ------------------
    // engine.testLmm();
    MTG::LiborFactorLoading *lfl = engine.testLmmCalibration();
    writeFactorLoading (lfl);

    // engine.testLiborDerivative();
    // engine.testLmmPaths (16);
    // engine.testLmmLattice();
    // engine.testLiborFactorLoading (10);
    log.msg (NOTICE, "all done");
    printf ("done\n");

    return 0;
  }

  if (mode == lmmS) { // ------- test Libor Market Model on Summit data--------

    MTG::LiborMarketModel* lmm = engine.setupModel ();

    if (lmm) writeFactorLoading (lmm->getFactorLoading());
    else {
      log.msg (ALERT, "lmm pointer is NULL, exit");
      return -1;
    }
    log.msg (NOTICE, "all done");
    printf ("done\n");

    return 0;
  }

  // read instruments
  log.msg (NOTICE, "read instruments");
  instruments.read (ifile);

  if (mode == lmmQ) { // ------- test Libor Market Model ----------------------
    // use first instrument in the list to define calibration set
    MTG::LiborMarketModel* lmm = engine.setupModel (*instruments.begin());

    writeFactorLoading (lmm->getFactorLoading());

    log.msg (NOTICE, "all done");
    printf ("done\n");

    return 0;
  }

  // --------------------------- process instrument xml file ------------------

  int cnt = 1;
  for (XML::list<RL::Instrument>::iterator inst = instruments.begin();
       inst != instruments.end(); inst++ ) {
    printf( "INSTRUMENT %d\n", cnt );

    if (engine.price (*inst)) {
      log.msg (ALERT, "error pricing trade %d", cnt);
    }

    cnt++;
//inst->logStats();
  }

  instruments.write (ofile);

  log.msg (NOTICE, "nvar = %d", RL::Formula::nvar );
  log.msg (NOTICE, "npop = %d", RL::Formula::npop );

  log.msg (NOTICE, "all done");
  printf ("done\n");

  return 0;
}

//-----------------------------------------------------------------------------
void writeFactorLoading (MTG::LiborFactorLoading *lfl) {
//-----------------------------------------------------------------------------
  static Debug log ("writeFactorLoading");

  log.msg (NOTICE, "write factor loadings");

  MTG::RealArray1D         scf = lfl->getScalingFactors();
  MTG::RealArray1D         ten = lfl->getTenorStructure();
  MTG::RealArray1D         del = lfl->getDeltas();
  MTG::RealArray1D         lib = lfl->getInitialLibors();
  MTG::VolSurface*         vol = lfl->getVolSurface();
  MTG::Correlations*       cor = lfl->getCorrelations();

  FILE * f = fopen ("FactorLoading.txt", "w");
  if (!f) {
    log.msg (ALERT, "error opening file, exit.");
    exit(1);
  }

  fprintf (f, "%s\n", lfl->getType()->as_string().c_str());
  fprintf (f, "%s\n", vol->as_string().c_str() );
  fprintf (f, "%s\n", cor->as_string().c_str() );

  fprintf (f, "## %-10s %-10s %-10s %-10s %-10s %-10s\n",
	   "Time", "Delta", "Libor", "Scaling", "Sigma", "Scaling*Sigma");
  for (int i = 0; i < lfl->getDimension(); i++) {
    fprintf (f, "%2d %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f\n",
	     i, ten[i], del[i], lib[i], scf[i], vol->sigma (0, ten[i]+del[i]),
	     scf[i] * vol->sigma (0, ten[i]+del[i]));
  }

  fclose (f);

  log.msg ("done");
}

//-----------------------------------------------------------------------------
void createLmmParameters () {
//-----------------------------------------------------------------------------
  static Debug log ("lmm parameters");

  XML::list<RL::LmmVolSurface> vols;
  XML::list<RL::LmmCorrelations> cors;

  vols.append (new RL::LmmVolSurface ("JR",   -2.0, -2.0, 1.0, 2.0) );
  vols.append (new RL::LmmVolSurface ("M",     1.0,  1.0, 1.0, 1.5) );
  vols.append (new RL::LmmVolSurface ("CONST", 1.0,  1.0, 1.0, 1.0) );

  cors.append (new RL::LmmCorrelations ("JR", 1.0, 0.2,  1.0) );
  cors.append (new RL::LmmCorrelations ("CS", 1.8, 0.36, 0.44) );

  RL::LmmParameters params ("DL", "JR", "CS", 3, 500, 1000, vols, cors);

  params.writeObject("lmm.xml");

  /*
    params.init();
    params.writeObject("lmm.xml.2");
    params.readObject("lmm.xml");
    params.writeObject("lmm.xml.3");
  */
}

//-----------------------------------------------------------------------------
void createFloater (QL::Date asofdate, char *file) {
//-----------------------------------------------------------------------------
  static Debug log ("floater");

  // set up schedules and index
  QL::Calendar cal   = mapCalendar ("LON");
  QL::Date startDate = cal.advance (asofdate, 2, Days);
  QL::Date endDate   = cal.advance (startDate, 10, Years);
  std::string start  = DateFormatter::toString (startDate, DateFormatter::ISO);
  std::string end    = DateFormatter::toString (endDate, DateFormatter::ISO);

  std::cout << "start date " << start << std::endl;
  std::cout << "end date   " << end   << std::endl;

  RL::Schedule fltres( start.c_str(), end.c_str(), "", "", "", "S", "Y", "F", "LON", 0, "B" );
  RL::Schedule fixing( start.c_str(), end.c_str(), "", "", "", "S", "Y", "F", "LON", -2, "B" );
  RL::Schedule pay   ( start.c_str(), end.c_str(), "", "", "", "S", "Y", "F", "LON", 2, "B" );

  XML::ptr<RL::Index> fltIdx( new RL::Index( "FLOAT", "EUR", "A360", fltres,
					       "EURIBOR", "6M", 10, "ADV", -2, fixing) );

  // set up notional list

  XML::list<RL::Notional> nrlist;

  nrlist.append (new RL::Notional (+100, "EUR"));

  // set up legs

  RL::Leg leg("rec_float", "INDEX", "EUR", nrlist, fltIdx, "ARR", 0, pay);

  XML::list<RL::Leg> legs;

  legs.append (leg);

  for( XML::list<RL::Leg>::iterator it = legs.begin(); it != legs.end(); it++ ) {
    printf( " (%s,%s)\n", it->getId().c_str(), it->getCcy().c_str() );
  }

  // set up structures and instruments

  RL::Structure linear ("LINEAR");
  XML::ptr<RL::Instrument> inst (new RL::Instrument("LINEAR", 1, "EUR", legs, linear));

  // set up stats

  inst->stats.ccy = "EUR";
  inst->stats.npv = 0;

  inst->print();

  // set up instrument list

  XML::list<RL::Instrument> instruments;

  instruments.append (inst);

  for( XML::list<RL::Instrument>::iterator inst = instruments.begin();
       inst != instruments.end();
       inst ++ ) {
    inst->print();
  }

  // export to xml

  instruments.write (file);

  // read and write again

  XML::list<RL::Instrument> inst2;

  inst2.read (file);

  std::string file2 (file);
  file2 += ".xml";

  inst2.write ((char*)file2.c_str());

  log.msg (NOTICE, "results written to %s and %s", file, file2.c_str());
}

//-----------------------------------------------------------------------------
void createPortfolio (QL::Date asofdate, char *file, int sample) {
//-----------------------------------------------------------------------------
  static Debug log ("createPortfolio");

  QL::Calendar cal   = mapCalendar ("LON");
  QL::Date startDate = cal.advance (asofdate, 2, Days);
  QL::Date endDate   = cal.advance (startDate, 10, Years);
  std::string  start = DateFormatter::toString (startDate, DateFormatter::ISO);
  std::string  end   = DateFormatter::toString (endDate, DateFormatter::ISO);

  std::cout << "start date " << start << std::endl;
  std::cout << "end date   " << end   << std::endl;

  log.msg (NOTICE, "*** test starting ***" );

  log.msg( NOTICE, "*** set up and evaluate test formulas ***" );

  RL::Formula formula (start.c_str(), "0.05*step(index-0.06)+0.02*step(0.06-index)");
  double var;

  log.msg (NOTICE, "sizeof(Formula) = %d", sizeof(RL::Formula) );

  var = 0.07;
  log.msg (NOTICE, "formula evaluation (%.2f) = %.4f", var, formula.eval(var) );
  var = 0.04;
  log.msg (NOTICE, "formula evaluation (%.2f) = %.4f", var, formula.eval(var) );

  RL::Formula formula2 = formula;
  var = 0.07;
  log.msg (NOTICE, "formula evaluation (%.2f) = %.4f", var, formula2.eval(var) );
  var = 0.04;
  log.msg (NOTICE, "formula evaluation (%.2f) = %.4f", var, formula2.eval(var) );

  log.msg( NOTICE, "*** set up dates and schedules ***" );

  // set up schedules and index

  RL::Schedule fixres (start.c_str(), end.c_str(), "", "", "", "S", "N", "F", "LON",  0, "B" );
  RL::Schedule fltres (start.c_str(), end.c_str(), "", "", "", "S", "Y", "F", "LON",  0, "B" );
  RL::Schedule fixing (start.c_str(), end.c_str(), "", "", "", "S", "Y", "F", "LON", -2, "B" );
  RL::Schedule pay    (start.c_str(), end.c_str(), "", "", "", "A", "Y", "F", "FRA", +2, "B" );

  XML::list<RL::Date> dates;
  dates.append( "2005-01-02" );
  dates.append( "2005-04-02" );
  dates.append( "2005-07-02" );
  RL::Schedule custom( dates );

  XML::list<RL::Formula> forms;

  XML::ptr<RL::Index> genIdx( new RL::Index( "FLOAT", "EUR", "A360", fltres,
					       0,
					       "EURIBOR", "6M", 0.1, "ADV", -2, fixing,
					       "Y", 0.05, "Y", 0.01, forms ) );

  XML::ptr<RL::Index> fixIdx( new RL::Index( "FIXED", "EUR", "ACT", fixres, 0.05 ) );

  XML::ptr<RL::Index> fltIdx( new RL::Index( "FLOAT", "EUR", "A360", fltres,
					       "EURIBOR", "6M", 0, "ADV", -2, fixing) );

  XML::ptr<RL::Index> collarIdx( new RL::Index( "FLOATCF", "EUR", "A360", fltres,
						  "EURIBOR", "6M", 0, "ADV", -2, fixing,
						  "Y", 0.04, "Y", 0.03 ) );

  XML::ptr<RL::Index> capIdx( new RL::Index( "FLOATCF", "EUR", "A360", fltres,
					       "EURIBOR", "6M", 0, "ADV", -2, fixing,
					       "Y", 0.04, "N", 0) );

  //  forms.append(new RL::Formula("2004-05-01", "0.04*step(index-0.04) + index*step(0.04-index)"));
  forms.append(new RL::Formula(start.c_str(), "0.04*step(index-0.04) + index*step(0.04-index)*step(index-0.03) + 0.03*step(0.03-index) "));
//   forms.append(new RL::Formula("2005-05-01", "0.06*step(index-0.06)+0.03*step(0.06-index)"));
//   forms.append(new RL::Formula("2006-05-01", "0.07*step(index-0.06)+0.04*step(0.06-index)"));
//   forms.append(new RL::Formula("2007-05-01", "0.08*step(index-0.06)+0.05*step(0.06-index)"));

  XML::ptr<RL::Index> formIdx( new RL::Index( "FORM", "EUR", "A360", fltres,
						"EURIBOR", "6M", 0, "ADV", -2, fixing,
						forms ) );

  // set up notional list

  XML::list<RL::Notional> nplist;
  RL::Notional n1 (-100., "EUR");
  nplist.append (n1);
  nplist.append (new RL::Notional (-200, "EUR"));

  XML::list<RL::Notional> nrlist;
  nrlist.append (new RL::Notional (+100, "EUR"));

  // set up legs

  log.msg (NOTICE,  "*** create legs, add to list<Leg> ***" );

  RL::Leg leg1("pay_fixed", "INDEX", "EUR", nplist, fixIdx, "ARR", 0, pay);
  RL::Leg leg2("rec_floater", "INDEX", "EUR", nrlist, fltIdx, "ARR", 0, pay);
  RL::Leg leg3("rec_capped_floater", "INDEX", "EUR", nrlist, capIdx, "ARR", 0, pay);
  RL::Leg leg4("rec_collar", "INDEX", "EUR", nrlist, collarIdx, "ARR", 0, pay);
  RL::Leg leg5("rec_form", "INDEX", "EUR", nrlist, formIdx, "ARR", 0, pay);

  XML::list<RL::Leg> legs1, legs2, legs3, legs4, legs5, legs, swaplegs;

  legs1.append( leg1 );
  legs2.append( leg2 );
  legs3.append( leg3 );
  legs4.append( leg4 );
  legs5.append( leg5 );

  legs.append( leg1 );
  legs.append( leg2 );
  legs.append( leg3 );
  legs.append( leg4 );
  legs.append( leg5 );

  swaplegs.append (leg1);
  swaplegs.append (leg2);

  for( XML::list<RL::Leg>::iterator it = legs.begin(); it != legs.end(); it++ ) {
    printf( " (%s,%s)\n", it->getId().c_str(), it->getCcy().c_str() );
  }

  // set up structures and instruments

  log.msg (NOTICE, "*** create structures and instruments ***" );

  RL::Schedule exercise( "2005-05-01", "2010-05-01", "", "", "", "S", "Y", "F", "LON", 0, "B" );
  RL::Schedule notice( "2005-05-01", "2010-05-01", "", "", "", "S", "Y", "F", "LON", -5, "B" );

  RL::Structure linear ("LINEAR");
  XML::ptr<RL::Instrument> u1(new RL::Instrument("LINEAR-1", 1, "EUR", legs1, linear));
  XML::ptr<RL::Instrument> u2(new RL::Instrument("LINEAR-2", 1, "EUR", legs2, linear));
  XML::ptr<RL::Instrument> u3(new RL::Instrument("LINEAR-3", 1, "EUR", legs3, linear));
  XML::ptr<RL::Instrument> u4(new RL::Instrument("LINEAR-4", 1, "EUR", legs4, linear));
  XML::ptr<RL::Instrument> u5(new RL::Instrument("LINEAR-5", 1, "EUR", legs5, linear));
  XML::ptr<RL::Instrument> u6(new RL::Instrument("SWAP", 1, "EUR", swaplegs, linear));
  XML::ptr<RL::Instrument> u7(new RL::Instrument("LINEAR", 1, "EUR", legs, linear));

  RL::Structure bermudan ("BERMUDAN", "ENTER", "", exercise, notice, u1 );
  XML::ptr<RL::Instrument> u8(new RL::Instrument("BERMUDAN", 2, "USD", legs, bermudan));

  RL::Structure callable ("CALLABLE", "",      "", exercise, notice, u2 );
  XML::ptr<RL::Instrument> top(new RL::Instrument("CALLABLE", 2, "GBP", legs, callable));

  // set up stats

  u1->stats.ccy = "EUR";
  u1->stats.npv = 11.2;

  u2->stats.ccy = "USD";
  u2->stats.npv = 22.2;

  top->stats.ccy = "GBP";
  top->stats.npv = 4711;

  u1->print();
  u2->print();
  u3->print();
  u4->print();
  u5->print();
  u6->print();
  u7->print();

  top->print();

  XML::list<RL::Instrument> instruments;

  for (int i=0; i<sample; i++) {
    instruments.append( u1 );
    instruments.append( u2 );
    instruments.append( u3 );
    instruments.append( u4 );
    instruments.append( u5 );
    instruments.append( u6 );
    instruments.append( u7 );
  }

  for( XML::list<RL::Instrument>::iterator inst = instruments.begin();
       inst != instruments.end();
       inst ++ ) {
    inst->print();
  }

  log.msg (NOTICE, "*** write file ***" );

  instruments.write( file );

  log.msg (NOTICE,  "*** read back from file ***" );

  XML::list<RL::Instrument> inst2;

  inst2.read( file );

  log.msg (NOTICE, "*** write again ***" );

  std::string file2(file);
  file2 += ".xml";

  inst2.write( (char*)file2.c_str() );

  log.msg (NOTICE, "results written to %s and %s", file, file2.c_str());

  log.msg (NOTICE, "test finished");
}

//-----------------------------------------------------------------------------
int usage( char *name ) {
//-----------------------------------------------------------------------------
  printf( "\n" );
  printf( "usage: %s [options]\n", name );
  printf( "options:\n" );
  printf( "  -h           Show this screen\n" );
  printf( "  -env         Show help on environment settings\n" );
  printf( "  -v           Show version\n" );
  printf( "  -e <N>       Build N example objects and write xml, default N=1\n" );
  printf( "  -lmm <mode>  Run a LMM calibration using:\n");
  printf( "          Q    - raw market data, basic QL pricing tools, and\n");
  printf( "                 calibration instruments adjusted to the reset\n");
  printf( "                 periods of the instrument provided (default, mode = Q)\n");
  printf( "          M    - Michael J. Meyer's synthetic data (mode = M)\n");
  printf( "          S    - SUMMIT forwards and calibration instruments\n");
  printf( "                 in sub-dir Summit (mode = S) - NOT YET IMPLEMENTED\n" );
  printf( "  -i <file>    Input xml file (instruments)\n" );
  printf( "  -o <file>    Output xml file (instruments and statistics)\n" );
  printf( "  -m <file>    Market data input xml file\n" );
  printf( "  -d <date>    Valuation date, valid formats:\n" );
  printf( "               YYYYMMDD, YYYY-MM-DD, DD/MM/YY, DD.MM.YY\n" );
  printf( "\n" );
  printf( "typical usage:\n" );
  printf( "  %s  -d date  -o out.xml  -e [samples]  \n", name );
  printf( "  %s  -d date  -o out.xml  -i in.xml  -m mkt.xml\n", name );
  printf( "  %s  -d date  -lmm    -m mkt.xml  -i in.xml\n", name );
  printf( "  %s  -d date  -lmm M  -m mkt.xml\n", name );
  printf( "  %s  -d date  -lmm S  -m mkt.xml\n", name );
  printf( "\n" );

  return 1;
}

//-----------------------------------------------------------------------------
int env_usage( char *name ) {
//-----------------------------------------------------------------------------
  printf( "\n" );
  printf( "pricing models:\n" );
  printf( "  ENV_MODEL   LMM\n\n" );
  printf( "debug environment variables:\n" );
  printf( "  DEBUG_FILE  Redirect messages to either file, stdout or stderr\n" );
  printf( "  DEBUG_MASK  Set external binary mask {E} for filtering messages as follows:\n" );
  printf( "              Messages are internally prioritised using internal masks I\n" );
  printf( "              where I is an 8 bit number varying between 1 and 255.\n" );
  printf( "              The internal mask is shown on each log file line in curly braces.\n" );
  printf( "              If {E} is set, a message is only written if bitwise E & I > 0\n" );
  printf( "              For example: I =  3 = 00000011 internal mask or priority\n" );
  printf( "                           E = 10 = 00001010 external mask set via DEBUG_MASK\n" );
  printf( "                                ------------\n" );
  printf( "                                2 = 00000010 > 0 => message is written\n" );
  printf( "              Setting DEBUG_MASK to 255 will show all messages.\n" );
  printf( "  DEBUG_FLUSH Flush output buffer after write if set\n" );
  printf( "  DEBUG_BEEP  Enable beep for each message if set\n" );
  printf( "\n" );
  return 1;
}

