/* WARANTY NOTICE AND COPYRIGHT
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Copyright (C) Michael J. Meyer

matmjm@mindspring.com
spyqqqdia@yahoo.com

*/

#ifndef martingale_volatilityandcorrelation_h
#define martingale_volatilityandcorrelation_h

#include "TypedefsMacros.h"
#include "Matrix.h"
#include <string>
#include <iostream>

//RL CHANGE
#include "internal.h"

MTGL_BEGIN_NAMESPACE(Martingale)

/*! \file VolatilityAndCorrelation.h
 * Interface and implementation of deterministic volatility surfaces and
 * constant correlations. These are used in the factor loadings of Ito processes
 * (such as asset or Libor logarithms (returns)), see {@link FactorLoading} and
 * {@link LiborFactorLoading}.
 */

/*******************************************************************************
 *
 *                    VOLATILITY SURFACE
 *
 ******************************************************************************/

/** <p>Deterministic volatility surface \f$\sigma(t,T)\f$ for a FactorLoading.
 *  The actual volatility function of the process \f$Y_j\f$ (in applications to
 *  finance usually asset returns, that is asset logarithms or Libor logarithms)
 *  is then given as
 *  \f[\sigma_j(t)=k_j\sigma(t,T_j),\quad j=1,\dots,n-1.\f]
 *  where \f$k_j\f$ is a scaling factor to calibrate the annualized volatility
 *  of \f$Y_j\f$. In the case of a {@link DriftlessLMM} \f$Y_j=log(U_j)\f$ is the
 *  the logarithm of forward transported Libor .
 *
 *  <p>To simplify calibration we assume that all volatility surface types
 *  depend on four Real parameters a,b,c,d. This suffices for all types which
 *  are implemented and the maximal parameter set becomes part of the interface.
 *  In general this is objectionable but here it simplifies the calibration of
 *  factorloadings. The calibration routines use the maximal (and hence the same)
 *  set of parameters for all types of surfaces even though any particular type
 *  may not depend on all parameters.
 *
 * <p>For simplicity k[0] is included in the calibration (although it has
 *  no effect in the case of Libors). The entire volatility structure then depends on
 *  the vector k of scaling factors and the parameters a,b,c,d, an n+4 dimensional
 *  vector. In the case of Libors the vector k is part of the {@link LiborFactorLoading}.
 */
class VolSurface
{

  protected:

    /** Flag for the type of vol surface: M,JR,CT.
     */
    int volType;

    Real a,b,c,d;                                 // the parameters

  public:

    /** Flags identifying the volatility surface type:<br>
     *  M: book, 6.11.1.<br>
     *  JR: Jaeckel-Rebonato, Jaeckel book, p164, 12.11.<br>
     *  CT: constant.
     */
    static const int M=0, JR=1, CONST=2;

    /** @param a0,b0,c0,d0 parameters of the vol surface
     *  @param type vol surface type M, JR or CONST.
     */
    VolSurface
      (Real a0, Real b0, Real c0, Real d0, int type) :
    volType(type), a(a0), b(b0), c(c0), d(d0)
      {    }

    // RL CHANGE: virtual destructor added
    virtual ~VolSurface () { }

    /** Type ID, implemented: M, JR, CONST.
     */
    int getType() const { return volType; }

    /** Printing the parameters a,b,c,d.*/
    void printParameters();

    // RL CHANGE
    double getA();
    double getB();
    double getC();
    double getD();

    /** A sample surface of the type = M, JR, CONST.
     */
    static VolSurface* sample(int type);

    /** "CONST", "M" or "JR", converts integer type ID to string. */
    static std::string volSurfaceType(int type);

    /** The volatility surface \f$\sigma(t,T)\f$.
     */
    virtual Real sigma(Real t, Real T) const = 0;

    /** The indefinite integral
     *  \f[\int\sigma(t,T_1)\sigma(t,T_2)ds.\f]
     *  Needed for covariation integrals.
     */
    virtual Real integral_sgsg(Real t, Real T_1, Real T_2) const = 0;

    /** The definite integral
     *  \f[\int_s^t\sigma(u,T_1)\sigma(u,T_2)du.\f]
     *  Needed for covariation integrals.
     */
    Real integral_sgsg(Real s, Real t, Real T_1, Real T_2) const
      { return integral_sgsg(t,T_1,T_2)-integral_sgsg(s,T_1,T_2); }

    /** Diagnostic. Monte Carlo test of the volatility integrals
     *  {@link integral_sgsg(Real,Real,Real,Real)}.
     *   Analytic formulas tested against Monte Carlo over a range of
     *   integration intervals.
     *
     * @param N number of Monte Carlo sample points.
     * @param precision maximal acceptable relative error in percent.
     */
    void testVolSurfaceIntegrals(int N, Real precision);

    /** Sets the parameters (calibration).
     */
    void setParameters(Real a1, Real b1, Real c1, Real d1)
      { a=a1; b=b1; c=c1; d=d1; }

    /** Message and fields.*/

    //RL CHANGE
    //virtual std::ostream& printSelf(std::ostream& os) const = 0;
    virtual std::string as_string() const = 0;

    /** Type of volatility surface "CONST, "JR", "M".
     */
    std::string volSurfaceType();

};                                                // end VolSurface
TO_STRING(VolSurface);

// JAECKEL-REBONATO VOL SURFACE

/** Jaeckel-Rebonato volatility surface. See Jaeckel book, p164, 12.11.
 */
class JR_VolSurface : public VolSurface
{

  public:

    JR_VolSurface
      (Real a0, Real b0, Real c0, Real d0) :
    VolSurface(a0,b0,c0,d0,JR)
      {    }

    // VOLATILITIES, CORRELATIONS, COVARIATION INTEGRALS

    /** Volatility surface
     *  \f$\sigma(t,T)=d+(a+b(T-t)\,e^{-c(T-t)})\f$.
     */
    Real sigma(Real t, Real T) const;

    /** See {@link VolSurface#integral_sgsg}.
     */
    Real integral_sgsg(Real t, Real T_1, Real T_2) const;

    /** Message and fields.*/
    //RL CHANGE
    //std::ostream& printSelf(std::ostream& os) const;
    std::string as_string() const;

    /** Sample surface.*/
    static VolSurface* sample();

};                                                // end JR_VolatilitySurface

TO_STRING(JR_VolSurface);

// OUR OWN VOL SURFACE

/** Our own volatility surface. See book, 6.11.1.
 *  \f[\sigma(t,T)=g(1-t/T), \quad g(s)=1+ase^{-s/d}.\f]
 */
class M_VolSurface : public VolSurface
{

  public:

    M_VolSurface
      (Real a0, Real b0, Real c0, Real d0) :
    VolSurface(a0,b0,c0,d0,M)
      {    }

    // VOLATILITIES, CORRELATIONS, COVARIATION INTEGRALS

    /** Volatility surface (book 6.11.1)
     *  \f[\sigma(t,T)=g(1-t/T), \quad g(s)=1+ase^{-s/d}.\f]
     */
    Real sigma(Real t, Real T) const { return g(1-t/T); }

    /** See {@link VolSurface#integral_sgsg}.
     */
    Real integral_sgsg(Real t, Real T_1, Real T_2) const;

    /** Message and fields.*/
    //RL CHANGE
    //std::ostream& printSelf(std::ostream& os) const;
    std::string as_string() const;

    /** Sample surface.*/
    static VolSurface* sample();

  private:

    // Exponential Integrals

    // integral of exp(s/D)ds
    Real F(Real D, Real s) const;

    // integral of  s*exp(s/D)ds
    Real G(Real D, Real s) const;

    // integral of  s^2*exp(s/D)ds
    Real H(Real D, Real s) const;

    /** The function g(x) defining the volatilities <code>sigma_j</code>
     *  as \f$\sigma_j(t)=c_jg(1-t/T_j)\f$. See book, 6.11.1
     */
    Real g(Real x) const;

};                                                // end M_VolatilitySurface
TO_STRING(M_VolSurface);

// CONSTANT VOLATILITY SURFACE

/** Constant volatility surface.
 */
class CONST_VolSurface : public VolSurface
{

  public:

    CONST_VolSurface(Real a0, Real b0, Real c0, Real d0) :
    VolSurface(a0,b0,c0,d0,CONST)
      {    }

    // VOLATILITIES, CORRELATIONS, COVARIATION INTEGRALS

    /** Volatility surface \f$\sigma(t,T)=1.\f$.
     */
    Real sigma(Real t, Real T) const { return 1; }

    /** See {@link VolSurface#integral_sgsg}.
     */
    Real integral_sgsg(Real t, Real T_1, Real T_2) const { return t; }

    /** Message and fields.*/
    //RL CHANGE
    //std::ostream& printSelf(std::ostream& os) const;
    std::string as_string() const;

    /** Sample (there is only one such surface).
     */
    static VolSurface* sample(){ return new CONST_VolSurface(0.0,0.0,0.0,0.0); }

};                                                // end CONST_VolSurface
TO_STRING(CONST_VolSurface);

/*******************************************************************************
 *
 *                    CORRELATION
 *
 ******************************************************************************/

/** <p>Constant instantaneous correlations of asset returns, log-Libors, ...
 *  To simplify calibration (the same parameters for all types) we assume that
 *  the correlations depend on three Real paramters
 *  \f[\alpha, \beta, r_\infty.\f]
 *
 * If more are needed the calibration routines will have to be rewritten in trivial
 * ways. Only Libor correlations are implemented. To cater to this case the
 * correlations are indexed
 * \f[\rho_{ij},\quad 1\leq i,j<n\f]
 * with index base 1. All matrices, vectors respect this convention.
 * The reason is that Libor \f$L_0\f$ is constant and is not evolved.
 */
class Correlations
{

  protected:

    /** Index range [1,n). */
    int n;

    /** The type of correlations JR, CS
     */
    int corrType;

    Real alpha, beta, r_oo;

    UTRRealMatrix correlationMatrix;

  public:

    /** Correlation type JR, CS. */
    static const int JR=0, CS=1;

    /** CS or JR. */
    int getType() const { return corrType; }

    /** Printing the parameters alpha, beta, r_oo*/
    void printParameters();

    // RL CHANGE
    double getAlpha();
    double getBeta();
    double getRho();

    /** "CS" or "JR", converts integer type ID to string.*/
    static std::string correlationType(int type);

    /** n: index range [1,n). */
    int getDimension() const { return n; }

    /** Correlation matrix is allocated but not initialized
     *  Do this from the concrete subclasses.
     *  @param correlationType JR or CS.
     *  @param alpha see class description.
     */
    Correlations(int n, Real alpha, Real beta, Real r_oo, int correlationType);

    virtual ~Correlations(){ }

    /** A sample correlation in dimension n of type = JR, CS.
     */
    static Correlations* sample(int n, int type);

    /** Update correlation matrix using the current parameters.
     */
    virtual void setCorrelations() = 0;

    /** Correlations \f$\rho_{ij}\f$ for \f$1\leq i,j<n\f$.
     */
    Real& operator()(int i, int j);

    /** The upper half of the correlation matrix
     */
    const UTRRealMatrix& getCorrelationMatrix() const
      { return correlationMatrix; }

    /** Sets the parameters (calibration).
     */
    void setParameters(Real _alpha, Real _beta, Real _r_oo);

    /** Message and fields.
     */
    //RL CHANGE
    //virtual std::ostream& printSelf(std::ostream& os) const = 0;
    virtual std::string as_string() const = 0;

    /** Converts integer type ID to string "JR", "CS".
     */
    std::string correlationType();

};                                                // end Correlations
TO_STRING(Correlations);

// JAECKEL-REBONATO CORRELATIONS

/** Jaeckel-Rebonato correlations of the form
 *  \f[\rho_{ij}=exp(-\beta(T_j-T_i)),\quad 1\leq i\leq j<n.\f]
 */
class JR_Correlations : public Correlations
{

  /** Tenor Structure of Liborprocess. */
  RealArray1D T;

  public:

    /** @param T tenor structure of Libor process.
     *  @param beta see class description.
     */
    JR_Correlations(const RealArray1D& T, Real beta);
    virtual ~JR_Correlations(){ }

    /** Updates correlation matrix using the current parameters.
     */
    void setCorrelations();

    /** Message and fields.*/
    //RL CHANGE
    //std::ostream& printSelf(std::ostream& os) const;
    std::string as_string() const;

    /** Sample correlations.
     *  @param n dimension.
     *  @param delta Libor accrual interval length.
     */
    static Correlations* sample(int n, Real delta=0.25);

};                                                // end JR_Correlations
TO_STRING(JR_Correlations);

// COFFEE-SHOENMAKERS CORRELATIONS

/** Coffee-Shoenmakers correlations. See book, 6.11.1.
 */
class CS_Correlations : public Correlations
{

  public:

    /** @param alpha see class description.
     */
    CS_Correlations(int n, Real alpha, Real beta, Real r_oo);
    virtual ~CS_Correlations(){ }

    /** Updates correlation matrix using the current parameters.
     */
    void setCorrelations();

    /** Message and fields.*/
    //RL CHANGE
    //std::ostream& printSelf(std::ostream& os) const;
    std::string as_string() const;

    /** Sample correlations.
     *  @param n dimension.
     */
    static Correlations* sample(int n);

  private:

    /** The convex function f(x) from which the log-Libor correlations
     *  are derived. See book, 6.11.1.
     */
    Real f(Real x) const;

};                                                // end CS_Correlations
TO_STRING(CS_Correlations);

MTGL_END_NAMESPACE(Martingale)
#endif
