
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef CURVE_H
#define CURVE_H

#include "obj.h"
#include "debug.h"
#include "mapping.h"
#include "util.h"
#include <ql/quantlib.hpp>

#define ENV_FLAT_CURVE "FLAT_CURVE_A360"

class Default {
 public:
  std::string       ccy;
  std::string       instrument;
  double            nominal;
  int               settlementDays;
  int               fixingDays;
  Calendar          calendar;
  BusinessDayConvention roll;
  BusinessDayConvention fixedRoll;
  Frequency         fixedFreq;
  Frequency         floatFreq;
  std::string       floatingTenor;  // generic 3M, 6M, ...
  bool              fixedIsAdjusted;
  DayCounter        mmDayCount;
  DayCounter        cmDayCount;
  DayCounter        actDayCount;
  boost::shared_ptr<Xibor>     index;

  Default(std::string type_,
	  std::string ccy_,
	  Handle<YieldTermStructure> termStructure);
};

class FlatCapFloor {
 public:
  std::string type;     // Cap or Floor
  std::string ccy;
  std::string tenor;    // generic floating tenor 3M, 6M
  std::string term;     // generic maturity 1Y, 2Y, ...
  QL::Date    maturity; // explicit maturity date
  double      strike;
  double      vol;
  double      capPrice;
  double      floorPrice;

  FlatCapFloor (std::string type_, QL::Date start,
		std::string ccy_="",
                std::string tenor_="",
                std::string term_="",
		double strike_=0, double vol_=0)
    : type(type_), ccy(ccy_), tenor(tenor_), term(term_),
      strike(0.01*strike_), vol(0.01*vol_) {
    maturity = parseDate (term, start);
    capPrice   = 0;
    floorPrice = 0;
  }
  void print() {
    printf( "%s %s %s %3s %s %8.4f %8.4f %8.4f %8.4f\n",
	    type.c_str(), ccy.c_str(), tenor.c_str(), term.c_str(),
	    DateFormatter::toString( maturity, DateFormatter::ISO ).c_str(),
	    strike, vol,
	    capPrice, floorPrice);
  }
};

class Caplet {
 public:
  std::string ccy;
  QL::Date start;
  QL::Date end;
  double flatVol;
  // price of cap with maturity = caplet maturity, flatStrike and flatVol
  double flatPrice;
  double strike;
  double price;
  double vol;
  std::vector<boost::shared_ptr<CashFlow> > leg;

  Caplet (std::string ccy_, QL::Date start_, QL::Date end_,
	  double strike_=-1, double price_=-1, double vol_=-1)
    : ccy(ccy_), start(start_), end(end_),
      strike(strike_), price(price_), vol(vol_) {}

  void print() {
    printf( "Caplet %s %s %s %7.4f %7.4f %7.4f %7.4f %7.4f\n",
	    ccy.c_str(),
	    DateFormatter::toString( start, DateFormatter::ISO ).c_str(),
	    DateFormatter::toString( end, DateFormatter::ISO ).c_str(),
	    strike, flatVol, flatPrice, vol, price);
  }
};

class Curve {
 public:
  Handle<YieldTermStructure> ts;

  SwaptionVolatilityMatrix     *volMatrix;
  boost::shared_ptr<YieldTermStructure>  tsDepoSwap;
  boost::shared_ptr<YieldTermStructure>  tsFlat;

  std::list<FlatCapFloor>       caps;
  std::list<Caplet>             caplets;

  Curve (std::string ccy_, XML::list<RL::MarketPoint> *mkt_);

  void link (int i=0) {
    if      (i==0) ts.linkTo( tsFlat );
    else if (i==1) ts.linkTo( tsDepoSwap );
    else {
      std::cout << "usage: link(i) where i=0,1" << std::endl;
      exit(1);
    }
  }

  void test (int N);

  // getters
  std::string ccy ()  { return currency; }
  QL::Date        asof () { return asofdate; }
  double      disc (QL::Date date);
  double      zero (QL::Date date);
  double      fwdvol (QL::Date date);
  double      capletPrice (QL::Date date, double strike);
  double      capletPrice (CapFloor::Type type,
			   const QL::Date& startDate,
			   const QL::Date& endDate,
			   Calendar calendar,
			   double nominal,
			   double strike,
			   boost::shared_ptr<Xibor> index);

  // setters
  int         setCcy (std::string ccy_);
  int         setDate (std::string asof_);
  int         setDate (QL::Date asof_);
  int         generate ();

  boost::shared_ptr<CapFloor> makeCapFloor (CapFloor::Type type,
                      const std::vector<boost::shared_ptr<CashFlow> >& leg,
                      Rate strike,
                      double volatility);
 private:
  XML::list<RL::MarketPoint> * mkt;
  std::string                  previousCurrency;
  std::string                  currency;
  Date                         asofdate;

  bool isUpToDate ();
  void init ();
  bool getRate (double *rate,
                const char *type, const char *ccy, const char *gdate);
  bool getVol (double *rate,
               const char *type, const char *ccy,
	       const char *term, const char *expiry);
  bool getCapVol (double *rate, double *strike,
		  const char *ccy, const char *term, const char *expiry);

  SwaptionVolatilityMatrix *buildSwaptionVolMatrix (std::string ccy,
                                                    QL::Date asof);
  boost::shared_ptr<YieldTermStructure> buildDepoSwapCurve (std::string ccy,
                                                            QL::Date asof);
  boost::shared_ptr<YieldTermStructure> buildFlatCurve (std::string ccy,
                                                        QL::Date asof,
                                                        double rate,
                                                        DayCounter dc);

  std::vector<boost::shared_ptr<CashFlow> > makeLeg (const Date& startDate,
                                                     int length,
                                                     Default& def);
  std::vector<boost::shared_ptr<CashFlow> > makeLeg (const Date& startDate,
                                                     const Date& endDate,
                                                     Default& def);
  boost::shared_ptr<PricingEngine> makeEngine (double volatility);

  boost::shared_ptr<CapFloor> makeCapFloorLet (CapFloor::Type type,
                                               const Date& startDate,
                                               const Date& endDate,
                                               Calendar calendar,
                                               double nominal,
                                               double strike,
                                               boost::shared_ptr<Xibor> index);

  double flatVol      (Date date, int mode);
  double atmStrike    (std::vector<boost::shared_ptr<CashFlow> > leg,
                       double vol);
  double impliedVol   (CapFloor::Type type,
                       std::vector<boost::shared_ptr<CashFlow> > leg,
                       double strike, double value);
  int    buildCaplets (std::string ccy, Date asof, Date start);
};

#endif
