
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


/*! \file z_debug.cpp
    \brief Classes and functions for log message handling.

    \fullpath
    ql/%z_debug.cpp
*/

#include "debug.h"

int          Debug::flag_          = 0;
unsigned int Debug::mask_          = 0;
int          Debug::flush_         = 0;
int          Debug::setup_         = 0;
int          Debug::open_          = 0;
int          Debug::beep_          = 0;
FILE        *Debug::file_          = NULL;
char        *Debug::fileName_      = NULL;
int          Debug::messageNumber_ = 0;
int          Debug::debugNumber_   = 0;

//-----------------------------------------------------------------------------
Debug::Debug( char *debugName, int trunc ) {
//-----------------------------------------------------------------------------

  debugName_ = NULL;
  debugNumber_ ++;

  if (!setup_) {

    //    printf( "Debug: setup called\n" );

    setup_ = 1;

    char *e;

    if ((e = getenv("DEBUG_FILE"))) {
      flag_ = 1;

      if ( strlen( e ) > 200 ) {
	file_ = stderr;
	printf("Debug: filename too long (max 200 char)\n" );
	printf("       output is written to stderr\n");
      } else if ( strcmp(e,"stdout") == 0 ) {
	file_ = stdout;
	printf("Debug: output is written to stdout\n");
      } else if ( strcmp(e,"stderr") == 0 ) {
	file_ = stderr;
	printf("Debug: output is written to stderr\n");
      } else {
	fileName_  = new char [256];

	strcpy( fileName_, e );

	char *pidpos = strstr(fileName_,"%pid");
	if (pidpos) sprintf(pidpos,"%i",(unsigned)getpid());

	if (trunc) file_ = fopen( fileName_, "w" );
	else       file_ = fopen( fileName_, "a+t" );

	if (file_) {
	  printf( "debug messages are redireced to file %s\n", fileName_ );
	  open_ = 1;
	  debugName_ = new char [256];
	  strcpy( debugName_, debugName );
	  msg( "Debug started" );
	} else {
	  printf( "ERR open %s failed\n", fileName_ );
	  open_ = 0;
	}
      }

      if ((e = getenv("DEBUG_BEEP"))) {
	beep_ = 1;
	msg( "beep set" );
      } else {
	beep_ = 0;
	msg( "beep not set" );
      }

      if ((e = getenv("DEBUG_MASK"))) {
	mask_ = strtol(e,NULL,0);
	msg( "external mask set to %d", mask_ );
      } else {
	mask_ = 0;
	msg( "external mask not set (%d)", mask_ );
      }

      if ((e = getenv("DEBUG_FLUSH"))) {
	flush_ = 1;
	msg( "flush set" );
      } else {
	flush_ = 0;
	msg( "flush not set" );
      }

    } else {
      flag_ = 0; // do not generate debug messages at all
      printf( "+--------------------------------------------------------------+\n" );
      printf( "| environment variable DEBUG_FILE not set => no debug messages |\n" );
      printf( "+--------------------------------------------------------------+\n" );
    }
  }

  if (flag_ && debugName && !debugName_) {
    debugName_ = new char [256];
    strcpy( debugName_, debugName );
  }
}

//-----------------------------------------------------------------------------
void Debug::enableBeep() {
//-----------------------------------------------------------------------------

  beep_ = 1;
}

//-----------------------------------------------------------------------------
void Debug::disableBeep() {
//-----------------------------------------------------------------------------

  beep_ = 0;
}

//-----------------------------------------------------------------------------
int Debug::isEnabled() {
//-----------------------------------------------------------------------------

  return flag_;
}

//-----------------------------------------------------------------------------
char* Debug::getFileName() {
//-----------------------------------------------------------------------------

  return fileName_;
}

//-----------------------------------------------------------------------------
void Debug::msg( unsigned long mask, char *format, ... ) {
//-----------------------------------------------------------------------------

  if (!flag_) return;

  time_t     t;
  struct tm *tm;

  if (!format)
    format = "ERR (nil) format";

  va_list   ap;
  va_start (ap,format);

  if ((file_) && (mask & mask_)) {

    messageNumber_ ++;

    time(&t);
    tm = localtime(&t);
    fprintf  (file_,"[%02i%02i%02i:%02i%02i%02i] [%3lu] [%d-%s] ",
	      tm->tm_year%100,
	      tm->tm_mon + 1,
	      tm->tm_mday,
	      tm->tm_hour,
	      tm->tm_min,
	      tm->tm_sec,
	      mask,
	      debugNumber_,
	      debugName_);
    vfprintf (file_,format,ap);
    fprintf  (file_,"\n");

    if (flush_) fflush(file_);
  }

  if (mask & beep_) fprintf(stderr,"%c",7);

  va_end(ap);

  if (mask == ALERT) exit(-1);
}

//-----------------------------------------------------------------------------
void Debug::msg( char *format, ... ) {
//-----------------------------------------------------------------------------

  if (!flag_) return;

  time_t     t;
  struct tm *tm;

  if (!format)
    format = "ERR (nil) format";

  va_list   ap;
  va_start (ap,format);

  if (file_) {

    messageNumber_ ++;

    time(&t);
    tm = localtime(&t);
    fprintf  (file_,"[%02i%02i%02i:%02i%02i%02i] [   ] [%d-%s] ",
	      tm->tm_year%100,
	      tm->tm_mon + 1,
	      tm->tm_mday,
	      tm->tm_hour,
	      tm->tm_min,
	      tm->tm_sec,
	      debugNumber_,
	      debugName_);
    vfprintf (file_,format,ap);
    fprintf  (file_,"\n");

    if (flush_) fflush(file_);
  }

  if (beep_) fprintf(stderr,"%c",7);

  va_end(ap);
}

// eof
