
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*! \file debug.h
    \brief Classes and functions for log message handling.

    \fullpath
    ql/%debug.h
*/

#ifndef debug_h
#define debug_h

                    //          accumulated 'filter' for 'external' DEBUG_MASK
#define ALERT     1 // 00000001   1 = 2^1-1
#define CRITICAL  2 // 00000010   3 = 2^2-1
#define ERROR     4 // 00000100   7
#define WARNING   8 // 00001000  15
#define NOTICE   16 // 00010000  31
#define DEBUG    32 // 00100000  63 = 2^6-1
#define DATA     64 // 01000000  127

#define CATCH_ALERT \
 catch (std::exception& e) { log.msg (ALERT, "%s, exit.", e.what()); } \
 catch (...)               { log.msg (ALERT, "unknown error, exit."); }
#define CATCH_CRITICAL \
 catch (std::exception& e) { log.msg (CRITICAL, "%s, exit.", e.what()); } \
 catch (...)               { log.msg (CRITICAL, "unknown error, exit."); }
#define CATCH_ERROR \
 catch (std::exception& e) { log.msg (ERROR, "%s, exit.", e.what()); } \
 catch (...)               { log.msg (ERROR, "unknown error, exit."); }

#define CATCH(L,M,T) \
 try { T; } \
 catch (std::exception& e) { L.msg (M, "%s, exit.", e.what()); } \
 catch (...)               { L.msg (M, "unknown error, exit."); }

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>

/*! \file debug.h
    \brief class for generating debug messages
           local instances should be created like
           'static Debug instance("name")'
           at the beginning of each function/method
           where debugging messages are required

    \fullpath
    ql/%debug.h
*/

class Debug
{
  /*! local instances of Debug should be created like
      'static Debug instance("name")'
      at the beginning of each function/method
      where debugging messages are required,
      'static' to create only one instance per function
  */
private:
  static unsigned int mask_;          // binary pattern from env DEBUG_MASK
  static int          flush_;         // from env DEBUG_FLUSH (0,1)
  static int          setup_;         // when initialised (1), not yet (0)
  static int          open_;          // when file is open (1), if open failed (0)
  static int          beep_;          // from env DEBUG_BEEP (0,1)
  static FILE        *file_;          // file pointer
  static char        *fileName_;      // from env DEBUG_FILE
  static int          messageNumber_; // sequence number of message
  static int          debugNumber_;   // sequence number of Debug class instance

  char               *debugName_;     // function or procedure name

public:
  static int          flag_;          // indicates whether debugging is turned on at all
                                      // depends on whether env DEBUG_FILE is set
                                      // choices are: stdout, stderr, <filename>
  Debug( char *debugName="", int trunc=0 );

  void msg( unsigned long mask, char *format, ... );
  void msg( char *format, ... );
  void enableBeep(void);
  void disableBeep(void);
  char* getFileName();
  int  isEnabled();
};

#endif
