
/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*! \file engine.cpp
    \brief Provide pricing functions: Set up LMM using structure of the
    instrument to be priced, invoke calibration, MC pricing using LMM,
    price instruments and legs using deterministic cash flow projection and
    discounting, apply black scholes pricing to cap/floor features.
    Provide curve and calibration tests.

   Hierarchy of calls to pricing functions:

   price (inst) - set up model
     |
     +- call legPricer (inst)
     |      |
     |    call price (leg): price each leg and add resulting PVs
     |      |
     |    priceFixedLeg (), priceFixedLegLMM ()
     |    priceFloatLeg (), priceFloatLegLMM ()
     |                      (each leg generates its own MC paths)
     |
     +- if ENV_MODEL = LMM, call pathPricer (inst)
        generate MC paths and call the following pricers for each path
        aggregate results across all instrument legs
            |
          pathPricerFixed (), pathPricerFloat ()
 */

#ifndef ENGINE_H
#define ENGINE_H

#include "obj.h"
#include "debug.h"
#include "mapping.h"
#include "curve.h"

#include <ql/quantlib.hpp>

#include <martingale.h>

namespace RL {

class Engine {
 private:

  QL::Date                      asof;
  XML::list<RL::MarketPoint> * mkt;
  Curve                         curve;

  std::string model;

  LmmParameters lmmParams;

  double max(double a, double b);
  double min(double a, double b);
  int    max(int a, int b);
  int    min(int a, int b);
  double fak(int n);
  double binko(int n, int k);
  int    binomialPrice( double *res, int type,
			double flt, double fix, double vol,
			double mat, int iMat, double *expect,
			int iTime, int iNode );

  boost::shared_ptr<SimpleSwap> makeSimpleSwap (
    const QL::Date& start,
    int length, QL::TimeUnit unit,
    double nominal,
    QL::Rate fixedRate,
    QL::Spread floatingSpread,
    bool payFixed,
    const char * ccy,
    const char * name = "");

  boost::shared_ptr<SimpleSwap> makeSimpleAtmSwap (
    const QL::Date& start,
    int length, TimeUnit unit,
    QL::Spread floatingSpread,
    bool payFixed,
    const char * ccy,
    const char * name = "");

  boost::shared_ptr<QL::Swaption> makeSwaption (
    const boost::shared_ptr<SimpleSwap>&,
    const QL::Date& exercise,
    double volatility);

  double getBinomialEuropeanSwaptionPrice (
    int type,
    const boost::shared_ptr<SimpleSwap>& swap,
    const QL::Date& exercise,
    double volatility,
    int    steps = 0);

  double getBinomialBermudanSwaptionPrice (
    int type,
    std::vector<boost::shared_ptr<SimpleSwap> > & swaps,
    std::vector<QL::Date> & exercises,
    std::vector<double> vols,
    int    steps,
    Default &def );

  std::vector<boost::shared_ptr<CashFlow> > cashFlowVector (RL::Leg& leg);

  MTG::LiborMarketModel* setupModel (
    std::vector<boost::shared_ptr<QL::CashFlow> > cashflow);

  int pathPricer (RL::Instrument& inst,
                  MTG::LiborMarketModel* lmm);

  int legPricer (RL::Instrument& inst,
                 MTG::LiborMarketModel* lmm = NULL);

  int pathPricerFixed (MTG::RealArray1D& EDCF,
                       RL::Leg& leg,
                       std::vector<boost::shared_ptr<CashFlow> >& cfv,
                       MTG::LiborMarketModel* lmm,
                       int path, int pmin, int pmax);

  int pathPricerFloat (MTG::RealArray1D& EDCF,
                       RL::Leg& leg,
                       std::vector<boost::shared_ptr<CashFlow> >& cfv,
                       MTG::LiborMarketModel* lmm,
                       int path, int pmin, int pmax);

  int priceFixedLeg (RL::Leg& leg);

  int priceFixedLegLMM (RL::Leg& leg,
                        MTG::LiborMarketModel* lmm = NULL);

  int priceFloatLeg (RL::Leg& leg);

  int priceFloatLegLMM (RL::Leg& leg,
                        MTG::LiborMarketModel* lmm = NULL);

 public:

  Engine (XML::list<RL::MarketPoint> *mkt_);

  // use calibration instruments adjusted to reset periods of "inst"
  MTG::LiborMarketModel* setupModel (RL::Instrument& inst);

  // use externaly provided calibration instruments
  MTG::LiborMarketModel* setupModel ();

  int price (RL::Instrument& inst, MTG::LiborMarketModel* lmm = NULL);
  int price (RL::Leg& leg, MTG::LiborMarketModel* lmm = NULL);

  void setCcy (std::string ccy_)   { curve.setCcy (ccy_); }

  void setDate (std::string asof_) {
    curve.setDate (asof_);
    asof = parseDate(asof_);
  }

  void setDate (QL::Date asof_) {
    curve.setDate (asof_);
    asof = asof_;
  }

  int testCurve ();
  int testSwap ();
  int testBinomial ();

  // my own Lmm tests
  void testLmm();
  MTG::LiborFactorLoading* testLmmCalibration();

  // Martingale test functions from TestLMM.h

  void testLiborFactorLoading (int n);

  void testLiborFactorLoadingFactorization(int n, int r);

  void testLmmPaths(int n);

  void testSwaptionPrice(int lmmType, int volType, int corrType);

  void testCapletPrice(int lmmType, int volType, int corrType);

  void testCallOnBondPrice(int lmmType, int volType, int corrType);

  void testCallOnZeroCouponBondPrice(int lmmType, int volType, int corrType);

  void testLmmLattice();

  void testSwaption();

  void testBermudanSwaption();

  void testCallOnBond();

  void testCallOnZeroCouponBond();

  void testCaplet();

  void testLiborDerivative();
};

} // end of namespace RL

#endif
