/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "obj.h"

int RL::Formula::nvar = 0;
int RL::Formula::npop = 0;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Schedule::init() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  custom        = "";
  startDate     = "";
  endDate       = "";
  stubType      = "";
  startStubDate = "";
  endStubDate   = "";
  frequency     = "";
  adjusted      = "";
  bdRule        = "";
  calendar      = "";
  gapDays       = 0;
  gapDaysType   = "";
  dates.init();
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RL::Schedule::isValid() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if (dates.size() > 0 ||
      (startDate != "" &&
       endDate != "" &&
       frequency != "" &&
       adjusted != "" &&
       bdRule != "" &&
       calendar != "") ) return  0;
  else                   return -1;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Schedule::Schedule( std::string startDate_,
			  std::string endDate_,
			  std::string stubType_,
			  std::string startStubDate_,
			  std::string endStubDate_,
			  std::string frequency_,
			  std::string adjusted_,
			  std::string bdRule_,
			  std::string calendar_,
			  int         gapDays_,
			  std::string gapDaysType_)
  : startDate(startDate_),
    endDate(endDate_),
    stubType(stubType_),
    startStubDate(startStubDate_),
    endStubDate(endStubDate_),
    frequency(frequency_),
    adjusted(adjusted_),
    bdRule(bdRule_),
    calendar(calendar_),
    gapDays(gapDays_),
    gapDaysType(gapDaysType_)
{
  custom = "N";
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Schedule::Schedule( XML::list<Date> dates_ ) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  init();
  custom = "Y";
  dates = dates_;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Schedule::set(const Schedule& sched_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  custom    = sched_.custom;
  startDate = sched_.startDate;
  endDate   = sched_.endDate;
  stubType  = sched_.stubType;
  startStubDate = sched_.startStubDate;
  endStubDate   = sched_.endStubDate;
  frequency     = sched_.frequency;
  adjusted      = sched_.adjusted;
  bdRule        = sched_.bdRule;
  calendar      = sched_.calendar;
  gapDays       = sched_.gapDays;
  gapDaysType   = sched_.gapDaysType;
  dates         = sched_.dates;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Schedule::Schedule( const Schedule& sched_ ) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set(sched_);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Schedule& RL::Schedule::operator=(const Schedule& sched_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set(sched_);
  return *this;
}

//--------------------------------------------------------------------------------------------------
RL::Index::Index( const Index& index_ )
//--------------------------------------------------------------------------------------------------
  : type(index_.type),
    ccy(index_.ccy),
    dayCount(index_.dayCount),
    resetSched(index_.resetSched),
    rate(index_.rate),
    name(index_.name),
    term(index_.term),
    spread(index_.spread),
    fixTiming(index_.fixTiming),
    fixGap(index_.fixGap),
    fixSched(index_.fixSched),
    isCapped(index_.isCapped),
    cap(index_.cap),
    isFloored(index_.isFloored),
    floor(index_.floor),
    formulas(index_.formulas)
{}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// generic constructor
RL::Index::Index( std::string type_,
		   std::string ccy_,
		   std::string dayCount_,
		   Schedule resetSched_,
		   double rate_,
		   std::string name_,
		   std::string term_,
		   double spread_,
		   std::string fixTiming_,
		   int fixGap_,
		   Schedule fixSched_,
		   std::string isCapped_,
		   double cap_,
		   std::string isFloored_,
		   double floor_,
		   XML::list<Formula> forms_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  type        = type_;
  ccy         = ccy_;
  dayCount    = dayCount_;
  resetSched  = resetSched_;

  rate        = rate_;

  name        = name_;
  term        = term_;
  spread      = spread_;
  fixTiming   = fixTiming_;
  fixGap      = fixGap_,
  fixSched    = fixSched_;
  isCapped    = isCapped_;
  cap         = cap_;
  isFloored   = isFloored_;
  floor       = floor_;

  formulas    = forms_;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// fixed index
RL::Index::Index( std::string type_,
		   std::string ccy_,
		   std::string dayCount_,
		   Schedule resetSched_,
		   double rate_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  init();

  if (type_ != "FIXED") {
    printf( "Index::Index This constructor needs type FIXED\n" );
    exit(-1);
  }

  type        = type_;
  ccy         = ccy_;
  dayCount    = dayCount_;
  resetSched  = resetSched_;

  rate        = rate_;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// plain vanilla floating index
RL::Index::Index( std::string type_,
		   std::string ccy_,
		   std::string dayCount_,
		   Schedule resetSched_,
		   std::string name_,
		   std::string term_,
		   double spread_,
		   std::string fixTiming_,
		   int         fixGap_,
		   Schedule    fixSched_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  init();

  if (type_ != "FLOAT") {
    printf( "Index::Index This constructor needs type FLOAT\n" );
    exit(-1);
  }

  type        = type_;
  ccy         = ccy_;
  dayCount    = dayCount_;
  resetSched  = resetSched_;

  name        = name_;
  term        = term_;
  spread      = spread_;
  fixTiming   = fixTiming_;
  fixGap      = fixGap_;
  fixSched    = fixSched_;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// capped/floored floating index
RL::Index::Index( std::string type_,
		   std::string ccy_,
		   std::string dayCount_,
		   Schedule    resetSched_,
		   std::string name_,
		   std::string term_,
		   double      spread_,
		   std::string fixTiming_,
		   int         fixGap_,
		   Schedule    fixSched_,
		   std::string isCapped_,
		   double      cap_,
		   std::string isFloored_,
		   double      floor_ ) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  init();

  if (type_ != "FLOATCF") {
    printf( "Index::Index This constructor needs type FLOAT\n" );
    exit(-1);
  }

  type        = type_;
  ccy         = ccy_;
  dayCount    = dayCount_;
  resetSched  = resetSched_;

  rate        = 0;

  name        = name_;
  term        = term_;
  spread      = spread_;
  fixTiming   = fixTiming_;
  fixGap      = fixGap_;
  fixSched    = fixSched_;
  isCapped    = isCapped_;
  cap         = cap_;
  isFloored   = isFloored_;
  floor       = floor_;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// formula index
RL::Index::Index( std::string type_,
		   std::string ccy_,
		   std::string dayCount_,
		   Schedule    resetSched_,
		   std::string name_,
		   std::string term_,
		   double      spread_,
		   std::string fixTiming_,
		   int         fixGap_,
		   Schedule    fixSched_,
		   XML::list<Formula> forms_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  init();

  if (type_ != "FORM") {
    printf( "Index::Index This constructor needs type FLOAT\n" );
    exit(-1);
  }

  type        = type_;
  ccy         = ccy_;
  dayCount    = dayCount_;
  resetSched  = resetSched_;

  name        = name_;
  term        = term_;
  spread      = spread_;
  fixTiming   = fixTiming_;
  fixGap      = fixGap_;
  fixSched    = fixSched_;

  formulas    = forms_;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Index::init() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  type      = "";
  ccy       = "";
  dayCount  = "";
  resetSched.init();

  rate      = 0;

  name      = "";
  term      = "";
  spread    = 0;
  fixTiming = "";
  fixGap    = 0;
  fixSched.init();
  isCapped  = "";
  cap       = 0;
  isFloored = "";
  floor     = 0;

  formulas.init();
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RL::Index::isValid () {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if ( type != "" &&
       ccy != "" &&
       dayCount != "" &&
       resetSched.isValid() >= 0 &&
       ( (type == "FIXED" && rate > 0) ||
	 ((type == "FLOAT" || type =="FORM") &&
	  name != "" &&
	  term != "" &&
	  (fixTiming == "ARR" || fixTiming == "ADV" || fixSched.isValid() >= 0) ) ) )
    return 0;
  else
    return -1;
}

//--------------------------------------------------------------------------------------------------
RL::Stats::Stats (const RL::Stats& stats_ ) {
//--------------------------------------------------------------------------------------------------
  ccy     = stats_.ccy;
  npv     = stats_.npv;
  gpv     = stats_.gpv;
  acc     = stats_.acc;
  pv01    = stats_.pv01;
  bpv     = stats_.bpv;
}

//--------------------------------------------------------------------------------------------------
RL::Stats& RL::Stats::operator = (const RL::Stats& stats_ ) {
//--------------------------------------------------------------------------------------------------
  ccy     = stats_.ccy;
  npv     = stats_.npv;
  gpv     = stats_.gpv;
  acc     = stats_.acc;
  pv01    = stats_.pv01;
  bpv     = stats_.bpv;

  return *this;
}

//--------------------------------------------------------------------------------------------------
RL::Leg::Leg (const Leg& leg_) {
//--------------------------------------------------------------------------------------------------
  set(leg_);

  if (_TRACE_) {
    //    printf( "CREATE Leg(&) %p %s\n", this, id.c_str() );
  }
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Leg& RL::Leg::operator = (const Leg& leg_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set(leg_);

  if (_TRACE_) {
    //printf( "CREATE Leg(=) %p %s\n", this, id.c_str() );
  }

  return *this;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Leg::set (const Leg& leg_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  id       = leg_.id;
  type     = leg_.type;
  ccy      = leg_.ccy;
  notional = leg_.notional;
  index    = leg_.index;
  payTiming = leg_.payTiming;
  payGap   = leg_.payGap;
  paySched = leg_.paySched;
  flows    = leg_.flows;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Leg::Leg( Leg * leg_ ) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if (leg_) {
    id       = leg_->getId();
    type     = leg_->getType();
    ccy      = leg_->getCcy();
    notional = leg_->getNotional();
    index    = leg_->getIndex();
    payGap   = leg_->getPayGap();
    payTiming= leg_->getPayTiming();
    paySched = leg_->getPaySched();
    flows    = leg_->getFlows();
  }
  else init();

  if (_TRACE_) {
    printf( "CREATE Leg(*) %p %s\n", this, id.c_str() );
  }
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Leg::Leg( std::string id_, std::string type_, std::string ccy_,
	       XML::list<Notional> notional_, XML::ptr<Index> index_,
	       std::string payTiming_, int payGap_, Schedule& paySched_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  id        = id_;
  type      = type_;
  ccy       = ccy_;
  notional  = notional_;
  index     = index_;
  payTiming = payTiming_;
  payGap    = payGap_;
  paySched  = paySched_;
  flows.init();

  if (_TRACE_) {
    printf( "CREATE Leg(.) %p %s\n", this, id.c_str() );
  }
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Leg::init() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  id   = "";
  type = "";
  ccy  = "";
  notional.init();
  index.init();
  payTiming = "";
  payGap = 0;
  paySched.init();
  flows.init();
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RL::Leg::isValid() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // FIXME
  if (id != "") return 0;
  else          return -1;
}

//--------------------------------------------------------------------------------------------------
void RL::Structure::set (const RL::Structure& str_ ) {
//--------------------------------------------------------------------------------------------------
  type          = str_.type;
  rightTo       = str_.rightTo;
  barrierPolicy = str_.barrierPolicy;
  exerciseSched = str_.exerciseSched;
  noticeSched   = str_.noticeSched;
  underlying    = str_.underlying;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Structure::init () {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  type          = "";
  rightTo       = "";
  barrierPolicy = "";
  exerciseSched.init();
  noticeSched.init();
  underlying.init();
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RL::Structure::isValid() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // FIXME
  if (type != "") return 0;
  else            return -1;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Structure::Structure () {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  init();
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Structure::Structure (const Structure& str_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set (str_);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Structure& RL::Structure::Structure::operator = (const Structure& str_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set (str_);
  return *this;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Structure::Structure (std::string type_,
			    std::string rightTo_,
			    std::string barrierPolicy_,
			    Schedule exerciseSched_,
			    Schedule noticeSched_,
			    XML::ptr<Instrument> underlying_ )
  : underlying (underlying_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  type          = type_;
  rightTo       = rightTo_;
  barrierPolicy = barrierPolicy_;
  exerciseSched = exerciseSched_;
  noticeSched   = noticeSched_;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Structure::Structure (std::string type_,
			    std::string rightTo_,
			    std::string barrierPolicy_,
			    Schedule exerciseSched_,
			    Schedule noticeSched_)
  : underlying () {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  type          = type_;
  rightTo       = rightTo_;
  barrierPolicy = barrierPolicy_;
  exerciseSched = exerciseSched_;
  noticeSched   = noticeSched_;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Structure::Structure (std::string type_) : underlying () {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  type          = type_;
  rightTo       = "";
  barrierPolicy = "";
  exerciseSched.init();
  noticeSched.init();
}

//--------------------------------------------------------------------------------------------------
void RL::Instrument::set(const RL::Instrument& inst_) {
//--------------------------------------------------------------------------------------------------
  id         = inst_.id;
  level      = inst_.level;
  ccy        = inst_.ccy;
  legs.init();
  legs       = inst_.legs;
  structure  = inst_.structure;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Instrument::Instrument(const RL::Instrument& inst_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set(inst_);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Instrument& RL::Instrument::operator=(const RL::Instrument& inst_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  set(inst_);
  return *this;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Instrument::Instrument( XML::ptr<RL::Instrument> inst_ )
  : structure (inst_->structure) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if (inst_) {
    id   = inst_->getId();
    ccy  = inst_->getCcy();
    legs = inst_->getLegs();
    level = -1;
  }
  else init();

  trace("ptr<Instrument>");
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
RL::Instrument::Instrument( std::string id_,
			      int level_,
			      std::string ccy_,
			      XML::list<Leg> legs_,
			      Structure structure_ )
  : structure (structure_) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  id    = id_;
  level = level_;
  ccy   = ccy_;
  legs  = legs_;
  trace("..., list<Leg>");
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Instrument::init() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  id    = "";
  level = 0;
  ccy   = "";
  legs.init();
  structure.init();
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
int RL::Instrument::isValid() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //FIXME
  if ( id != "" )
    return 0;
  else
    return -1;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Instrument::trace(const char *s) {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if (_TRACE_) {
    printf( "CREATE Instrument(%s) %p\n",  s, this );
    printf( "          id         %s\n",   id.c_str() );
    printf( "          ccy        %s\n",   ccy.c_str() );
    printf( "          legs       %ld\n",  legs.size() );
    printf( "          type       %s\n",   structure.getType().c_str() );
//    printf( "          underlying %p\n",   underlying.get() );
  }
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void RL::Instrument::print() {
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  printf( "Instrument\n" );
  printf( "----------\n" );
  printf( "id      = %s\n",   id.c_str() );
  printf( "address = %p\n",   this );
  printf( "level   = %d\n",   getLevel() );
  printf( "type    = %s\n",   structure.getType().c_str() );
  printf( "ccy     = %s\n",   ccy.c_str() );
  printf( "# legs  = %ld ",    getLegs().size() );

  XML::list<Leg> l = getLegs();
  XML::list<Leg>::iterator it;
  for( it = l.begin(); it != l.end(); it++ ) {
    printf( "(%s,%s) ", it->getId().c_str(), it->getCcy().c_str() );
  }
  printf( "\n" );
  printf( "underl. = %p\n", getUnderlying().get() );

  if (getUnderlying().use_count() > 0) {
    printf( "\n" );
    printf( "=== Underlying ===>\n\n" );
    getUnderlying().get()->print();
  }
  else printf( "\n" );
}

// eof
