/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#ifndef OBJ_H
#define OBJ_H

#define _TRACE_ 0

#include <stdlib.h>
#include <stdio.h>
#include <list>
#include <typeinfo>
#include <string>

#include "xml.h"
#include "formula.h"

namespace RL {

class Engine;
//class MainEngine;

//-----------------------------------------------------------------------------------------------
class MarketPoint : public XmlObject {
//-----------------------------------------------------------------------------------------------
 protected:

 public:
  std::string type;   // Deposit, Swap, Swaption, Cap
  std::string ccy;
  std::string term;   // generic term/tenor (2D, 3M, 2Y, ... )
  std::string expiry; // generic expiry/maturity (2D, 3M, 2Y, ... ), blank for Deposit and Swap
  double      rate;   // interest rate or volatility / percentage
  double      strike; // strike interest rate, 0 assumes ATM

  MarketPoint() {
    init();
  }

  MarketPoint( std::string type_, std::string expiry_,
	       std::string ccy_ = "EUR", std::string term_ = "2M",
	       double rate_ = 3.14, double strike_ = 0.0 )
    : type(type_), ccy(ccy_), term(term_), expiry(expiry_), rate(rate_), strike(strike_) {}

  void init() {
    type    = "";
    expiry  = "";
    ccy     = "";
    term    = "";
    rate    = 0;
    strike  = 0;
  }

  int isValid() {
    if (type == "") return -1;
    return 0;
  }

  //  const char * getType() { return "MarketPoint"; }
  //  const char * getName() { return "N/A"; }

  // XML
  XML_TAG t_type;
  XML_TAG t_expiry;
  XML_TAG t_term;
  XML_TAG t_ccy;
  XML_TAG t_rate;
  XML_TAG t_strike;

  XML_IO;
};

//-----------------------------------------------------------------------------------------------
class String : public XmlObject, public std::string {
//-----------------------------------------------------------------------------------------------
 public:
  String () { clear(); }
  String (const String& s)                  { assign(s); }
  String (std::string s)                    { assign(s); }
  String& operator = (const String& s)      { assign(s); return *this; }
  String& operator = (const std::string& s) { assign(s); return *this; }

  void init ()    { clear(); }
  int  isValid () { return *this == "" ? -1 : 0; }

  XML_IO;
};

//-----------------------------------------------------------------------------------------------
class Date : public XmlObject {
//-----------------------------------------------------------------------------------------------
 protected:
  std::string iso;
  std::string gen;

 public:
  Date() { init(); }
  Date( std::string iso_ ) : iso(iso_) {}
  Date( const char *s ) : iso(s) {}
  Date& operator=(const Date& iso_) { iso = iso_.iso; return *this; }

  void        set( std::string iso_ ) { iso = iso_; }
  std::string get() { return iso; }

  // XML
  void init()    { iso = ""; gen = ""; }
  int  isValid() { return iso == "" ? -1 : 0; }

  XML_TAG t_iso;
  XML_TAG t_gen;

  XML_IO;
};

//-----------------------------------------------------------------------------------------------
class Amount : public XmlObject {
//-----------------------------------------------------------------------------------------------
 protected:
   double      amt;
   std::string ccy;

 public:
   Amount () { init(); }
   Amount (double amt_, std::string ccy_) : amt(amt_), ccy(ccy_) {}
   Amount& operator = (const Amount& n_) {
     amt = n_.amt;
     ccy = n_.ccy;
     return *this;
   }

   void        set (double amt_, std::string ccy_) { amt = amt_; ccy = ccy_; }
   double      getAmt () { return amt; }
   std::string getCcy () { return ccy; }

  // XML
  void init ()    { set (0, ""); }
  int  isValid () { return getCcy () == "" ? -1 : 0; }

  XML_TAG t_amt;
  XML_TAG t_ccy;

  XML_IO;
};

//-----------------------------------------------------------------------------------------------
class Notional : public XmlObject {
//-----------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

 protected:
  double      amt;
  std::string ccy;

 public:
  Notional () { init(); }
  Notional (double amt_, std::string ccy_) : amt(amt_), ccy(ccy_) {}
  Notional& operator = (const Notional& n_) { amt = n_.amt; ccy = n_.ccy; return *this; }

  void        set (double amt_, std::string ccy_) { amt = amt_; ccy = ccy_; }
  double      getAmt () { return amt; }
  std::string getCcy () { return ccy; }

  // XML
  void init ()    { set (0, ""); }
  int  isValid () { return getCcy () == "" ? -1 : 0; }

  XML_TAG t_amt;
  XML_TAG t_ccy;

  XML_IO;
};

//-----------------------------------------------------------------------------------------------
class Payment : public XmlObject {
//-----------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

 protected:
  double      amt;
  std::string ccy;

 public:
  Payment () { init(); }
  Payment (double amt_, std::string ccy_) : amt(amt_), ccy(ccy_) {}
  Payment& operator = (const Payment& n_) { amt = n_.amt; ccy = n_.ccy; return *this; }

  void        set (double amt_, std::string ccy_) { amt = amt_; ccy = ccy_; }
  double      getAmt () { return amt; }
  std::string getCcy () { return ccy; }

  // XML
  void init ()    { set (0, ""); }
  int  isValid () { return getCcy () == "" ? -1 : 0; }

  XML_TAG t_amt;
  XML_TAG t_ccy;

  XML_IO;
};

//-----------------------------------------------------------------------------------------------
class Rate : public XmlObject {
//-----------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

 protected:
   double rate;
   double units; // absolute 1, percent 0.01, basis points 0.0001

 public:
   Rate () { init(); }
   Rate (double rate_, double units_) { rate = rate_; units = units_; }
   Rate& operator = (const Rate& rate_) { rate = rate_.rate; units = rate_.units; return *this; }

   void   set (double rate_, double units_) { rate = rate_; units = units_; }
   double getRate ()  { return rate; }
   double getUnits () { return units; }

  // XML
  void init()    { rate = 0; units = 0; }
  int  isValid() { return units < 1e-6 ? -1 : 0; }

  XML_TAG t_rate;
  XML_TAG t_units;
  XML_IO;
};

//-----------------------------------------------------------------------------------------------
class Spread : public XmlObject {
//-----------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

 protected:
   double rate;
   double units; // absolute 1, percent 0.01, basis points 0.0001

 public:
   Spread () { init(); }
   Spread (double spread_) { set(spread_); }
   Spread& operator = (const Spread& spread_) { set(spread_.rate); return *this; }

   void set(double spread_, double units_ = 0.0001) { rate = spread_; units = units_; }
   double get() { return rate; }

  // XML
  void init()    { set(0.,0.); }
  int  isValid() { return units < 1e-6 ? -1 : 0; }

  XML_TAG t_spread;
  XML_TAG t_units;

  XML_IO;
};


//-----------------------------------------------------------------------------------------------
class Schedule : public XmlObject {
//-----------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

  // protected:
 public:
  std::string custom;

  std::string startDate;
  std::string endDate;
  std::string stubType;
  std::string startStubDate;
  std::string endStubDate;
  std::string frequency;
  std::string adjusted; // if yes, dates are adjusted according to holiday calendar and bdRule
  std::string bdRule;
  std::string calendar;
  int         gapDays;
  std::string gapDaysType;

  XML::list<Date> dates; // custom

  void set (const Schedule& sched);

 public:

  Schedule () { init(); }
  Schedule (const Schedule&);
  Schedule& operator = (const Schedule&);
  Schedule (XML::ptr<Schedule>);

  Schedule( std::string startDate_,
	    std::string endDate_,
	    std::string stubType_,
	    std::string startStubDate_,
	    std::string endStubDate_,
	    std::string frequency_,
	    std::string adjusted_,
	    std::string bdRule_,
	    std::string calendar_,
	    int         gapDays,
	    std::string gapDaysType);

  Schedule( XML::list<Date> dates_ );

  // XML extension

  void init();
  int  isValid();

  XML_TAG t_custom;
  XML_TAG t_startDate;
  XML_TAG t_endDate;
  XML_TAG t_stubType;
  XML_TAG t_startStubDate;
  XML_TAG t_endStubDate;
  XML_TAG t_frequency;
  XML_TAG t_adjusted;
  XML_TAG t_bdRule;
  XML_TAG t_calendar;
  XML_TAG t_dates;
  XML_TAG t_gapDays;
  XML_TAG t_gapDaysType;

  // attributes
  // none

  XML_IO;
};


//-----------------------------------------------------------------------------------------------
class Formula : public XmlObject {
//-----------------------------------------------------------------------------------------------
// class Engine needs full access to all members for pricing
  friend class Engine;

 protected:
  std::string date;
  std::string formula;

  double      index;
  PRVar       var;
  PROperation pop;

  void setup(const char *name) {
    if (!var) {
      //printf( "create var/pop from %s\n", name ); fflush(stdout);
      var = new RVar( "index" , &index );
      pop = new ROperation ( (char*)formula.c_str(), 1, &var );
      nvar++;
      npop++;
    }
    else if (!pop) {
      //printf( "create pop from %s\n", name ); fflush(stdout);
      pop = new ROperation ( (char*)formula.c_str(), 1, &var );
      npop++;
    }
  }

 public:
  static int nvar;
  static int npop;

  Formula () { init(); }

  Formula (const Formula& f) {
    init();
    date    = f.date;
    formula = f.formula;
    setup("Formula(Formula&)");
  }

  Formula (std::string date_, std::string form_) {
    init();
    date    = date_;
    formula = form_;
    setup("Formula(string,string)");
  }

  Formula& operator = (const Formula& f) {
    init();
    date    = f.date;
    formula = f.formula;
    setup("Formula& operator=(Formula&)");
    return *this;
  }

  ~Formula() {

    //if (pop) delete pop;
    //if (var) delete var;
  }

  double eval (double index_) {
    setup("eval");
    index = index_;
    return pop->Val();
  }

  void init ()    {
    date.clear();
    formula.clear();
    index = 0;
    //if (pop) delete pop;
    pop = NULL;
    //if (var) delete var;
    var = NULL;
  }

  int  isValid () { return (date == "" || formula == "") ? -1 : 0; }

  XML_TAG t_date;
  XML_TAG t_formula;

  XML_IO;
};

//------------------------------------------------------------------------------------------------
class Index : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

  // protected:
 public:
  // always:
  std::string type;        // fixed, floating, cms, fwd-cms, sticky, memory, formula, ...
  std::string ccy;
  std::string dayCount;
  Schedule    resetSched;  // start of first to end of last period, basis for interest calculation

  // if fixed:
  double      rate;        // TODO: rate schedule

  // if floating or cms or formula:
  std::string name;        // EURIBOR, LIBOR, ... neccessary ?
  std::string term;        // generic term, e.g. 6M or 10Y
  double      spread;      // TODO: spread schedule
  std::string fixTiming;   // fixing in arrears / advance, front / back
  int         fixGap;      // fixing gap days with respect to reset period start/end (for arr/adv)
  Schedule    fixSched;    // fixing schedule, optional
  std::string isCapped;
  double      cap;         // TODO: cap schedule
  std::string isFloored;
  double      floor;       // TODO: floor schedule

  // if formula (OVERRIDES cap and floor!):
  // The formula module can take an arbitrary number of variables.
  // However, currently only one default variable ("index") referring
  // to the underlying reference index is processed.
  // This allows already for relevant payoff modifications like
  // caps/floors using max()/min(), binary payoffs using step(), etc.

  XML::list<Formula> formulas;

  // if memory, sticky:
  // TODO

 public:
  Index() { init(); }
  Index(const Index&);
  //Index& operator = (const Index&);
  Index(XML::ptr<Index>);

  // generic constructor for all types
  Index( std::string type_, std::string ccy_, std::string dayCount, Schedule resetSched_,
	 double rate,
	 std::string name_, std::string term_, double spread_, std::string timing_, int gap_,
	 Schedule fixSched_,
	 std::string isCapped_, double cap_, std::string isFloored_, double floor_,
	 XML::list<Formula> forms_ );

  // fixed index
  Index( std::string type_, std::string ccy_, std::string dayCount, Schedule resetSched_,
	 double rate );

  // plain vanilla float index
  Index( std::string type_, std::string ccy_, std::string dayCount, Schedule resetSched_,
	 std::string name_, std::string term_, double spread_, std::string timing_, int gap_,
	 Schedule fixSched_ );

  // capped/floored floating index
  Index( std::string type_, std::string ccy_, std::string dayCount, Schedule resetSched_,
	 std::string name_, std::string term_, double spread_, std::string timing_, int gap_,
	 Schedule fixSched_,
	 std::string isCapped_, double cap_, std::string isFloored_, double floor_);

  // formula index
  Index( std::string type_, std::string ccy_, std::string dayCount, Schedule resetSched_,
	 std::string name_, std::string term_, double spread_, std::string timing_, int gap_,
	 Schedule fixSched_,
	 XML::list<Formula> forms_);


  // XML ---
  void init();
  int  isValid();

  XML_TAG t_type;
  XML_TAG t_ccy;
  XML_TAG t_dayCount;
  XML_TAG t_resetSched;

  XML_TAG t_rate;

  XML_TAG t_name;
  XML_TAG t_term;
  XML_TAG t_spread;
  XML_TAG t_fixTiming;
  XML_TAG t_fixGap;
  XML_TAG t_fixSched;
  XML_TAG t_isCapped;
  XML_TAG t_cap;
  XML_TAG t_isFloored;
  XML_TAG t_floor;

  XML_TAG t_formulas;

  XML_ATTR a_resetSched;
  XML_ATTR a_fixSched;

  XML_IO;
};

//------------------------------------------------------------------------------------------------
class Stats : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

 public:
  std::string date;
  std::string ccy;
  double      npv;
  double      gpv;
  double      acc;
  double      pv01;
  double      bpv;

  Stats () { init(); }
  Stats (const Stats&);
  Stats& operator = (const Stats&);

  // XML extension

  void init() {
    date = "";
    ccy  = "";
    npv  = 0;
    gpv  = 0;
    acc  = 0;
    pv01 = 0;
    bpv  = 0;
  }

  int isValid() {
    if (date != "")
      return 0;
    else
      return -1;
  }

  XML_TAG t_date;
  XML_TAG t_ccy;
  XML_TAG t_npv;
  XML_TAG t_gpv;
  XML_TAG t_acc;
  XML_TAG t_pv01;
  XML_TAG t_bpv;

  XML_IO
};

//------------------------------------------------------------------------------------------------
class Flow : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

 public:
  std::string startDate;
  std::string endDate;
  std::string irFixingDate;
  std::string fxFixingDate;
  double      period;
  double      rate;
  double      spread;
  double      notional;
  std::string paymentDate;
  double      amount;
  std::string ccy;
  std::string type;
  int         days;
  double      zero;
  double      discount;

  Flow () {
    init();
  }

  Flow (std::string startDate_,
	std::string endDate_,
	double      period_,
	double      rate_,
	double      spread_,
	double      notional_,
	std::string paymentDate_,
	double      amount_,
	std::string ccy_,
	double      zero_,
	double      discount_) {
    init();
    startDate   = startDate_;
    endDate     = endDate_;
    period      = period_;
    rate        = rate_;
    spread      = spread_;
    notional    = notional_;
    paymentDate = paymentDate_;
    amount      = amount_;
    ccy         = ccy_;
    zero        = zero_;
    discount    = discount_;
  }

  // XML extension

  void init() {
    startDate    = "";
    endDate      = "";
    irFixingDate = "";
    fxFixingDate = "";
    paymentDate  = "";
    period       = 0;
    rate         = 0;
    spread       = 0;
    notional     = 0;
    amount       = 0;
    ccy          = "";
    type         = "";
    days         = 0;
    zero         = 0;
    discount     = 0;
  }

  int isValid() {
    if (ccy != "")
      return 0;
    else
      return -1;
  }

  XML_TAG t_startDate;
  XML_TAG t_endDate;
  //XML_TAG t_irFixingDate;
  //XML_TAG t_fxFixingDate;
  XML_TAG t_period;
  XML_TAG t_rate;
  XML_TAG t_spread;
  XML_TAG t_notional;
  XML_TAG t_paymentDate;
  XML_TAG t_amount;
  XML_TAG t_ccy;
  //XML_TAG t_type;
  //XML_TAG t_days;
  XML_TAG t_zero;
  XML_TAG t_discount;

  XML_IO
};

//------------------------------------------------------------------------------------------------
class Leg : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access to all members, its fuction price() takes an Leg argument
  friend class Engine;

  // protected:
 public:
  std::string         id;
  std::string         type;       // index or payment
  std::string         ccy;        // neccessary ?
  XML::list<Notional> notional;
  XML::ptr<Index>     index;
  std::string         payTiming;  // ARR, ADV
  int                 payGap;     // gap days with respect to interest period start/end
  Schedule            paySched;   // payment schedule

  void set (const Leg& leg_);

 public:

  XML::list<Flow>     flows;
  Stats               stats;

  Leg () { init(); }
  Leg (const Leg& leg_);
  Leg (Leg * leg_);
  Leg (std::string id_, std::string type_, std::string ccy_, XML::list<Notional> notional_, XML::ptr<Index> index_, std::string payTiming_, int payGap_, Schedule& paySched_);

  Leg& operator = (const Leg& leg_);

  ~Leg() {}

  std::string         getId()        { return id; }
  std::string         getType()      { return type; }
  std::string         getCcy()       { return ccy; }
  XML::list<Notional> getNotional()  { return notional; }
  XML::ptr<Index>     getIndex()     { return index; }
  int                 getPayGap()    { return payGap; }
  std::string         getPayTiming() { return payTiming; }
  Schedule            getPaySched()  { return paySched; }
  XML::list<Flow>     getFlows()     { return flows; }

  // XML extension

  void init();
  int  isValid();

  XML_TAG t_id;
  XML_TAG t_type;
  XML_TAG t_ccy;
  XML_TAG t_notional;
  XML_TAG t_index;
  XML_TAG t_payTiming;
  XML_TAG t_payGap;
  XML_TAG t_paySched;
  XML_TAG t_flows;
  XML_TAG t_stats;

  //  XML_ATTR a_notional;

  XML_IO;
};

//------------------------------------------------------------------------------------------------
class Instrument;
//------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------
class Structure : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access to all members for pricing
  friend class Engine;

 public:
  std::string          type;
  std::string          rightTo;       // enter, cancel
  std::string          barrierPolicy; // UI, UO, DI, DO
  Schedule             exerciseSched;
  Schedule             noticeSched;
  XML::ptr<Instrument> underlying;

  Structure ();
  Structure (const Structure& s);
  Structure& operator = (const Structure& s);

  Structure (std::string type_,
	     std::string rightTo_,
	     std::string barrierPolicy_,
	     Schedule exerciseSched_,
	     Schedule noticeSched_,
	     XML::ptr<Instrument> underlying_ );

  Structure (std::string type_,
	     std::string rightTo_,
	     std::string barrierPolicy_,
	     Schedule exerciseSched_,
	     Schedule noticeSched_);

  Structure (std::string type_);

  void set (const RL::Structure&);

  ~Structure() {
    underlying.reset();
    if (_TRACE_) printf( "DELETE Structure %p\n", this );
  }

  std::string          getType ()       { return type; }
  XML::ptr<Instrument> getUnderlying () { return underlying; }

  // XML

  void init ();
  int  isValid ();

  XML_TAG t_type;
  XML_TAG t_rightTo;
  XML_TAG t_barrierPolicy;
  XML_TAG t_exerciseSched;
  XML_TAG t_noticeSched;
  XML_TAG t_underlying;

  XML_ATTR a_exerciseSched;
  XML_ATTR a_noticeSched;

  XML_IO;
};

//------------------------------------------------------------------------------------------------
class Instrument : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access, its price() function takes an instrument argument
  friend class Engine;

  // protected:
 public:
  std::string          id;
  int                  level;
  std::string          ccy;
  XML::list<Leg>       legs;
  Structure            structure;

  void trace(const char *s = "");
  void set  (const Instrument& inst_);

 public:

  // return values
  Stats stats;

  // constructors --------------------------------------------------------------------------------

  Instrument() {
    init();
    trace("");
  }

  Instrument(const Instrument&);
  Instrument& operator=(const Instrument&);

  Instrument(XML::ptr<Instrument>);

  Instrument( std::string id_,
	      int level_,
	      std::string ccy_,
	      XML::list<Leg> legs_,
	      Structure structure_ );

  ~Instrument() {
    //underlying.reset();
    if (_TRACE_) printf( "DELETE Instrument %p\n", this );
  }

  // public inspectors

  void                 print();
  int                  getLevel()      { return level; }
  std::string          getId()         { return id; }
  std::string          getCcy()        { return ccy; }
  XML::list<Leg>       getLegs()       { return legs; }

  std::string          getType()       { return structure.getType(); }
  Structure            getStructure()  { return structure; }
  XML::ptr<Instrument> getUnderlying() { return structure.getUnderlying(); }

  void logStats() {
    Debug log ("logStats");
    log.msg (NOTICE, "instrument (%s) ccy %s npv %.2f",
	     id.c_str(), ccy.c_str(), stats.npv );
  }

  // XML extension

  void init();
  int  isValid();

  XML_TAG t_id;
  XML_TAG t_level;
  XML_TAG t_ccy;
  XML_TAG t_legs;
  XML_TAG t_structure;
  XML_TAG t_underlying;
  XML_TAG t_stats;

  XML_IO;
};

//------------------------------------------------------------------------------------------------
class LmmVolSurface : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access, its price() function takes an instrument argument
  friend class Engine;

 private:
  std::string type;
  double a;
  double b;
  double c;
  double d;

 public:
  LmmVolSurface () { init(); }
  // the follwoing constructor is not required by the SXP mechanism
  LmmVolSurface (std::string type_,
		 double a_,
		 double b_,
		 double c_,
		 double d_)
    : type(type_), a(a_), b(b_), c(c_), d(d_) {}
  LmmVolSurface (const LmmVolSurface& n_) {
    type = n_.type;
    a    = n_.a;
    b    = n_.b;
    c    = n_.c;
    d    = n_.d;
   }
  LmmVolSurface& operator = (const LmmVolSurface& n_) {
    type = n_.type;
    a    = n_.a;
    b    = n_.b;
    c    = n_.c;
    d    = n_.d;
    return *this;
  }

  void init() {
    type = "";
    a = b = c = d = 0;
  }

  int isValid() {
    if (type != "")
      return 0;
    else
      return -1;
  }

  XML_TAG t_type;
  XML_TAG t_a;
  XML_TAG t_b;
  XML_TAG t_c;
  XML_TAG t_d;

  XML_IO;
};

//------------------------------------------------------------------------------------------------
class LmmCorrelations : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access, its price() function takes an instrument argument
  friend class Engine;

 public:
  std::string type;
  double alpha;
  double beta;
  double rho;

 public:
   LmmCorrelations () { init(); }
   // the follwoing constructor is not required by the SXP mechanism
   LmmCorrelations (std::string type_,
		    double alpha_,
		    double beta_,
		    double rho_)
     : type(type_), alpha(alpha_), beta(beta_), rho(rho_) {}
   LmmCorrelations (const LmmCorrelations& n_) {
     type  = n_.type;
     alpha = n_.alpha;
     beta  = n_.beta;
     rho   = n_.rho;
   }
   LmmCorrelations& operator = (const LmmCorrelations& n_) {
     type  = n_.type;
     alpha = n_.alpha;
     beta  = n_.beta;
     rho   = n_.rho;
     return *this;
   }

  void init() {
    type = "";
    alpha = beta = rho = 0;
  }

  int isValid() {
    if (type != "")
      return 0;
    else
      return -1;
  }

  XML_TAG t_type;
  XML_TAG t_alpha;
  XML_TAG t_beta;
  XML_TAG t_rho;

  XML_IO;
};

//------------------------------------------------------------------------------------------------
class LmmParameters : public XmlObject {
//------------------------------------------------------------------------------------------------
  // class Engine needs full access, its price() function takes an instrument argument
  friend class Engine;

 public:
  std::string     lmmType;
  int             lowFactor;
  std::string     volSurfaceType;
  std::string     correlationType;
  int             nVals;
  int             nPaths;
  XML::list<LmmVolSurface>   volSurfaces;
  XML::list<LmmCorrelations> correlations;

 public:
   LmmParameters () { init(); }
   // the follwoing constructor is not required by the SXP mechanism
   LmmParameters (std::string lmmType_,
		  std::string volSurfaceType_,
		  std::string correlationType_,
		  int lowFactor_,
		  int nVals_,
		  int nPaths_,
		  XML::list<LmmVolSurface> volSurfaces_,
		  XML::list<LmmCorrelations> correlations_) {
     lmmType = lmmType_;
     volSurfaceType = volSurfaceType_;
     correlationType = correlationType_;
     lowFactor = lowFactor_;
     nVals = nVals_;
     nPaths = nPaths_;
     volSurfaces = volSurfaces_;
     correlations = correlations_;
   }

   LmmParameters (const LmmParameters& n_) {
     lmmType         = n_.lmmType;
     volSurfaceType  = n_.volSurfaceType;
     correlationType = n_.correlationType;
     lowFactor       = n_.lowFactor;
     nVals           = n_.nVals;
     nPaths          = n_.nPaths;
     volSurfaces     = n_.volSurfaces;
     correlations    = n_.correlations;
   }
   LmmParameters& operator = (const LmmParameters& n_) {
     lmmType         = n_.lmmType;
     volSurfaceType  = n_.volSurfaceType;
     correlationType = n_.correlationType;
     lowFactor       = n_.lowFactor;
     nVals           = n_.nVals;
     nPaths          = n_.nPaths;
     volSurfaces     = n_.volSurfaces;
     correlations    = n_.correlations;
     return *this;
   }

  void init() {
    lmmType         = "";
    volSurfaceType  = "";
    correlationType = "";
    lowFactor       = 0;
    nVals           = 0;
    nPaths          = 0;
    volSurfaces.init();
    correlations.init();
  }

  int isValid() {
    if (lmmType != "" &&
	volSurfaceType != "" &&
	correlationType != "" &&
	volSurfaces.isValid() == 0 &&
	correlations.isValid() == 0 &&
	lowFactor > 0 &&
	nVals >= 0 &&
	nPaths > 0)
      return 0;
    else
      return -1;
  }

  XML_TAG t_lmmType;
  XML_TAG t_lowFactor;
  XML_TAG t_volSurfaceType;
  XML_TAG t_correlationType;
  XML_TAG t_nVals;
  XML_TAG t_nPaths;
  XML_TAG t_volSurfaces;
  XML_TAG t_correlations;

  XML_IO;
};

} // end of namespace RL {

#endif

