/*
   Copyright (C) 2005 Roland Lichters

   This file is part of MTM, an experimental program for tutorials in
   financial modelling - mtm@lichters.net

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "util.h"

// ----------------------------------------------------------------------------
int getarg( int argc, char **argv, const char *flag, char *argument ) {
// ----------------------------------------------------------------------------
  int i;
  unsigned int j;
  char a[1000]; // arguments converted to upper case
  char f[1000]; // flag converted to upper case

  if ( strlen(flag) > 1000 ) {
    printf( "string length of flag is too long (%ld)\n", strlen(flag) );
    exit(1);
  }

  // convert flag to upper case
  for( j=0; j<strlen(flag); j++ ) f[j] = toupper(flag[j]);
  f[j] = '\0';

  for( i=1; i<argc; i++ ) {

    if ( strlen(argv[i]) > 1000 ) {
      printf( "string length of argument %s is too long (%ld)\n", argv[i], strlen(flag) );
      exit(1);
    }

    // convert command line argument to uppercase
    for( j=0; j<strlen(argv[i]); j++ ) a[j] = toupper(argv[i][j]);
    a[j] = '\0';

    if ( strcmp(a,f)==0 ) {
      //      printf( "flag: %s -> %s\n", argv[i], a );
      if (i==argc-1 || strncmp(argv[i+1],"-",1)==0 ) return 0;
      else {
	strcpy( argument, argv[i+1] );
	return 1;
      }
    }
  }

  return -1;
}

// Basic data formatters from QuantLib releases < 0.3.10

std::string DateFormatter::toString(const Date& d,
                                    DateFormatter::Format f) {
  std::ostringstream out;
  switch (f) {
  case Long:
    out << io::long_date(d);
    break;
  case Short:
    out << io::short_date(d);
    break;
  case ISO:
    out << io::iso_date(d);
    break;
  default:
    QL_FAIL("unknown date format");
  }
  return out.str();
}

std::string WeekdayFormatter::toString(Weekday wd,
                                       WeekdayFormatter::Format f) {
  std::ostringstream out;
  switch (f) {
  case Long:
    out << io::long_weekday(wd);
    break;
  case Short:
    out << io::short_weekday(wd);
    break;
  case Shortest:
    out << io::shortest_weekday(wd);
    break;
  default:
    QL_FAIL("unknown weekday format");
  }
  return out.str();
}

std::string FrequencyFormatter::toString(Frequency freq) {
  std::ostringstream out;
  out << freq;
  return out.str();
}

std::string IntegerFormatter::toString(BigInteger l, Integer digits) {
  std::ostringstream out;
  out << std::setw(digits) << io::checknull(l);
  return out.str();
}

std::string IntegerFormatter::toPowerOfTwo(BigInteger l, Integer digits) {
  std::ostringstream out;
  out << std::setw(digits) << io::power_of_two(l);
  return out.str();
}

std::string SizeFormatter::toString(Size l, Integer digits) {
  std::ostringstream out;
  out << std::setw(digits) << io::checknull(l);
  return out.str();
}

std::string SizeFormatter::toOrdinal(Size l) {
  std::ostringstream out;
  out << io::ordinal(l);
  return out.str();
}

std::string SizeFormatter::toPowerOfTwo(Size l, Integer digits) {
  std::ostringstream out;
  out << std::setw(digits) << io::power_of_two(l);
  return out.str();
}

std::string DecimalFormatter::toString(Decimal x, Integer precision,
                                       Integer digits) {
  std::ostringstream out;
  out << std::fixed << std::setprecision(precision);
  out << std::setw(digits) << io::checknull(x);
  return out.str();
}

std::string DecimalFormatter::toExponential(Decimal x, Integer precision,
                                            Integer digits) {
  std::ostringstream out;
  out << std::scientific << std::setprecision(precision);
  out << std::setw(digits) << io::checknull(x);
  return out.str();
}

std::string DecimalFormatter::toPercentage(Decimal x, Integer precision,
                                           Integer digits) {
  std::ostringstream out;
  out << std::setw(digits) << std::setprecision(precision);
  out << io::checknull(x);
  return out.str();
}

std::string StringFormatter::toLowercase(const std::string& s) {
  return lowercase(s);
}

std::string StringFormatter::toUppercase(const std::string& s) {
  return uppercase(s);
}

std::string RateFormatter::toString(Rate rate, Integer precision) {
  return DecimalFormatter::toPercentage(rate,precision);
}

std::string VolatilityFormatter::toString(Volatility vol,
                                          Integer precision) {
  return DecimalFormatter::toPercentage(vol,precision);
}

// Math functions from QuantLib releases < 0.3.6

double interpolate1D (const std::vector<double>& x_values,
		      const std::vector<double>& y_values,
		      double x,
		      int interpolationType,
		      bool allowExtrapolation) {
  double result = 0.0;

  switch (interpolationType) {
  case 1:
    result = LinearInterpolation (x_values.begin(), x_values.end(),
				  y_values.begin())(x, allowExtrapolation);
    break;
  case 2:
    result = NaturalCubicSpline (x_values.begin(), x_values.end(),
				 y_values.begin())(x, allowExtrapolation);
    break;
  case 3:
    result = LogLinearInterpolation (x_values.begin(), x_values.end(),
				     y_values.begin())(x, allowExtrapolation);
    break;
  default:
    QL_FAIL("interpolate: invalid interpolation type");
  }

  return result;
}

double interpolate2D(const std::vector<double>& x_values,
		     const std::vector<double>& y_values,
		     const Matrix& dataMatrix,
		     double x, double y, int interpolation2DType,
		     bool allowExtrapolation) {
  double result = 0.0;

  switch (interpolation2DType) {
  case 1:
    result = BilinearInterpolation(x_values.begin(), x_values.end(),
				   y_values.begin(), y_values.end(), dataMatrix)(x,y,
								     allowExtrapolation);
            break;
  case 2:
    result = BicubicSpline(x_values.begin(), x_values.end(),
			   y_values.begin(), y_values.end(), dataMatrix)(x,y,
									 allowExtrapolation);
    break;
  default:
    QL_FAIL("interpolate2D: invalid interpolation type");
  }

  return result;
}

//eof
